import logging

from docpipe.exc import ConfigException
from docpipe.constants import EXTRACT_PHASE, TRANSFORM_PHASE
from docpipe.tasks import load

log = logging.getLogger(__name__)


def resolve_dependencies(nodes_):
    """ Figure out which order the nodes in the graph can be executed
    in to satisfy all requirements. """
    nodes = []
    for node in nodes_:
        if node.phase is TRANSFORM_PHASE:
            nodes.append(node)
    done, last = set(), 0
    while True:
        if len(done) == len(nodes):
            break
        for node in nodes:
            if node.name not in done:
                match = done.intersection(node.requires)
                if len(match) == len(node.requires):
                    done.add(node.name)
                    yield node
                    break
        else:
            raise ConfigException('Invalid requirements in pipeline!')


class Node(object):

    _tasks = load()

    def __init__(self, pipeline, phase, name, config):
        self.name = name
        self.phase = phase
        self.config = config
        self.pipeline = pipeline
        self._task = None
        self._requires = None

    @property
    def task(self):
        if self._task is None:
            task = self.config.get('task')
            if task not in self._tasks:
                msg = '%s: invalid task %s' % (self.name, task)
                raise ConfigException(msg)
            ep = self._tasks.get(task)
            self._task = ep.plugin(self.pipeline, self.name, self.config)
        return self._task

    def extract(self):
        log.debug("Running extract: %s (%s)" % (self.name, self.task.type))
        if self.phase is EXTRACT_PHASE:
            self.task.extract()

    def transform(self, document):
        log.debug("Running transform: %s (%s) on %s" %
                  (self.name, self.task.type, document))
        if self.phase is TRANSFORM_PHASE:
            self.task.transform(document)

    @property
    def requires(self):
        if self._requires is None:
            reqs = self.config.get('requires', [])
            if reqs is None:
                reqs = []
            if not isinstance(reqs, (list, tuple, set)):
                reqs = [unicode(reqs)]
            self._requires = set(reqs)
        return self._requires

    def __repr__(self):
        return '<Node("%s", %s)>' % (self.name, self.phase)
