import os
import sys
import logging
import time
import signal
import logging
from optparse import OptionParser

import tornado.ioloop
import tornado.web
import tornado.options
import tornado.httpserver
import tornado.autoreload

from dorothy.settings import settings


def watch_dir(dirname, include_subdirs=True):
    """Add a dir and all of its files to the watch list.
    Any file or subdirectory that begins with a period is ignored.
    """
    for _root, _dirs, _files in os.walk(dirname):
        for _file in _files:
            if not _file.startswith('.'):
                tornado.autoreload.watch('/'.join([_root, _file]))
        if include_subdirs:
            for _dir in _dirs:
                if not _dir.startswith('.'):
                    watch_dir('/'.join([_root, _dir]))


class CLO(object):
    def __init__(self):
        self.parser = OptionParser()
        self.parser.add_option("-p", "--port", dest="port", default=9000, type="int",
                          help="port to run the server on")
        self.parser.add_option("-d", "--debug", action="store_true", dest="debug",
                          help="run the server in debug mode")
        self.parser.add_option('-t', '--time-out', dest="timeout", default=5, type="int",
                            help="Time to wait when stopping the server")

    def add_option(self, *args, **kawrgs):
        self.parser.add_option(*args, **kwargs)

    def parse(self):
        (self.options, self.args) = self.parser.parse_args()

class Dorothy(object):
    def __init__(self, routes=None, prelaunch=None, shutdown=None, **kwargs):
        self._routes = routes
        self._prelaunch = prelaunch
        self._shutdown_cb = shutdown

        self.clo = CLO()

        for k, v in kwargs.iteritems():
            if k != 'dorothy':
                setattr(settings, k, v)

        logging.root.setLevel(settings.log_level)
        logger = logging.getLogger('dorothy')
        sh = logging.StreamHandler(stream=sys.stderr)
        sh.setLevel(settings.log_level)
        formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
        sh.setFormatter(formatter)
        logger.addHandler(sh)
        settings.log = logger

    @property
    def routes(self):
        return self._routes

    @routes.setter
    def routes(self, val):
        self._routes = val

    @property
    def prelaunch(self):
        return self._prelaunch

    @prelaunch.setter
    def prelaunch(self, val):
        self._prelaunch = val

    @property
    def shutdown(self):
        return self._shutdown_cb

    @shutdown.setter
    def shutdown(self, val):
        self._shutdown_cb = val

    def _shutdown(self, sig, frame):
        settings.log.info('Stopping the server')

        if not self.clo.options.debug:
            # give current connection 2 seconds to finish up but no in debug
            time.sleep(self.clo.options.timeout)

        if self.shutdown:
            settings.io_loop.add_callback_from_signal(self.shutdown)
            time.sleep(self.clo.options.timeout)

        self.server.stop()
        settings.io_loop.stop()

    def run(self, **kwargs):
        self.clo.parse()
        self.clo.options.debug = self.clo.options.debug if self.clo.options.debug else False
        settings.debug = self.clo.options.debug
        settings.port = self.clo.options.port

        application = tornado.web.Application(self.routes, port=self.clo.options.port, debug=self.clo.options.debug, **kwargs)
        self.server = tornado.httpserver.HTTPServer(application, xheaders=True)
        self.server.listen(self.clo.options.port)

        signal.signal(signal.SIGINT, self._shutdown)
        signal.signal(signal.SIGTERM, self._shutdown)
        signal.signal(signal.SIGQUIT, self._shutdown)

        settings.io_loop = tornado.ioloop.IOLoop.instance()

        if self.prelaunch:
            self.prelaunch()

        settings.log.info('Starting server on port {0}'.format(self.clo.options.port))
        settings.io_loop.start()
