import os
from copy import deepcopy
import logging

import yaml

from dorothy.lib.util import parse_config
from dorothy.lib.exceptions import SettingsError


class AttrDict(dict):
    def __init__(self, init={}):
        dict.__init__(self, init)

    def __getstate__(self):
        return self.__dict__.items()

    def __setstate__(self, items):
        for key, val in items:
            self.__dict__[key] = val

    def __setitem__(self, key, value):
        return super(AttrDict, self).__setitem__(key, value)

    def __getitem__(self, name):
        return super(AttrDict, self).__getitem__(name)

    __getattr__ = __getitem__
    __setattr__ = __setitem__

    def copy(self):
        ch = AttrDict(self)
        return ch


def to_attr_dict(d):
    result = AttrDict()
    for k, v in d.iteritems():
        if isinstance(v, dict):
            v = to_attr_dict(v)
        result[k] = v
    return result


def dict_merge(a, b):
    if not isinstance(b, dict):
        return b
    result = deepcopy(a) or dict()
    for k, v in b.iteritems():
        if k in result and isinstance(result[k], dict):
                result[k] = dict_merge(result[k], v)
        else:
            result[k] = deepcopy(v)
    return to_attr_dict(result)


class _Settings(object):
    DATA = 1
    CONFIG = 2

    def __init__(self, data=None, is_config=False):
        self.__dict__['__data'] = data or AttrDict()
        self.__dict__['__config'] = None

        if is_config:
            self.__dict__['__is'] = self.CONFIG
        else:
            self.__dict__['__is'] = self.DATA

    def get(self, attr, default=None):
        try:
            return getattr(self, attr)
        except:
            return default

    def __getattr__(self, name):
        if name == 'config' and self.__dict__['__is'] == self.DATA:
            if self.__dict__['__config'] is None:
                self.__dict__['__config'] = self.__class__(parse_config(self.environment), is_config=True)
            return self.__dict__['__config']
        return self.__dict__['__data'][name]

    def __setattr__(self, name, val):
        self.__dict__['__data'][name] = val

    @property
    def environment(self):
        return os.environ.get(self.get('env', 'DOROTHY_ENVIRONMENT'), 'development')

    @property
    def log_level(self):
        logging_level = self.config.get('logging_level', 'debug')
        if logging_level == 'debug':
            return logging.DEBUG
        elif logging_level == 'warning':
            return logging.WARNING,
        elif logging_level == 'info':
            return logging.INFO,
        elif logging_level == 'critical':
            return logging.CRITICAL,
        elif logging_level == 'error':
            return logging.ERROR


settings = _Settings()
