import time
from datetime import datetime
import os

import tornado.template
import tornado.web

from dorothy.settings import settings
from dorothy.lib import xjson
from dorothy.lib.exceptions import ApiError, SettingsError


class MainBaseHandler(tornado.web.RequestHandler):
    def __init__(self, application, request, **kwargs):
        super(MainBaseHandler, self).__init__(application, request, **kwargs)

        self.query = {}
        self.body = {}
        self._page_data = dict()
        self._redirect = None

    ##################################################
    # The follow method are what should be implemented
    # individual handlers.
    def get_(self, *args, **kwargs):
        raise tornado.web.HTTPError(405)

    def post_(self, *args, **kwargs):
        raise tornado.web.HTTPError(405)

    def put_(self, *args, **kwargs):
        raise tornado.web.HTTPError(405)

    def delete_(self, *args, **kwargs):
        raise tornado.web.HTTPError(405)

    def on_finish_get(self):
        ''' Called after get_() '''
        pass

    def on_finish_post(self):
        ''' Called after post_() '''
        pass

    def on_finish_put(self):
        ''' Called after put_() '''
        pass

    def on_finish_delete(self):
        ''' Called after delete_() '''
        pass

    def on_finish_options(self):
        ''' Called after options_() '''
        pass

    def on_finish_all(self):
        ''' Called after get_(), post_(), put_(), delete_(), options_()
        
        NOTE: This is called AFTER the method specific finish
        '''
        pass

    # def _handle_request_exception(self, e):
    #     if isinstance(e, tornado.web.HTTPError):
    #         response = e.log_message
    #         code = e.status_code
    #     elif isinstance(e, ApiError):
    #         response = e.message
    #         code = e.code
    #     else:
    #         code = 500
    #         response = str(e)
    # 
    #     # TODO: add some logging here
    #     # TODO: if in debug mode, raise all exceptions
    # 
    #     return self._respond((code, response))


    ##################################################

    def redirect(self, url, permanent=False, status=None):
        self._redirect = (url, permanent, status)

    def get(self, *args, **kwargs):
        return self._process(*args, **kwargs)

    def post(self, *args, **kwargs):
        return self._process(*args, **kwargs)

    def put(self, *args, **kwargs):
        return self._process(*args, **kwargs)

    def delete(self, *args, **kwargs):
        return self._process(*args, **kwargs)

    def options(self, *args, **kwargs):
        self.finish()

    @property
    def method(self):
        return self.request.method.lower()

    def get_current_user(self):
        # implement whatever logic you want to
        # authenticate users here
        pass

    def http_error(self, code, message=None):
        raise tornado.web.HTTPError(code, message)

    def error(self, code, message=None):
        self.http_error(code, message)

    def on_finish(self):
        if self.method == 'get':
            self.on_finish_get()
        elif self.method == 'post':
            self.on_finish_post()
        elif self.method == 'put':
            self.on_finish_put()
        elif self.method == 'delete':
            self.on_finish_delete()
        elif self.method == 'options':
            self.on_finish_options()

        self.on_finish_all()

    def template(self, tmpl, data=None):
        data = data or {}
        pth = tmpl.strip('/')
        if len(pth.split('/')) == 1:
            pth = ''
        else:
            pth = '/'.join(pth.split('/')[:-1] + [''])
        tmpl = tmpl.replace('/', '.')

        if not settings.get('template_path'):
            raise SettingsError('Can not load template files. No template_path given.')

        self._page_data = data
        loader = tornado.template.Loader(settings.template_path)
        data.update(self.get_template_namespace())
        return loader.load('{0}{1}.html'.format(pth, tmpl)).generate(**data)

    def _process(self, *args, **kwargs):
        self._parse_query()
        self._parse_body()

        args = tuple([a for a in args if a])
        if self.method == 'get':
            response = self.get_(*args, **kwargs)
        elif self.method == 'post':
            response = self.post_(*args, **kwargs)
        elif self.method == 'put':
            response = self.put_(*args, **kwargs)
        elif self.method == 'delete':
            response = self.delete_(*args, **kwargs)
        elif self.method == 'options':
            response = self.options_(*args, **kwargs)

        self._respond(response)

    def _respond(self, response):
        pass

    def _parse_query(self):
        uri = self.request.uri
        params = uri.split('?', 2)
        if len(params) == 2:
            params = [p for p in params[1].split('&') if p]
            for p in params:
                name, val = p.split('=', 2)
                name = name.strip()
                val = val.strip() if isinstance(val, basestring) else val
                self.query[name] = val

    def _parse_body(self):
        if self.request.headers.get('Content-Type') == 'application/json':
            body = self.request.body or '{}'
            body = xjson.loads(body)
        else:
            body = {i2[0]: i2[1] for i2 in [i.split('=') for i in self.request.body.split('&') if i]}
        for name, val in body.iteritems():
            name = name.strip()
            val = val.strip() if isinstance(val, basestring) else val
            self.body[name] = val


class WebBaseHandler(MainBaseHandler):
    def __init__(self, application, request, **kwargs):
        super(WebBaseHandler, self).__init__(application, request, **kwargs)

        self._js = []
        self._css = []

    def js(self, *args):
        for js in args:
            self._js.append(js)

    def css(self, *args):
        for css in args:
            self._css.append(css)

    @property
    def js(self):
        return self._js

    @property
    def css(self):
        return self._css

    def _respond(self, response):
        if self._redirect:
            return tornado.web.RequestHandler.redirect(self, *self._redirect)

        if not isinstance(response, tuple):
            response = (200, response)
        status_code, content = response

        self.set_status(status_code)
        self.write(self.prepare_page(content, self._page_data))
        self.finish()

    def prepare_page(self, content, page_data):
        return content


class ApiBaseHandler(MainBaseHandler):
    def initialize(self):
        super(ApiBaseHandler, self).initialize()

        # All API request should return JSON
        self.set_header('Content-Type', 'application/json')

        accept = {
            'get_': 'GET',
            'post_': 'POST',
            'put_': 'PUT',
            'patch_': 'PATCH',
            'delete_': 'DELETE'}

        origin = '://'.join([self.request.protocol, self.request.host])
        meths = set(self.__class__.__dict__.keys())
        inter = set.intersection(meths, set(accept.keys()))
        poss = ['OPTIONS']
        for m in inter:
            poss.append(accept[m])

        self.set_header('Access-Control-Allow-Origin', self.request.headers.get('Origin', origin))
        self.set_header('Access-Control-Allow-Methods', ', '.join(poss))
        self.set_header('Access-Control-Max-Age', 86400)

        self.set_header('Access-Control-Allow-Credentials', 'true')

        a_head = self.request.headers.keys()
        a_head = [h.strip() for h in self.request.headers.get('Access-Control-Request-Headers', '').split(',')]

        self.set_header('Access-Control-Allow-Headers', ','.join(a_head))
        self.set_header('Content-Type', 'application/json; charset=UTF-8')

        if self.method == 'options':
            self.set_header('Content-Type', 'text/plain')

    def error(self, code, message):
        raise ApiError(code, message)

    def _respond(self, response):
        if self._redirect:
            response = dict(redirect=self._redirect[0])

        if not isinstance(response, tuple):
            response = (200, response)
        status_code, content = response

        response = dict(
            code=status_code,
            time=int(time.time()),
            response=content)

        self.set_status(response['code'])
        self.write(xjson.dumps(response))
        self.finish()