##
# Copyright 2009-2013 Ghent University
#
# This file is part of EasyBuild,
# originally created by the HPC team of Ghent University (http://ugent.be/hpc/en),
# with support of Ghent University (http://ugent.be/hpc),
# the Flemish Supercomputer Centre (VSC) (https://vscentrum.be/nl/en),
# the Hercules foundation (http://www.herculesstichting.be/in_English)
# and the Department of Economy, Science and Innovation (EWI) (http://www.ewi-vlaanderen.be/en).
#
# http://github.com/hpcugent/easybuild
#
# EasyBuild is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation v2.
#
# EasyBuild is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with EasyBuild.  If not, see <http://www.gnu.org/licenses/>.
##
"""
EasyBuild support for building and installing HDF5, implemented as an easyblock

@author: Stijn De Weirdt (Ghent University)
@author: Dries Verdegem (Ghent University)
@author: Kenneth Hoste (Ghent University)
@author: Pieter De Baets (Ghent University)
@author: Jens Timmerman (Ghent University)
"""

import os

import easybuild.tools.environment as env
from easybuild.easyblocks.generic.configuremake import ConfigureMake
from easybuild.tools.modules import get_software_root


class EB_HDF5(ConfigureMake):
    """Support for building/installing HDF5"""

    def configure_step(self):
        """Configure build: set require config and make options, and run configure script."""

        # configure options
        deps = ["Szip", "zlib"]
        for dep in deps:
            root = get_software_root(dep)
            if root:
                self.cfg.update('configopts', '--with-%s=%s' % (dep.lower(), root))
            else:
                self.log.error("Dependency module %s not loaded." % dep)

        fcomp = 'FC="%s"' % os.getenv('F90')

        self.cfg.update('configopts', "--with-pic --with-pthread --enable-shared")
        self.cfg.update('configopts', "--enable-cxx --enable-fortran %s" % fcomp)

        # MPI and C++ support enabled requires --enable-unsupported, because this is untested by HDF5
        if self.toolchain.options['usempi']:
            self.cfg.update('configopts', "--enable-unsupported")

        # make options
        self.cfg.update('makeopts', fcomp)

        # set RUNPARALLEL
        env.setvar('RUNPARALLEL', 'mpirun -np \$\${NPROCS:=2}')

        super(EB_HDF5, self).configure_step()

    # default make and make install are ok

    def sanity_check_step(self):
        """
        Custom sanity check for HDF5
        """

        if self.toolchain.options['usempi']:
            extra_binaries = ["bin/%s" % x for x in ["h5perf", "h5pcc", "h5pfc", "ph5diff"]]
        else:
            extra_binaries = ["bin/%s" % x for x in ["h5cc", "h5fc"]]

        custom_paths = {
                        'files': ["bin/h5%s" % x for x in ["2gif", "c++", "copy", "debug", "diff",
                                                           "dump", "import", "jam","ls", "mkgrp",
                                                           "perf_serial", "redeploy", "repack",
                                                           "repart", "stat", "unjam"]] +
                                 ["bin/gif2h5"] + extra_binaries +
                                 ["lib/libhdf5%s.so" % x for x in ["_cpp", "_fortran", "_hl_cpp",
                                                                   "_hl", "hl_fortran", ""]],
                        'dirs': ['include']
                       }

        super(EB_HDF5, self).sanity_check_step(custom_paths=custom_paths)
