# #
# Copyright 2013-2014 Ghent University
#
# This file is part of EasyBuild,
# originally created by the HPC team of Ghent University (http://ugent.be/hpc/en),
# with support of Ghent University (http://ugent.be/hpc),
# the Flemish Supercomputer Centre (VSC) (https://vscentrum.be/nl/en),
# the Hercules foundation (http://www.herculesstichting.be/in_English)
# and the Department of Economy, Science and Innovation (EWI) (http://www.ewi-vlaanderen.be/en).
#
# http://github.com/hpcugent/easybuild
#
# EasyBuild is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation v2.
#
# EasyBuild is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with EasyBuild.  If not, see <http://www.gnu.org/licenses/>.
# #
"""
Unit tests for EasyBuild configuration.

@author: Kenneth Hoste (Ghent University)
@author: Stijn De Weirdt (Ghent University)
"""
import copy
import os
import shutil
import tempfile
from test.framework.utilities import EnhancedTestCase, init_config
from unittest import TestLoader
from unittest import main as unittestmain
from vsc.utils.fancylogger import setLogLevelDebug, logToScreen

import easybuild.tools.options as eboptions
from easybuild.tools.config import build_path, source_paths, install_path, get_repository, get_repositorypath
from easybuild.tools.config import log_file_format, set_tmpdir, BuildOptions, ConfigurationVariables
from easybuild.tools.config import get_build_log_path, DEFAULT_PATH_SUBDIRS, init_build_options, build_option
from easybuild.tools.environment import modify_env
from easybuild.tools.filetools import write_file
from easybuild.tools.repository import FileRepository, init_repository


class EasyBuildConfigTest(EnhancedTestCase):
    """Test cases for EasyBuild configuration."""

    tmpdir = None

    def cleanup(self):
        """Cleanup enviroment"""
        for envvar in os.environ.keys():
            if envvar.startswith('EASYBUILD'):
                del os.environ[envvar]

    def setUp(self):
        """Prepare for running a config test."""
        super(EasyBuildConfigTest, self).setUp()

        self.tmpdir = tempfile.mkdtemp()
        self.cleanup()
        # keep track of original environment to restore
        self.orig_environ = copy.deepcopy(os.environ)

    def purge_environment(self):
        """Remove any leftover easybuild variables"""
        for x in os.environ.keys():
            # oldstyle and newstyle
            if x.startswith('EASYBUILD'):
                del os.environ[x]

    def tearDown(self):
        """Clean up after a config test."""
        super(EasyBuildConfigTest, self).tearDown()

        self.cleanup()
        try:
            shutil.rmtree(self.tmpdir)
        except OSError:
            pass
        tempfile.tempdir = None

    def configure(self, args=None):
        """(re)Configure and return configfile"""
        options = init_config(args=args)
        return options.config

    def test_default_config(self):
        """Test default configuration."""
        eb_go = eboptions.parse_options(args=[])
        config_options = eb_go.get_options_by_section('config')

        # check default subdirs
        self.assertEqual(DEFAULT_PATH_SUBDIRS['buildpath'], 'build')
        self.assertEqual(DEFAULT_PATH_SUBDIRS['installpath'], '')
        self.assertEqual(DEFAULT_PATH_SUBDIRS['subdir_modules'], 'modules')
        self.assertEqual(DEFAULT_PATH_SUBDIRS['repositorypath'], 'ebfiles_repo')
        self.assertEqual(DEFAULT_PATH_SUBDIRS['sourcepath'], 'sources')
        self.assertEqual(DEFAULT_PATH_SUBDIRS['subdir_software'], 'software')

        # check whether defaults are honored, use hardcoded paths/subdirs
        eb_homedir = os.path.join(os.path.expanduser('~'), '.local', 'easybuild')
        self.assertEqual(config_options['buildpath'], os.path.join(eb_homedir, 'build'))
        self.assertEqual(config_options['sourcepath'], os.path.join(eb_homedir, 'sources'))
        self.assertEqual(config_options['installpath'], eb_homedir)
        self.assertEqual(config_options['subdir_software'], 'software')
        self.assertEqual(config_options['subdir_modules'], 'modules')
        self.assertEqual(config_options['repository'], 'FileRepository')
        self.assertEqual(config_options['repositorypath'], [os.path.join(eb_homedir, 'ebfiles_repo')])
        self.assertEqual(config_options['logfile_format'][0], 'easybuild')
        self.assertEqual(config_options['logfile_format'][1], "easybuild-%(name)s-%(version)s-%(date)s.%(time)s.log")
        self.assertEqual(config_options['tmp_logdir'], tempfile.gettempdir())

    def test_legacy_env_vars(self):
        """Test legacy environment variables."""
        self.purge_environment()

        # build path
        test_buildpath = os.path.join(self.tmpdir, 'build', 'path')
        os.environ['EASYBUILDBUILDPATH'] = test_buildpath
        self.configure(args=[])
        self.assertEqual(build_path(), test_buildpath)
        del os.environ['EASYBUILDBUILDPATH']

        # source path(s)
        test_sourcepaths = [
            os.path.join(self.tmpdir, 'source', 'path'),
            ':'.join([
                os.path.join(self.tmpdir, 'source', 'path1'),
                os.path.join(self.tmpdir, 'source', 'path2'),
            ]),
            ':'.join([
                os.path.join(self.tmpdir, 'source', 'path1'),
                os.path.join(self.tmpdir, 'source', 'path2'),
                os.path.join(self.tmpdir, 'source', 'path3'),
            ]),
        ]
        for test_sourcepath in test_sourcepaths:
            init_config()
            os.environ['EASYBUILDSOURCEPATH'] = test_sourcepath
            self.configure(args=[])
            self.assertEqual(build_path(), os.path.join(os.path.expanduser('~'), '.local', 'easybuild',
                                                        DEFAULT_PATH_SUBDIRS['buildpath']))
            self.assertEqual(source_paths(), test_sourcepath.split(':'))
            del os.environ['EASYBUILDSOURCEPATH']

        test_sourcepath = os.path.join(self.tmpdir, 'source', 'path')

        # install path
        init_config()
        test_installpath = os.path.join(self.tmpdir, 'install', 'path')
        os.environ['EASYBUILDINSTALLPATH'] = test_installpath
        self.configure(args=[])
        self.assertEqual(source_paths()[0], os.path.join(os.path.expanduser('~'), '.local', 'easybuild',
                                                          DEFAULT_PATH_SUBDIRS['sourcepath']))
        self.assertEqual(install_path(), os.path.join(test_installpath, DEFAULT_PATH_SUBDIRS['subdir_software']))
        self.assertEqual(install_path(typ='mod'), os.path.join(test_installpath,
                                                                 DEFAULT_PATH_SUBDIRS['subdir_modules']))
        del os.environ['EASYBUILDINSTALLPATH']

        # prefix: should change build/install/source/repo paths
        init_config()
        test_prefixpath = os.path.join(self.tmpdir, 'prefix', 'path')
        os.environ['EASYBUILDPREFIX'] = test_prefixpath
        self.configure(args=[])
        self.assertEqual(build_path(), os.path.join(test_prefixpath, DEFAULT_PATH_SUBDIRS['buildpath']))
        self.assertEqual(source_paths()[0], os.path.join(test_prefixpath, DEFAULT_PATH_SUBDIRS['sourcepath']))
        self.assertEqual(install_path(), os.path.join(test_prefixpath, DEFAULT_PATH_SUBDIRS['subdir_software']))
        self.assertEqual(install_path(typ='mod'), os.path.join(test_prefixpath,
                                                               DEFAULT_PATH_SUBDIRS['subdir_modules']))
        repo = init_repository(get_repository(), get_repositorypath())
        self.assertTrue(isinstance(repo, FileRepository))
        self.assertEqual(repo.repo, os.path.join(test_prefixpath, DEFAULT_PATH_SUBDIRS['repositorypath']))

        # build/source/install path overrides prefix
        init_config()
        os.environ['EASYBUILDBUILDPATH'] = test_buildpath
        self.configure(args=[])
        self.assertEqual(build_path(), test_buildpath)
        self.assertEqual(source_paths()[0], os.path.join(test_prefixpath, DEFAULT_PATH_SUBDIRS['sourcepath']))
        self.assertEqual(install_path(), os.path.join(test_prefixpath, DEFAULT_PATH_SUBDIRS['subdir_software']))
        self.assertEqual(install_path(typ='mod'), os.path.join(test_prefixpath,
                                                               DEFAULT_PATH_SUBDIRS['subdir_modules']))
        repo = init_repository(get_repository(), get_repositorypath())
        self.assertTrue(isinstance(repo, FileRepository))
        self.assertEqual(repo.repo, os.path.join(test_prefixpath, DEFAULT_PATH_SUBDIRS['repositorypath']))
        del os.environ['EASYBUILDBUILDPATH']

        init_config()
        os.environ['EASYBUILDSOURCEPATH'] = test_sourcepath
        self.configure(args=[])
        self.assertEqual(build_path(), os.path.join(test_prefixpath, DEFAULT_PATH_SUBDIRS['buildpath']))
        self.assertEqual(source_paths()[0], test_sourcepath)
        self.assertEqual(install_path(), os.path.join(test_prefixpath, DEFAULT_PATH_SUBDIRS['subdir_software']))
        self.assertEqual(install_path(typ='mod'), os.path.join(test_prefixpath,
                                                               DEFAULT_PATH_SUBDIRS['subdir_modules']))
        repo = init_repository(get_repository(), get_repositorypath())
        self.assertTrue(isinstance(repo, FileRepository))
        self.assertEqual(repo.repo, os.path.join(test_prefixpath, DEFAULT_PATH_SUBDIRS['repositorypath']))
        del os.environ['EASYBUILDSOURCEPATH']

        init_config()
        os.environ['EASYBUILDINSTALLPATH'] = test_installpath
        self.configure(args=[])
        self.assertEqual(build_path(), os.path.join(test_prefixpath, DEFAULT_PATH_SUBDIRS['buildpath']))
        self.assertEqual(source_paths()[0], os.path.join(test_prefixpath, DEFAULT_PATH_SUBDIRS['sourcepath']))
        self.assertEqual(install_path(), os.path.join(test_installpath, DEFAULT_PATH_SUBDIRS['subdir_software']))
        self.assertEqual(install_path(typ='mod'), os.path.join(test_installpath,
                                                               DEFAULT_PATH_SUBDIRS['subdir_modules']))
        repo = init_repository(get_repository(), get_repositorypath())
        self.assertTrue(isinstance(repo, FileRepository))
        self.assertEqual(repo.repo, os.path.join(test_prefixpath, DEFAULT_PATH_SUBDIRS['repositorypath']))

    def test_legacy_config_file(self):
        """Test finding/using legacy configuration files."""
        self.purge_environment()

        cfg_fn = self.configure(args=[])
        self.assertTrue(cfg_fn.endswith('easybuild/easybuild_config.py'))

        configtxt = """
build_path = '%(buildpath)s'
source_path = '%(sourcepath)s'
install_path = '%(installpath)s'
repository_path = '%(repopath)s'
repository = FileRepository(repository_path)
log_format = ('%(logdir)s', '%(logtmpl)s')
log_dir = '%(tmplogdir)s'
software_install_suffix = '%(softsuffix)s'
modules_install_suffix = '%(modsuffix)s'
"""

        buildpath = os.path.join(self.tmpdir, 'my', 'test', 'build', 'path')
        sourcepath = os.path.join(self.tmpdir, 'my', 'test', 'source', 'path')
        installpath = os.path.join(self.tmpdir, 'my', 'test', 'install', 'path')
        repopath = os.path.join(self.tmpdir, 'my', 'test', 'repo', 'path')
        logdir = 'somedir'
        logtmpl = 'test-eb-%(name)s%(version)s_date-%(date)s__time-%(time)s.log'
        tmplogdir = os.path.join(self.tmpdir, 'my', 'test', 'tmplogdir')
        softsuffix = 'myfavoritesoftware'
        modsuffix = 'modulesgohere'

        configdict = {
            'buildpath': buildpath,
            'sourcepath': sourcepath,
            'installpath': installpath,
            'repopath': repopath,
            'logdir': logdir,
            'logtmpl': logtmpl,
            'tmplogdir': tmplogdir,
            'softsuffix': softsuffix,
            'modsuffix': modsuffix
        }

        # create user config file on default location
        myconfigfile = os.path.join(self.tmpdir, '.easybuild', 'config.py')
        if not os.path.exists(os.path.dirname(myconfigfile)):
            os.makedirs(os.path.dirname(myconfigfile))
        write_file(myconfigfile, configtxt % configdict)

        # redefine home so we can test user config file on default location
        home = os.environ.get('HOME', None)
        os.environ['HOME'] = self.tmpdir
        init_config()
        cfg_fn = self.configure(args=[])
        if home is not None:
            os.environ['HOME'] = home

        # check finding and use of config file
        self.assertEqual(cfg_fn, myconfigfile)
        self.assertEqual(build_path(), buildpath)
        self.assertEqual(source_paths()[0], sourcepath)
        self.assertEqual(install_path(), os.path.join(installpath, softsuffix))
        self.assertEqual(install_path(typ='mod'), os.path.join(installpath, modsuffix))
        repo = init_repository(get_repository(), get_repositorypath())
        self.assertTrue(isinstance(repo, FileRepository))
        self.assertEqual(repo.repo, repopath)
        self.assertEqual(log_file_format(return_directory=True), logdir)
        self.assertEqual(log_file_format(), logtmpl)
        self.assertEqual(get_build_log_path(), tmplogdir)

        # redefine config file entries for proper testing below
        buildpath = os.path.join(self.tmpdir, 'my', 'custom', 'test', 'build', 'path')
        sourcepath = os.path.join(self.tmpdir, 'my', 'custom', 'test', 'source', 'path')
        installpath = os.path.join(self.tmpdir, 'my', 'custom', 'test', 'install', 'path')
        repopath = os.path.join(self.tmpdir, 'my', 'custom', 'test', 'repo', 'path')
        logdir = 'somedir_custom'
        logtmpl = 'test-custom-eb-%(name)_%(date)s%(time)s__%(version)s.log'
        tmplogdir = os.path.join(self.tmpdir, 'my', 'custom', 'test', 'tmplogdir')
        softsuffix = 'myfavoritesoftware_custom'
        modsuffix = 'modulesgohere_custom'

        configdict = {
            'buildpath': buildpath,
            'sourcepath': sourcepath,
            'installpath': installpath,
            'repopath': repopath,
            'logdir': logdir,
            'logtmpl': logtmpl,
            'tmplogdir': tmplogdir,
            'softsuffix': softsuffix,
            'modsuffix': modsuffix }

        # create custom config file, and point to it
        mycustomconfigfile = os.path.join(self.tmpdir, 'mycustomconfig.py')
        if not os.path.exists(os.path.dirname(mycustomconfigfile)):
            os.makedirs(os.path.dirname(mycustomconfigfile))
        write_file(mycustomconfigfile, configtxt % configdict)
        os.environ['EASYBUILDCONFIG'] = mycustomconfigfile

        # reconfigure
        init_config()
        cfg_fn = self.configure(args=[])

        # verify configuration
        self.assertEqual(cfg_fn, mycustomconfigfile)
        self.assertEqual(build_path(), buildpath)
        self.assertEqual(source_paths()[0], sourcepath)
        self.assertEqual(install_path(), os.path.join(installpath, softsuffix))
        self.assertEqual(install_path(typ='mod'), os.path.join(installpath, modsuffix))
        repo = init_repository(get_repository(), get_repositorypath())
        self.assertTrue(isinstance(repo, FileRepository))
        self.assertEqual(repo.repo, repopath)
        self.assertEqual(log_file_format(return_directory=True), logdir)
        self.assertEqual(log_file_format(), logtmpl)
        self.assertEqual(get_build_log_path(), tmplogdir)

    def test_generaloption_config(self):
        """Test new-style configuration (based on generaloption)."""
        self.purge_environment()

        # check whether configuration via environment variables works as expected
        prefix = os.path.join(self.tmpdir, 'testprefix')
        buildpath_env_var = os.path.join(self.tmpdir, 'envvar', 'build', 'path')
        os.environ['EASYBUILD_PREFIX'] = prefix
        os.environ['EASYBUILD_BUILDPATH'] = buildpath_env_var
        options = init_config(args=[])
        self.assertEqual(build_path(), buildpath_env_var)
        self.assertEqual(install_path(), os.path.join(prefix, 'software'))
        del os.environ['EASYBUILD_PREFIX']
        del os.environ['EASYBUILD_BUILDPATH']

        # check whether configuration via command line arguments works
        prefix = os.path.join(self.tmpdir, 'test1')
        install = os.path.join(self.tmpdir, 'test2', 'install')
        repopath = os.path.join(self.tmpdir, 'test2', 'repo')
        config_file = os.path.join(self.tmpdir, 'nooldconfig.py')

        write_file(config_file, '')

        args = [
            '--config', config_file,  # force empty oldstyle config file
            '--prefix', prefix,
            '--installpath', install,
            '--repositorypath', repopath,
        ]

        options = init_config(args=args)

        self.assertEqual(build_path(), os.path.join(prefix, 'build'))
        self.assertEqual(install_path(), os.path.join(install, 'software'))
        self.assertEqual(install_path(typ='mod'), os.path.join(install, 'modules'))

        self.assertEqual(options.installpath, install)
        self.assertEqual(options.config, config_file)

        # check mixed command line/env var configuration
        prefix = os.path.join(self.tmpdir, 'test3')
        install = os.path.join(self.tmpdir, 'test4', 'install')
        subdir_software = 'eb-soft'
        args = [
            '--config', config_file,  # force empty oldstyle config file
            '--installpath', install,
        ]

        os.environ['EASYBUILD_PREFIX'] = prefix
        os.environ['EASYBUILD_SUBDIR_SOFTWARE'] = subdir_software

        options = init_config(args=args)

        self.assertEqual(build_path(), os.path.join(prefix, 'build'))
        self.assertEqual(install_path(), os.path.join(install, subdir_software))

        del os.environ['EASYBUILD_PREFIX']
        del os.environ['EASYBUILD_SUBDIR_SOFTWARE']

    def test_generaloption_config_file(self):
        """Test use of new-style configuration file."""
        self.purge_environment()

        oldstyle_config_file = os.path.join(self.tmpdir, 'nooldconfig.py')
        config_file = os.path.join(self.tmpdir, 'testconfig.cfg')

        testpath1 = os.path.join(self.tmpdir, 'test1')
        testpath2 = os.path.join(self.tmpdir, 'testtwo')

        write_file(oldstyle_config_file, '')

        # test with config file passed via command line
        cfgtxt = '\n'.join([
            '[config]',
            'installpath = %s' % testpath2,
        ])
        write_file(config_file, cfgtxt)

        args = [
            '--configfiles', config_file,
            '--debug',
            '--buildpath', testpath1,
        ]
        options = init_config(args=args)

        self.assertEqual(build_path(), testpath1)  # via command line
        self.assertEqual(source_paths(), [os.path.join(os.getenv('HOME'), '.local', 'easybuild', 'sources')])  # default
        self.assertEqual(install_path(), os.path.join(testpath2, 'software'))  # via config file

        # test with config file passed via environment variable
        cfgtxt = '\n'.join([
            '[config]',
            'buildpath = %s' % testpath1,
        ])
        write_file(config_file, cfgtxt)

        os.environ['EASYBUILD_CONFIGFILES'] = config_file
        args = [
            '--debug',
            '--sourcepath', testpath2,
        ]
        options = init_config(args=args)

        self.assertEqual(install_path(), os.path.join(os.getenv('HOME'), '.local', 'easybuild', 'software'))  # default
        self.assertEqual(source_paths(), [testpath2])  # via command line
        self.assertEqual(build_path(), testpath1)  # via config file

        testpath3 = os.path.join(self.tmpdir, 'testTHREE')
        os.environ['EASYBUILD_SOURCEPATH'] = testpath2
        args = [
            '--debug',
            '--installpath', testpath3,
        ]
        options = init_config(args=args)

        self.assertEqual(source_paths(), [testpath2])  # via environment variable $EASYBUILD_SOURCEPATHS
        self.assertEqual(install_path(), os.path.join(testpath3, 'software'))  # via command line
        self.assertEqual(build_path(), testpath1)  # via config file

        del os.environ['EASYBUILD_CONFIGFILES']

    def test_set_tmpdir(self):
        """Test set_tmpdir config function."""
        self.purge_environment()
        for tmpdir in [None, os.path.join(tempfile.gettempdir(), 'foo')]:
            parent = tmpdir
            if parent is None:
                parent = tempfile.gettempdir()

            mytmpdir = set_tmpdir(tmpdir=tmpdir)

            for var in ['TMPDIR', 'TEMP', 'TMP']:
                self.assertTrue(os.environ[var].startswith(os.path.join(parent, 'easybuild-')))
                self.assertEqual(os.environ[var], mytmpdir)
            self.assertTrue(tempfile.gettempdir().startswith(os.path.join(parent, 'easybuild-')))
            tempfile_tmpdir = tempfile.mkdtemp()
            self.assertTrue(tempfile_tmpdir.startswith(os.path.join(parent, 'easybuild-')))
            fd, tempfile_tmpfile = tempfile.mkstemp()
            self.assertTrue(tempfile_tmpfile.startswith(os.path.join(parent, 'easybuild-')))

            # cleanup
            os.close(fd)
            shutil.rmtree(mytmpdir)
            modify_env(os.environ, self.orig_environ)
            tempfile.tempdir = None

    def test_configuration_variables(self):
        """Test usage of ConfigurationVariables."""
        # delete instance of ConfigurationVariables
        ConfigurationVariables.__metaclass__._instances.pop(ConfigurationVariables, None)

        # make sure ConfigurationVariables is a singleton class (only one available instance)
        cv1 = ConfigurationVariables()
        cv2 = ConfigurationVariables()
        cv3 = ConfigurationVariables({'foo': 'bar'})  # note: argument is ignored, an instance is already available
        self.assertTrue(cv1 is cv2)
        self.assertTrue(cv1 is cv3)

    def test_build_options(self):
        """Test usage of BuildOptions."""
        # delete instance of BuildOptions
        BuildOptions.__metaclass__._instances.pop(BuildOptions, None)

        # make sure BuildOptions is a singleton class
        bo1 = BuildOptions()
        bo2 = BuildOptions()
        bo3 = BuildOptions({'foo': 'bar'})  # note: argument is ignored, an instance is already available
        self.assertTrue(bo1 is bo2)
        self.assertTrue(bo1 is bo3)

        # test basic functionality
        BuildOptions.__metaclass__._instances.pop(BuildOptions, None)
        bo = BuildOptions({
            'debug': False,
            'force': True
        })
        self.assertTrue(not bo['debug'])
        self.assertTrue(bo['force'])

        # updating is impossible (methods are not even available)
        self.assertErrorRegex(TypeError, '.*item assignment.*', lambda x: bo.update(x), {'debug': True})
        self.assertErrorRegex(AttributeError, '.*no attribute.*', lambda x: bo.__setitem__(*x), ('debug', True))

        # only valid keys can be set
        BuildOptions.__metaclass__._instances.pop(BuildOptions, None)
        msg = "Encountered unknown keys .* \(known keys: .*"
        self.assertErrorRegex(KeyError, msg, BuildOptions, {'thisisclearlynotavalidbuildoption': 'FAIL'})

        # test init_build_options and build_option functions
        self.assertErrorRegex(KeyError, msg, init_build_options, {'thisisclearlynotavalidbuildoption': 'FAIL'})
        bo = init_build_options({
            'robot_path': '/some/robot/path',
            'stop': 'configure',
        })

        # specific build options should be set
        self.assertEqual(bo['robot_path'], '/some/robot/path')
        self.assertEqual(bo['stop'], 'configure')

        # all possible build options should be set (defaults are used where needed)
        self.assertEqual(sorted(bo.keys()), sorted(BuildOptions.KNOWN_KEYS))

        # there should be only one BuildOptions instance
        bo2 = BuildOptions()
        self.assertTrue(bo is bo2)



def suite():
    return TestLoader().loadTestsFromTestCase(EasyBuildConfigTest)

if __name__ == '__main__':
    #logToScreen(enable=True)
    #setLogLevelDebug()
    unittestmain()
