# Amazon EC2
Ever try to query for some instances with boto? It sucks.

```python
>>> import ec2
>>> ec2.instances.filter(state='running', name__startswith='production')
[...]
```

## Install
`$ pip install ec2`

## Usage
### AWS credentials
Credentials are defined as a global state, either through an environment variable, or in Python.
```python
ec2.credentials.ACCESS_KEY_ID = 'xxx'
ec2.credentials.SECRET_ACCESS_KEY = 'xxx'
ec2.credentials.REGION_NAME = 'us-west-2'  # (optional) defaults to us-east-1
```

Credentials can also be loaded from a CSV file generated by Amazon's IAM.
**Note**: `REGION_NAME` still needs to be specified.
```python
ec2.credentials.from_file('credentials.csv')
```

## Querying
### All instances
```python
ec2.instances.all()
```

### All Security Groups
```python
ec2.security_groups.all()
```

### All Virtual Private Clouds
```python
ec2.vpcs.all()
```

### Filtering
*Filter style is based on Django's ORM*
All filters map directly to instance/security group properties.
```python
ec2.instances.filter(id='i-xxx')  # Exact instance id
ec2.instances.filter(state='running')  # Exact instance state
```

Filters will also dig into tags.
```python
ec2.instances.filter(name='production-web')  # Exact "Name" tag
```

Filters support many types of comparisons, similar to Django's ORM filters.
```python
ec2.instances.filter(name__exact='production-web-01')  # idential to `name='...'`
ec2.instances.filter(name__iexact='PRODUCTION-WEB-01')  # Case insensitive "exact"
ec2.instances.filter(name__like=r'^production-web-\d+$')  # Match against a regular expression
ec2.instances.filter(name__ilike=r'^production-web-\d+$')  # Case insensitive "like"
ec2.instances.filter(name__contains='web')  # Field contains the search string
ec2.instances.filter(name__icontains='WEB')  # Case insensitive "contains"
ec2.instances.filter(name__startswith='production')  # Fields starts with the search string
ec2.instances.filter(name__istartswith='PRODUCTION')  # Case insensitive "startswith"
ec2.instances.filter(name__endswith='01')  # Fields ends with the search string
ec2.instances.filter(name__iendswith='01')  # Case insensitive "endswith"
ec2.instances.filter(name__isnull=False)  # Match if the field exists
```

Filters can also be chained.
```python
ec2.instances.filter(state='running', name__startswith='production')
```

Filters can also be used with security groups.
```python
ec2.security_groups.filter(name__iexact='PRODUCTION-WEB')
```

Filters can also be used with virtual private clouds.
```python
ec2.vpcs.filter(cidr_blocks__startswith='10.10')
```

`get()` works exactly the same as `filter()`, except it returns just one instance and raises an exception for anything else.
```python
ec2.instances.get(name='production-web-01')  # Return a single instance
ec2.instances.get(name='i-dont-exist')  # Raises an `ec2.instances.DoesNotExist` exception
ec2.instances.get(name__like=r'^production-web-\d+$')  # Raises an `ec2.instances.MultipleObjectsReturned` exception if matched more than one instance
ec2.security_groups.get(name__startswith='production')  # Raises an `ec2.security_groups.MultipleObjectsReturned` exception
ec2.vpcs.get(cidr_block='10.10.0.0/16')
```

### Search fields
#### Instances
 * id *(Instance id)*
 * state *(running, terminated, pending, shutting-down, stopping, stopped)*
 * public_dns_name
 * ip_address
 * private_dns_name
 * private_ip_address
 * root_device_type *(ebs, instance-store)*
 * key_name *(name of the SSH key used on the instance)*
 * image_id *(Id of the AMI)*

All fields can be found at: https://github.com/boto/boto/blob/d91ed8/boto/ec2/instance.py#L157-204

#### Security Groups
 * id *(Security Group id)*
 * name
 * vpc_id

#### Virtual Private Clouds
 * id *(Virtual Private Cloud id)*
 * cidr_block *(CIDR Network Block of the VPC)*
 * state *(Current state of the VPC, creation is not instant)*
 * is_default
 * instance_tenancy
 * dhcp_options_id *(DHCP options id)*


## Examples
### Get public ip addresses from all running instances who are named production-web-{number}
```python
import ec2
ec2.credentials.ACCESS_KEY_ID = 'xxx'
ec2.credentials.SECRET_ACCESS_KEY = 'xxx'

for instance in ec2.instances.filter(state='running', name__like=r'^production-web-\d+$'):
    print instance.ip_address
```

### Add a role to a security group
```python
import ec2
ec2.credentials.ACCESS_KEY_ID = 'xxx'
ec2.credentials.SECRET_ACCESS_KEY = 'xxx'

try:
    group = ec2.security_groups.get(name='production-web')
except ec2.security_groups.DoesNotExist:
    import sys
    sys.stderr.write('Not found.')
    sys.exit(1)
group.authorize('tcp', 80, 80, cidr_ip='0.0.0.0/0')
```
