#! /usr/bin/env python
# -*- coding: utf-8 -*-
#
# Interpreter version: python 2.7
#
# HTML parser generated by Autoparser
# (https://github.com/edeposit/edeposit.amqp.harvester)
#
import os
import os.path

import httpkie
import dhtmlparser


# Utilities
def _get_source(link):
    """
    Return source of the `link` whether it is filename or url.

    Args:
        link (str): Filename or URL.

    Returns:
        str: Content.

    Raises:
        UserWarning: When the `link` couldn't be resolved.
    """
    if link.startswith("http://") or link.startswith("https://"):
        down = httpkie.Downloader()
        return down.download(link)

    if os.path.exists(link):
        with open(link) as f:
            return f.read()

    raise UserWarning("html: '%s' is neither URL or data!" % link)


def _get_encoding(dom, default="utf-8"):
    """
    Try to look for meta tag in given `dom`.

    Args:
        dom (obj): pyDHTMLParser dom of HTML elements.
        default (default "utr-8"): What to use if encoding is not found in
                                   `dom`.

    Returns:
        str/default: Given encoding or `default` parameter if not found.
    """
    encoding = dom.find("meta", {"http-equiv": "Content-Type"})

    if not encoding:
        return default

    encoding = encoding[0].params.get("content", None)

    if not encoding:
        return default

    return encoding.lower().split("=")[-1]


def handle_encodnig(html):
    """
    Look for encoding in given `html`. Try to convert `html` to utf-8.

    Args:
        html (str): HTML code as string.

    Returns:
        str: HTML code encoded in UTF.
    """
    encoding = _get_encoding(
        dhtmlparser.parseString(
            html.split("</head>")[0]
        )
    )

    if encoding == "utf-8":
        return html

    return html.decode(encoding).encode("utf-8")


def is_equal_tag(element, tag_name, params, content):
    """
    Check is `element` object match rest of the parameters.

    All checks are performed only if proper attribute is set in the HTMLElement.

    Args:
        element (obj): HTMLElement instance.
        tag_name (str): Tag name.
        params (dict): Parameters of the tag.
        content (str): Content of the tag.

    Returns:
        bool: True if everyhing matchs, False otherwise.
    """
    if tag_name and tag_name != element.getTagName():
        return False

    if params and not element.containsParamSubset(params):
        return False

    if content is not None and content.strip() != element.getContent().strip():
        return False

    return True


def has_neigh(tag_name, params=None, content=None, left=True):
    """
    This function generates functions, which matches all tags with neighbours
    defined by parameters.

    Args:
        tag_name (str): Tag has to have neighbour with this tagname.
        params (dict): Tag has to have neighbour with this parameters.
        params (str): Tag has to have neighbour with this content.
        left (bool, default True): Tag has to have neigbour on the left, or
                                   right (set to ``False``).

    Returns:
        bool: True for every matching tag.

    Note:
        This function can be used as parameter for ``.find()`` method in
        HTMLElement.
    """
    def has_neigh_closure(element):
        if not element.parent \
           or not (element.isTag() and not element.isEndTag()):
            return False

        # filter only visible tags/neighbours
        childs = element.parent.childs
        childs = filter(
            lambda x: (x.isTag() and not x.isEndTag()) \
                      or x.getContent().strip() or x is element,
            childs
        )
        if len(childs) <= 1:
            return False

        ioe = childs.index(element)
        if left and ioe > 0:
            return is_equal_tag(childs[ioe - 1], tag_name, params, content)

        if not left and ioe + 1 < len(childs):
            return is_equal_tag(childs[ioe + 1], tag_name, params, content)

        return False

    return has_neigh_closure


# Generated parsers
def get_publisher(dom):
    el = dom.find(
        'td',
        {'class': 'property-value'},
        fn=has_neigh('td', {'class': 'property-name'}, 'Vydavatel', left=True)
    )

    if not el:
        raise UserWarning(
            "Can't locate variable 'publisher' with content 'Zoner Press'!\n" +
            'Tag name: td' + '\n' +
            'El:' + str(el) + '\n' +
            'Dom:' + str(dom)
        )

    el = el[0]

    return el


def get_ISBN(dom):
    el = dom.find(
        'td',
        {'class': 'property-value'},
        fn=has_neigh('td', {'class': 'property-name'}, 'ISBN', left=True)
    )

    # pick element from list
    el = el[0] if el else None

    return el


def get_title(dom):
    el = dom.find(
        'h2',
        {'class': 'nadpisStrankySloupec2'},
        fn=has_neigh('input', {'type': 'hidden', 'name': 'action', 'value': 'addtocart', 'id': 'ctl01_ctl07_ctl00_ctl00_ctl00_ctl00_ctl00_ctl00_ctl00_ctl08'}, '', left=True)
    )

    if not el:
        raise UserWarning(
            "Can't locate variable 'title' with content 'O\xc4\x8dima fotografa: Grafick\xc3\xbd pr\xc5\xafvodce'!\n" +
            'Tag name: input' + '\n' +
            'El:' + str(el) + '\n' +
            'Dom:' + str(dom)
        )

    el = el[0]

    return el


def get_price(dom):
    el = dom.match(['div', {'class': 'obalka'}], ['p', {'class': 'puvodniCena'}], ['span', None])

    if not el:
        raise UserWarning(
            "Can't locate variable 'price' with content '360&nbsp;K\xc4\x8d'!\n" +
            "Tag name: ['div', {'class': 'obalka'}]" + '\n' +
            'El:' + str(el) + '\n' +
            'Dom:' + str(dom)
        )

    el = el[0]

    return el


def get_author(dom):
    el = dom.find(
        'td',
        {'class': 'property-value'},
        fn=has_neigh('td', {'class': 'property-name'}, 'Autor', left=True)
    )

    # pick element from list
    el = el[0] if el else None

    return el


def get_binding(dom):
    el = dom.find(
        'td',
        {'class': 'property-value'},
        fn=has_neigh('td', {'class': 'property-name'}, 'Typ vazby', left=True)
    )

    # pick element from list
    el = el[0] if el else None

    return el


def get_pub_date(dom):
    el = dom.find(
        'td',
        {'class': 'property-value'},
        fn=has_neigh('td', {'class': 'property-name'}, 'Rok vyd\xc3\xa1n\xc3\xad', left=True)
    )

    # pick element from list
    el = el[0] if el else None

    return el


def get_pages(dom):
    el = dom.find(
        'td',
        {'class': 'property-value'},
        fn=has_neigh('td', {'class': 'property-name'}, 'Po\xc4\x8det stran', left=True)
    )

    # pick element from list
    el = el[0] if el else None

    return el


# Unittest
def test_parsers():
    # Test parsers against http://www.zonerpress.cz/e-kniha-vyvoj-univerzalnych-aplikacii-pre-windows-8-a-windows-phone-8.1
    html = handle_encodnig(
        _get_source('http://www.zonerpress.cz/e-kniha-vyvoj-univerzalnych-aplikacii-pre-windows-8-a-windows-phone-8.1')
    )
    dom = dhtmlparser.parseString(html)
    dhtmlparser.makeDoubleLinked(dom)

    publisher = get_publisher(dom)
    assert publisher.getContent().strip() == 'Zoner Press'

    ISBN = get_ISBN(dom)
    assert ISBN.getContent().strip() == '978-80-7413-282-7'

    title = get_title(dom)
    assert title.getContent().strip() == 'E-kniha: V\xc3\xbdvoj univerz\xc3\xa1lnych aplik\xc3\xa1ci\xc3\xad pre windows 8 a Windows Phone 8.1'

    price = get_price(dom)
    assert price.getContent().strip() == '199&nbsp;K\xc4\x8d'

    author = get_author(dom)
    assert author.getContent().strip().split() == ['Luboslav', 'Lacko', '(<a', 'href="http://www.zonerpress.cz/inshop/scripts/shop.aspx?action=DoSearch&limitedlevels=1&ParamID_1=Luboslav', 'Lacko">Zobrazit', 'v\xc5\xa1echny', 'knihy', 'tohoto', 'autora</a>)']

    binding = get_binding(dom)
    assert binding.getContent().strip() == 'bro\xc5\xbeovan\xc3\xa1'

    pub_date = get_pub_date(dom)
    assert pub_date.getContent().strip() == '2014 (e-kniha)'

    pages = get_pages(dom)
    assert pages.getContent().strip() == '96 (pap\xc3\xadrov\xc3\xa1 kniha)'

    # Test parsers against http://www.zonerpress.cz/ocima-fotografa-graficky-pruvodce
    html = handle_encodnig(
        _get_source('http://www.zonerpress.cz/ocima-fotografa-graficky-pruvodce')
    )
    dom = dhtmlparser.parseString(html)
    dhtmlparser.makeDoubleLinked(dom)

    publisher = get_publisher(dom)
    assert publisher.getContent().strip() == 'Zoner Press'

    ISBN = get_ISBN(dom)
    assert ISBN.getContent().strip() == '978-80-7413-275-9'

    title = get_title(dom)
    assert title.getContent().strip() == 'O\xc4\x8dima fotografa: Grafick\xc3\xbd pr\xc5\xafvodce'

    price = get_price(dom)
    assert price.getContent().strip() == '360&nbsp;K\xc4\x8d'

    author = get_author(dom)
    assert author.getContent().strip().split() == ['Michael', 'Freeman', '(<a', 'href="http://www.zonerpress.cz/inshop/scripts/shop.aspx?action=DoSearch&limitedlevels=1&ParamID_1=Michael', 'Freeman">Zobrazit', 'v\xc5\xa1echny', 'knihy', 'tohoto', 'autora</a>)']

    binding = get_binding(dom)
    assert binding.getContent().strip() == 'M\xc4\x9bkk\xc3\xa1 s klopnami'

    pub_date = get_pub_date(dom)
    assert pub_date.getContent().strip() == '2014'

    pages = get_pages(dom)
    assert pages.getContent().strip() == '192'

    # Test parsers against http://www.zonerpress.cz/konec-prokrastinace
    html = handle_encodnig(
        _get_source('http://www.zonerpress.cz/konec-prokrastinace')
    )
    dom = dhtmlparser.parseString(html)
    dhtmlparser.makeDoubleLinked(dom)

    publisher = get_publisher(dom)
    assert publisher.getContent().strip() == 'Jan Melvil Publishing'

    ISBN = get_ISBN(dom)
    assert ISBN.getContent().strip() == '978-80-87270-51-6'

    title = get_title(dom)
    assert title.getContent().strip() == 'Konec prokrastinace'

    price = get_price(dom)
    assert price.getContent().strip() == '349&nbsp;K\xc4\x8d'

    author = get_author(dom)
    assert author.getContent().strip().split() == ['Petr', 'Ludwig', '(<a', 'href="http://www.zonerpress.cz/inshop/scripts/shop.aspx?action=DoSearch&limitedlevels=1&ParamID_1=Petr', 'Ludwig">Zobrazit', 'v\xc5\xa1echny', 'knihy', 'tohoto', 'autora</a>)']

    binding = get_binding(dom)
    assert binding.getContent().strip() == 'bro\xc5\xbeovan\xc3\xa1 s chlopn\xc4\x9bmi'

    pub_date = get_pub_date(dom)
    assert pub_date.getContent().strip() == '2013'

    pages = get_pages(dom)
    assert pages.getContent().strip() == '272'


# Run tests of the parser
if __name__ == '__main__':
    test_parsers()
