# coding=UTF-8
import datetime, sys
from etmVersion import version
from etmParsers import *
from etmData import relpath

colorMainBG = 'white'
colorMainFG = 'black'

contexts = ['home', 'errands', 'work']

today = datetime.date.today()
thisyear = today.strftime("%Y")

if thisyear == '2009':
    copyright = '2009'
else:
    copyright = '2009-%s' % thisyear

def sysinfo():
    from platform import python_version as pv
    from dateutil import __version__ as dv
    try:
        import wx
        wxv = "%s.%s.%s" % (wx.MAJOR_VERSION, 
                wx.MINOR_VERSION, wx.RELEASE_VERSION)
    except:
        wxv = "none"
    sysinfo = "platform: %s; python %s; dateutil %s; wx(Python) %s" %\
            (sys.platform, pv(), dv, wxv)
    return(sysinfo)

def etminfo():
    paths = [etmdata, etmActions, etmEvents, etmTasks, etmNotes]
    common_prefix = os.path.commonprefix(paths)
    tmp = ["etmdir: '%s'; common prefix for data files: '%s'<br>" % (etmdir, common_prefix)]
    for n, p in [('etmdata', etmdata), ('etmActions', etmActions),
            ('etmEvents', etmEvents), ('etmNotes', etmNotes),
            ('etmTasks', etmTasks)]:
        r = relpath(p, common_prefix)
        if r:
            tmp.append("%s: '%s';" % (n, r))
    etminfo = " ".join(tmp)
    return(etminfo)

# For About
description = """\
etm provides a format for using simple text files to store 
event, task and action information, a command line interface 
for viewing tasks and events in a variety of convenient ways 
and a wx(python)-based GUI for creating and modifying events 
and tasks as well as viewing them.
"""

# license = "GNU General Public License (GPL)"
license = """\
This program is free software; you can redistribute it 
and/or modify it under the terms of the GNU General Public 
License as published by the Free Software Foundation; either 
version 2 of the License, or (at your option) any later 
version. [ http://www.gnu.org/licenses/gpl.html ]

This program is distributed in the hope that it will be 
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE. See the GNU General Public License for more 
details.
"""

entry_help = """
<em>&lt;Esc&gt; cancels entry; &lt;Shift-Return&gt; accepts entry.<br> Press TAB to select an insertion from a from a list of templates.<br> Press Shift and Space after entering @c [word], @k [word], @l [word] or @z [word] to complete [word] from a matching list of previously used c)ontexts, k)eywords, l)ocations or time z)ones.</em> """ 

project_html="""
<title>project options</title>
<body text="%(colorMainFG)s" bgcolor="%(colorMainBG)s"><font size="+1"><b>The project line format</b></font>
<pre>
%(project_text)s
</pre>
%(entry_help)s
</body>
"""
examples_html= """
<title>etm</title>
<body text="%(colorMainFG)s" bgcolor="%(colorMainBG)s">
<center><font size="+1"><b>Item Examples</b></font></center>
<pre>
<b>Repetition</b>
    Get a haircut (a task) [r]epeatedly at (d)aily [i]ntervals of (14) 
    days and, [o]n completion, (r)estart from the completion date:

        - get haircut @d 2011-07-08 @r d @i 14 @o r

    Take out the trash (a task) [r]epeatedly at (w)eekly intervals on 
    [w]eekday Friday and, [o]n completion, (s)kip to the next Friday.

        - take out trash @d 2011-07-08 @r w @w FR @o s

    Payday (an event) on the last weekday of each month. The '@S -1' 
    entry extracts the last date which is both a weekday and falls 
    within the last three days of the month. E.g., when the last day of 
    the month is a Sunday, then the day selected will be FR, -3.

        * payday @d 2010-07-01 @r m @w (MO, TU, WE, TH, FR) 
            @m (-1, -2, -3) @S -1

    Payday on the last weekday on or before the 25th of the month.

        * payday @d 2010-07-01 @r m @w (MO, TU, WE, TH, FR) 
            @m (23, 24, 25) @S -1

    Prepare income tax return (a task) (y)early by April 15 with a 
    warning to (b)egin the task 30 days earlier.

        - gather tax related receipts @d 2012-04-15 @r y @M 4 @m 15 @b 30
        + get W2 forms @d 2012-04-15 @r y @M 4 @m 15 @b 30
        -- prepare tax return @d 2012-04-15 @r y @M 4 @m 15 @b 30

    Note that the '+' before 'get' means that 'gather' will join 'get' as
    prerequisites for 'prepare'. If the return actually must be filed by
    the first weekday on or after April 15, then the repetition part could
    be changed to '@r y @M 4 @m (15, 16, 17) @w range(0, 5) @S 1' where 
    'range(0, 5)' is equivalent to '(0,1,2,3,4)' or '(MO,TU,WE,TH,FR)'
    and '@S 1' extracts the first date that is a weekday and falls on 
    or after the 15th.

    Take a prescribed medication daily (an event) from the 23rd through 
    the 27th at 10am, 2pm, 6pm and 10pm and trigger an alert at each
    of these times:

        * take Rx @d 23 @s (10a, 2p, 6p, 10p) @r d @u 27 @a 0 

    Presidential election day (an event) every four years on the first 
    Tuesday after a Monday in November:

        * Presidential Election Day @d 2012-11-06 @r y @i 4 @M 11 
            @m range(2, 9) @w TU 

     where 'range(2, 9)' is equivalent to '(2,3,4,5,6,7,8)'. 
</pre>
</body>
"""

add_html= """
<title>etm</title>
<body text="%(colorMainFG)s" bgcolor="%(colorMainBG)s">
<center>
<a name="toc"><font size="+1" color="blue"><b>Contents</b></font></a>
</center>

<pre><a href="#basic" id="id1">Basic Options</a>
<a href="#repetition" id="id1">Repetition</a></pre>
<hr width="100%%"/>

<pre><a name="basic" href="#toc"><b>Basic Options</b></a></pre>

Press either 'a' or '~' to create an action, 'e' or '*' to create an 
event, 'n' or '!' to create a note and either 't', '-' or '+' to create 
a task. Optional (O) and required (R) fields are listed below.
<table>
<tr><th>Key</th><th>Value</th>
    <th>Action</th>
    <th>Event</th>
    <th>Note</th>
    <th>Task</th>
    <th align="left">Description</th></tr>

<tr>
    <td valign="top">@d</td>
    <td valign="top">DATE</td>
    <td align="center" valign="top">R</td>
    <td align="center" valign="top">R</td>
    <td align="center" valign="top">O</td>
    <td align="center" valign="top">O</td>
    <td valign="top">
        a date (fuzzy parsed).
    </td>
</tr>

<tr>
    <td valign="top">@b</td>
    <td valign="top">BEGIN</td>
    <td align="center" valign="top">&nbsp;</td>
    <td align="center" valign="top">&nbsp;</td>
    <td align="center" valign="top">&nbsp;</td>
    <td align="center" valign="top">O</td>
    <td valign="top">
        an integer number of days before a task with a due date at which 
        to begin giving advance warnings of the upcomming due date.
    </td>
</tr>

<tr>
    <td valign="top">@f</td>
    <td valign="top">FINISH</td>
    <td align="center" valign="top">&nbsp;</td>
    <td align="center" valign="top">&nbsp;</td>
    <td align="center" valign="top">&nbsp;</td>
    <td align="center" valign="top">O</td>
    <td valign="top">
        a date or, for a repeating task, a list of dates  in 
        parentheses, on which the task was completed (fuzzy parsed).
    </td>
</tr>

<tr>
    <td valign="top">@s</td>
    <td valign="top">STARTTIME</td>
    <td align="center" valign="top">&nbsp;</td>
    <td align="center" valign="top">O</td>
    <td align="center" valign="top">&nbsp;</td>
    <td align="center" valign="top">&nbsp;</td>
    <td valign="top">
        a time or a list of times in parentheses (fuzzy parsed). An 
        event without a start time is regarded as an all day event.
    </td>
</tr>

<tr>
    <td valign="top">@a</td>
    <td valign="top">ALERTS</td>
    <td align="center" valign="top">&nbsp;</td>
    <td align="center" valign="top">O</td>
    <td align="center" valign="top">&nbsp;</td>
    <td align="center" valign="top">&nbsp;</td>
    <td valign="top">
        an interger or a list of integer minutes in parentheses. Alerts 
        will be triggered at times corresponding the these numbers of 
        minutes before STARTTIME.
    </td>
</tr>

<tr>
    <td valign="top">@A</td>
    <td valign="top">ALTCMD</td>
    <td align="center" valign="top">&nbsp;</td>
    <td align="center" valign="top">O</td>
    <td align="center" valign="top">&nbsp;</td>
    <td align="center" valign="top">&nbsp;</td>
    <td valign="top">
        an alternative command to run when alerts are triggered instead 
        of the default. Accepts the same substitution strings as 
        <tt>alertcmd</tt> in the etm rc file.
    </td>
</tr>

<tr>
    <td valign="top">@e</td>
    <td valign="top">EXTENT</td>
    <td align="center" valign="top">R</td>
    <td align="center" valign="top">O</td>
    <td align="center" valign="top">&nbsp;</td>
    <td align="center" valign="top">O</td>
    <td valign="top">
        a integer number of minutes preceeded by a "<tt>+</tt>" (plus 
        sign). Optionally, an ending time (fuzzy parsed) can be given 
        for an event with a single starting time. An event with 
        one or more start times but without an extent is treated as a 
        reminder.
    </td>
</tr>

<tr>
    <td valign="top">@n</td>
    <td valign="top">NOTE</td>
    <td align="center" valign="top">O</td>
    <td align="center" valign="top">O</td>
    <td align="center" valign="top">O</td>
    <td align="center" valign="top">O</td>
    <td valign="top">
        a string. Newlines and leading whitespace are preserved within 
        the body of the note.
    </td>
</tr>

<tr>
    <td valign="top">@g</td>
    <td valign="top">GOTO</td>
    <td align="center" valign="top">O</td>
    <td align="center" valign="top">O</td>
    <td align="center" valign="top">O</td>
    <td align="center" valign="top">O</td>
    <td valign="top">
        a file path or URL, or a comma separated list of paths or URLs in 
        parentheses, to be opened using the system default application if 
        the user presses "g" when the item is selected. If a list is 
        given, a selection list will be provided from which to choose the 
        item to be opened.
    </td>
</tr>

<tr>
    <td valign="top">@z</td>
    <td valign="top">TIMEZONE</td>
    <td align="center" valign="top">O</td>
    <td align="center" valign="top">O</td>
    <td align="center" valign="top">O</td>
    <td align="center" valign="top">O</td>
    <td valign="top">
        a time zone, e.g., "US/Eastern".
    </td>
</tr>

<tr>
    <td valign="top">@l</td>
    <td valign="top">LOCATION</td>
    <td align="center" valign="top">O</td>
    <td align="center" valign="top">O</td>
    <td align="center" valign="top">O</td>
    <td align="center" valign="top">O</td>
    <td valign="top">
        a string.
    </td>
</tr>

<tr>
    <td valign="top">@U</td>
    <td valign="top">USER</td>
    <td align="center" valign="top">O</td>
    <td align="center" valign="top">O</td>
    <td align="center" valign="top">O</td>
    <td align="center" valign="top">O</td>
    <td valign="top">
        a string.
    </td>
</tr>

<tr>
    <td valign="top">@c</td>
    <td valign="top">CONTEXT</td>
    <td align="center" valign="top">O</td>
    <td align="center" valign="top">O</td>
    <td align="center" valign="top">O</td>
    <td align="center" valign="top">O</td>
    <td valign="top">
        a string.
    </td>
</tr>

<tr>
    <td valign="top">@k</td>
    <td valign="top">KEYWORD</td>
    <td align="center" valign="top">O</td>
    <td align="center" valign="top">O</td>
    <td align="center" valign="top">O</td>
    <td align="center" valign="top">O</td>
    <td valign="top">
        a string which optionally contains colons, e.g, "@k 
        first:second:third:fourth". When grouping/sorting 'k1' would refer to 
        'first', 'k2' to 'second' and 'k3' to 'third:fourth'.
    </td>
</tr>

<tr>
    <td valign="top">@p</td>
    <td valign="top">PRIORITY</td>
    <td align="center" valign="top">&nbsp;</td>
    <td align="center" valign="top">&nbsp;</td>
    <td align="center" valign="top">&nbsp;</td>
    <td align="center" valign="top">O</td>
    <td valign="top">
        an integer in the range from 1 (highest priority) to 9 (lowest 
        priority).
    </td>
</tr>

<tr>
    <td valign="top">@t</td>
    <td valign="top">TAGS</td>
    <td align="center" valign="top">O</td>
    <td align="center" valign="top">O</td>
    <td align="center" valign="top">O</td>
    <td align="center" valign="top">O</td>
    <td valign="top">
        a string or a comma separated list of strings in parenthesis, 
        e.g, "@t (tag 1, tag 2)".
    </td>
</tr>
</table>

<pre><a name="repetition" href="#toc"><b>Repetition</b></a></pre>

<p>
Events and task can be repeated using further keys and values.</p>

<table>
<tr>
    <th>Key</th>
    <th>Value</th>
    <th align="left">Description</th></tr>

<tr>
    <td valign="top">@r</td>
    <td valign="top">REPEAT</td>
    <td valign="top">
        a single character from "d" (daily), "w" (weekly), "m" 
        (monthly), "y" (yearly) or "l" (list) describing the frequency 
        at which the event or task repeats. <font color="red">This field
        is always required for repeating items.</font>.</td>
</tr>

<tr>
    <td valign="top">@i</td>
    <td valign="top">INTERVAL</td>
    <td valign="top">
        a positive integer interval at which the task repeats. E.g., "<tt>@r 
        w @i 3</tt>" would repeat every third week. Default: 1.
    </td>
</tr>

<tr>
    <td valign="top">@u</td>
    <td valign="top">UNTIL</td>
    <td valign="top">
        a date at which repetitions should end (fuzzy parsed). Default: 
        none, i.e., repetitions continue forever.
    </td>
</tr>

<tr>
    <td valign="top">@C</td>
    <td valign="top">COUNT</td>
    <td valign="top">
        an integer number of repetitions to allow.  E.g., "<tt>@d 
        2011-01-01 @r m @m (15, 30) @C 4</tt>" would repeat on Jan 15, 
        Jan 30, Feb 15 and Mar 15 of 2011. Default: none, i.e., 
        repetitions continue forever.
    </td>
</tr>

<tr>
    <td valign="top">@w</td>
    <td valign="top">WEEKDAY</td>
    <td valign="top">
        a string or list of strings in parentheses from 
        "MO", "TU", "WE", "TH", "FR", "SA", "SU" or an integer or list 
        of integers from 0 (MO) through 6 (SU). Default: any weekday. 
        E.g., "<tt>@r m @w (MO(1), FR(-1))</tt>" would repeat on the 
        first Monday and the last Friday of each month.
    </td>
</tr>

<tr>
    <td valign="top">@W</td>
    <td valign="top">WEEKNUM</td>
    <td valign="top">
        an integer week number or a list of week numbers from 0 through 
        53. Default: any week number. 
    </td>
</tr>

<tr>
    <td valign="top">@m</td>
    <td valign="top">MONTHDAY</td>
    <td valign="top">
        an integer month day number or a list of month day numbers from 
        1 through 31. Note that "<tt>@r m @m 31</tt>" would only repeat 
        on months that have 31 days but "<tt>@r m @m -1</tt>" would 
        repeat on the last day of every month. Default: any month day.
    </td>
</tr>

<tr>
    <td valign="top">@M</td>
    <td valign="top">MONTHNUM</td>
    <td valign="top">
        an integer month number or a list of month numbers from 
        1 through 12. Default: any month number.
    </td>
</tr>

<tr>
    <td valign="top">@+</td>
    <td valign="top">INCLUDEDATES</td>
    <td valign="top">
        a date or a list of dates to add to the dates that would 
        otherwise be generated by the recurrence rule. E.g., "<tt>@r m @m 1 
        @+ (Feb 5, Feb 7)</tt>" would repeat on the 1st of each month 
        and, additionally, on Feb 5 and Feb 7. Note that it is possible 
        with this field to specify specific dates for an item. E.g., "<tt>@r 
        l @+ (Feb 28, Jul 3, Aug 27)</tt>" would repeat only on the 
        three specified dates. When '@r l' (list) is used an entry for '@+ 
        INCLUDEDATES' is required and an entry for '@d DATE' is optional. 
        Otherwise, an entry for '@d DATE' is required and an entry for '@+ 
        INCLUDEDATES' is optional. Default: none.
    </td>
</tr>

<tr>
    <td valign="top">@-</td>
    <td valign="top">EXCLUDEDATES</td>
    <td valign="top">
        a date or a list of dates to remove from the dates that would 
        otherwise be generated by the recurrence rule. E.g., "<tt>@r m @m 1 
        @- (Mar 1, Jun 1)</tt>" would repeat on the 1st of each month 
        except for Mar 1 and Jun 1. Default: none.
    </td>
</tr>

<tr>
    <td valign="top">@S</td>
    <td valign="top">SETPOSITION</td>
    <td valign="top">
        a integer specifying the date from the list of dates that 
        would otherwise satisfy the recurrence rule. E.g., "@r m @w (MO, 
        TU, WE, TH, FR) @m (23, 24, 25)" would repeat on every 
        weekday that is a 23rd, 24th or 25th of the month. For June 
        2011, for example, this would include Thursday, June 23 and 
        Friday, June 24. By adding "@S -1", only the last of these, 
        Friday, June 24, would be included. Thus  "@r m @w (MO, 
        TU, WE, TH, FR) @m (23, 24, 25) @S -1" would repeat on the
        last weekday in each month that falls on or before the 25th. 
        Default: none.
    </td>
</tr>

<tr>
    <td valign="top">@o</td>
    <td valign="top">OVERDUE</td>
    <td valign="top">
        either 'k' (keep), 's' (skip) or 'r' (restart). The 
        specification of how to handle due dates for repeating tasks. 
        This is explained below. Default: k. 
    </td>
</tr>

</table>

    <p>A repeating task that is finished on its due date presents no 
    difficulty. But what if it's finished early or becomes overdue? 
    There are three options with etm:</p>

    <p>@o k: Keep the current due date if it becomes overdue and use the 
    next due date from the recurrence rule if it is finished early. This 
    would be appropriate, for example, for the task 'file tax return'. 
    The return due April 15, 2009 must still be filed even if it is 
    overdue and the 2010 return won't be due until April 15, 2010 even 
    if the 2009 return is finished early.</p>

    <p>@o s: Skip overdue due dates and set the due date for the next 
    repetition to the first due date from the recurrence rule on or 
    after the current date. This would be appropriate, for example, for 
    the task 'put out the trash' since there is no point in putting it 
    out on Tuesday if it's picked up on Mondays. You might just as well 
    wait until the next Monday to put it out. There's also no point in 
    being reminded until the next Monday.</p>

    <p>@o r: Restart the repetitions based on the last completion date. 
    Suppose you want to mow the grass once every ten days and that when 
    you mowed yesterday, you were already nine days past due. Then you 
    want the next due date to be ten days from yesterday and not today. 
    Similarly, if you were one day early when you mowed yesterday, then 
    you would want the next due date to be ten days from yesterday and 
    not ten days from today.</p>

</body>
"""

helpview_html = """\
        <title>etm: event and task manager</title>
<center>
etm version %(version)s. Copyright %(copyright)s Daniel A Graham. All rights reserved.
<br>
%(etminfo)s
<br>
%(sysinfo)s
<hr width="100%%"/>
<a name="toc"><font size="+1" color="blue"><b>Contents</b></font></a>
</center>
<pre>
<a href="#gui" id="id1">Overview</a>
    <a href="#interface" id="id1">GUI Display</a>
    <a href="#quick" id="id1">Quick Start</a>
    <a href="#classifiers" id="id1">Item Classifiers</a>
    <a href="#fuzzy" id="id1">Dates, Times and Alerts</a>
    <a href="#zone" id="id1">Time Zones</a>
    <a href="#grouping" id="id1">Grouping and Sorting</a>
    <a href="#regex" id="id1">Regular Expression Filters</a>
<a href="#key" id="id1">Key Bindings</a>
    <a href="#commands" id="id1">General</a>
    <a href="#navigation" id="id1">Outline Panel</a>
    <a href="#details" id="id1">Details Panel</a>
    <a href="#html" id="id1">HTML Windows</a>
<a href="#files" id="id1">Files</a>
    <a href="#config" id="id1">Configuration</a>
    <a href="#rotate" id="id1">Monthly</a>
<hr width="100%%"/>
<a name="gui" href="#toc"><b>Overview</b></a>

  <a name="interface" href="#toc"><b>GUI Display</b></a>
<font color="darkgreen">
</pre>

<center>
<table rows="9" cols="2" width="70%%" border="1" 
padding="0" spacing="0">
<tr>
    <td colspan="2" align="center"> Date Bar</td>
</tr>
<tr>
    <td>
        <table rules="all" border="1" frame="void" padding="0" 
        spacing="0" rows="9" width="100%%" height="100%%">
            <tr rowspan="5">
                <td valign="center" align="center">
                    &nbsp;<br>
                    Monthly<br>
                    Calendar<br>
                    &nbsp;
                </td>
            </tr>
            <tr rowspan="7">
                <td valign="center" align="center">
                    &nbsp;<br> 
                    &nbsp;<br>
                    Busy<br>
                    Panel<br>
                    &nbsp;<br> 
                    &nbsp;
                </td>
            </tr>
        </table>
    </td>
    <td>
        <table rules="all" border="1" padding="0" 
        spacing="0" rows="9" width="100%%" height="100%%">
            <tr rowspan="8" width="70%%">
                <td valign="center" align="center">
                    &nbsp;<br>
                    &nbsp;<br> 
                    &nbsp;<br> 
                    Outline Panel<br>
                    &nbsp;<br> 
                    &nbsp;<br> 
                    &nbsp;
                </td>
            </tr>
            <tr rowspan="4">
                <td valign="center" align="center">
                    &nbsp;<br>
                    Details Panel<br>
                    &nbsp;
                </td>
            </tr>
        </table>
    </td>
</tr>
<tr>
    <td colspan="2" align="center"> Status Line </td>
</tr>
</table>
</center>

<pre>
</font>
    <b>Date bar</b>
        The current date and time is displayed at the left. This is 
        followed by the interval of days currently being displayed and 
        an entry area which permits a case-insensitve search for all 
        items matching a regular expression. Matching items are listed 
        in the outline panel below.

    <b>Monthly calendar</b>
        The color of each month day number reflects the number of 
        minutes of scheduled events for that date. The days of the 
        currently selected week (the selected day and the six following 
        days) are highlighted. The selected day will automatically be
        entered as the initial value of the date for any item you 
        create and, after any data change, etm will open the display to 
        the selected day. 

        With the focus in the calendar, the right and left arrow keys 
        change the selection forward and backward one day at a time. 
        Down and up arrow keys change the selection forward and backward 
        one week at a time and Page Down and Page Up change the 
        selection one month at a time.

    <b>Busy panel</b>
        The periods of schduled events for the selected days in calendar 
        are highlighted. Clicking on a highlighted area will select the 
        corresponding event in the outline panel and double clicking 
        will open the event for editing in the details panel.

    <b>Outline panel</b>
        This panel displays data reflecting the current options in 
        outline form. When an item is selected, the item's details are 
        displayed below in the details panel. 

    <b>Details panel</b>
        When an item is selected in the outline panel, the item's 
        details are displayed here.

        Additionally, this area is used to edit exising items, to create 
        new items and to set the options for the outline panel display 
        and for busy reports. See the <em>Item Options</em> and <em>Display 
        Options</em> help tabs for details.

        Press Shift-Return or Ctrl-S to process the contents or press 
        Escape to cancel.

    <b>Status Line</b>
        At the left, the status line displays a permanent reminder that 
        pressing F1 activates the help system. If one or more alarms is 
        in the queue for the current date, then the time of the next 
        scheduled alarm follows. This is followed by a list of the 
        current display options. The remaining space is used to display 
        a variety of information. If a timer is running for an action, 
        for instance, then the name of the action and the current 
        elapsed time and status (running or paused) is displayed.

  <a name="quick" href="#toc"><b>Quick Start</b></a>

    In etm there are four types of items: 

    <em>Action</em> 
        A record of the time-consuming action required to complete a
        task or participate in an event. Actions are not reminders, 
        they are instead records of how time was actually spent. 
        Action lines begin with a '~' character.
    <em>Event</em>
        Something that will happen on a particular day (or days) 
        and, perhaps, at particular times. Alerts are optional for 
        events with starting times. Event lines begin with a '*'
        character.
    <em>Note</em> 
        A record of some useful information. Note lines begin with a '!' 
        character.
    <em>Task</em>
        Something that needs to be done. It may or may not have a due 
        date. Task lines begin with one or more '-' or '+' characters.

    Etm data files begin with a 'project line' that gives the project 
    title and, perhaps, options. Items follow with each item occupying 
    one or more lines. Each item begins with '~', '*', '!', '-' or '+' 
    as the first character on the line, followed by the title of the 
    item and then field keys and values. The item  continues either 
    until the file ends or another line is found that begins with '~', 
    '*', '!', '-' or '+'. E.g,

    ----------file begins-------------------------------------
    the title of the project file
    <font color="grey">// other items</font>
    * my event title @d 8/23 @s 9a @n A note for my event that
    continues on the next line.
    <font color="grey">// other items</font>
    ----------file ends --------------------------------------

    See the <em>Item Options</em> help tab for details about the formats 
    used for actions, events, notes and tasks. 

    Note that options added to the project line  become the defaults for 
    each of the items in the same file. Suppose, for example, that the 
    project file concerns 'Project A' for client 'John Smith'. Then 
    adding '@k Smith, John:Project A' to the project line would set the 
    client and project keyword for every item in the file.

    To create a new event, press 'e' (or '*') and the details panel will 
    be focused and ready for your entry. To see how this works, press 
    'e' with, say, 2011-06-25 as the selected day and the details panel 
    should open with

        *  | @d 2011-06-25

    With the cursor at '|'. Now change the entry to

        * test event @d 2011-06-25 @s 3p @e 4p 

    and press <b>Shift and Return</b> at the same time to record your 
    event. You will then be prompted for a file to store the entry. 
    Before you choose a file, drag the file selection dialog to one side 
    and notice that your entry has been changed to something like

        * test event @d 2011-06-25 @s (3:00p) @e +1:00 @z Europe/Paris

    with the starting time expanded, the ending time replaced with the 
    extent (duration) of the event and with the local timezone added.

    Now return to the file dialog and note that the default selection is
    something like 'evnts/YYYY/MM_evnts.text' where YYYY is the current
    year and MM is the current month number. Make a selection from the 
    dialog and etm will display "working" in the status line while it 
    records the new event and reloads the data files. The display will 
    then reopen to the selected day where you can see your newly created 
    event.

    Creating an action (press 'a' or '~'), a note (press 'n' or '!') or 
    a task (press 't', '-' or '+') is similar.

    An event without either a starting time or an extent (duratation) is 
    regarded as an <em>all day event</em>. All day events are displayed at the
    top of the daily schedule.

    An event with one or more staring times but without an extent is 
    regarded as a <em>reminder</em>. E.g.,

        * test reminder @d 6/25 @s (10a, 2p, 4p) @a (15, 0)

    would create a reminder for 10:00am, 2:00pm and 4:00pm on June 25 
    with alerts to be triggered at both 15 minutes before and at the 
    time of the reminder. Reminders are displayed only once in the daily 
    schedule and without starting times(s).

    An event with one or more starting times and an extent is treated as 
    a <em>scheduled event</em>. Each starting time is displayed on the 
    daily schedule and the corresponding periods are displayed in the 
    busy panel.

    An <b>action</b> might be regarded as an event with an extent but 
    without a starting time but since actions are important for uses such 
    as time billing, they are treated as a separate type. When an action 
    is created by pressing either 'a' or '~', you can provide an initial 
    entry and then press Shift-Return. A timer will then be started 
    which can be paused and restarted by pressing 'a'. A reminder will 
    be sounded periodically to remind you when the timer is running. You 
    can press 'A' to stop the timer and make final changes to the action 
    entry. The total elapsed time will be entered automatically.

    A <b>note</b> is created by pressing either 'n' or '!. You could, 
    for example, use notes to keep a daily log or diary. As with other 
    items, new lines and other white space within a field are preserved 
    and can be used for rudimentary formatting.

    A <b>task</b> is created by pressing either 't', '-' or '+'. A task 
    without an "@d" entry is regarded as a <em>floating task</em> and 
    always appears to be due on the current date. A task with an "@d" 
    entry is treated as being due on the specified date. If unfinished, 
    it will appear on the daily schedule for the due date. If it becomes 
    past due then it will also appear on the daily schedule for the 
    current date with a reminder of the number of days it is past due. 
    Etm not only can nag you about past due tasks, it can also alert you 
    to an upcomming due date as well. E.g.

        - take out trash @d 6/1 @r w @w FR @b 1 @o s

    would remind you to put out the trash each Friday and, because of 
    the "@b 1" entry, you would be notified of this beginning one day 
    before the due date. The "@o s" in this entry, by the wqy, means 
    that when the task is (o)verdue, etm should (s)kip the nagging and 
    simply inform you of the next due date.

    When a task is completed, it will appear in the daily schedule for 
    the date it was finished with an indication that it has been 
    completed.

    Tasks in etm can have prerequisites. E.g.,

        - check sizes of existing air filters
        -- buy replacement filters
        --- install new air filters

    The outline structure gives the dependencies.

    Tasks with unfinished prerequisites are classified as <em>waiting</em>
    by etm.

    It is also possible for tasks to have multiple, independent 
    prerequisites. E.g.,

        - borrow power drill from Ed
        + pick up hardware 
        -- repair door

    Here the '+' adds 'borrow' to 'pick up' as a prerequisite for 
    'repair'. 

    Thanks to etm's use of <em>dateutil</em>, events and tasks can be repeated
    in powerful ways. E.g.,

        * payday @d 6/1 @r m @w (MO,TU,WE,TH,FR) @m (-3,-2,-1) @S -1

    would repeat on the last weekday in each month. 

    See the <em>Item Options</em> help tab for details. 

  <a name="classifiers" href="#toc"><b>Item Classifiers</b></a>

    A variety of fields are available in etm to classify items:

        <tt>@c CONTEXT</tt>
            For GTD (Getting Things Done) usage. Typical contexts 
            include 'home', 'office', 'phone', 'computer' and 'errands'. 
            If you are about to run errands, for example, you could use 
            the outline display option '<tt>-c errands</tt>' to show 
            only items with the context 'errands'.

        <tt>@k KEYWORD</tt>
            For time billing. One pattern of usage would be to use 
            keywords with the format 'client:project'. Then the outline 
            display option '<tt>-g (k1, (y,m), k2)</tt>' would group and 
            total times first by k1 (client), then by year and month 
            combined and finally by k2 (project).

        <tt>@l LOCATION</tt>
            This field is supported when exporting in vCal format.

        <tt>@p PRIORITY</tt>
            This field is supported when exporting in vCal format. Note 
            that etm's integer priorities may be converted by some 
            applications. iCal, for example, makes the following 
            conversions:
                etm     iCal
               -----   ------
                1-4     High
                 5     Medium
                6-9     Low

        <tt>@t TAGS</tt>
            This field is exported as 'categories' in vCal format. 
            Multiple tags can be used when separated by commas and 
            enclosed in parentheses.

        <tt>@U USER</tt>
            For multiuser application. This field could be used to 
            distinguish responsibility for billing times or task 
            completion. 

        Items can be grouped and/or filtered by any combination of these 
        classifiers and additionally by the name of the file and by the 
        title of the project (from the first line of the file). For 
        details of these and other options see the 'Display Options' 
        help tab.

  <a name="fuzzy" href="#toc"><b>Dates, Times and Alerts</b></a>

      <b>Fuzzy Date Parsing</b>
         Suppose that it is currently Tuesday, February 17, 2009. Then 
         in fields calling for a date:

          entered           after fuzzy parsing
         ---------         ---------------------
            tue                 2009-02-17
            thu                 2009-02-19
            mon                 2009-02-23
            23                  2009-02-23
            mar 2               2009-03-02
            3/2                 2009-03-02
            +0                  2009-02-17 (today)
            +1                  2009-02-18 (tomorrow)
            +45                 2009-04-03 (45 days from now)
            -1                  2009-02-16 (yesterday)
            -90                 2008-11-19 (90 days ago)

      <b>Fuzzy Time Parsing</b>
         Similarly, in fields calling for a time, entering '6p' would 
         give after parsing '06:00PM' and so would '18:00'. 

         A further option is available when entering the extent, @e, for 
         an event. The extent can be specified as the duration or extent 
         of the event by preceeding the entry with a plus sign or, for 
         events with a single starting time, by the ending time of the 
         event. For example, all of the following are equivalent:

            @s 9a @e 10:30a 

            @s 9a @e +90m

            @s 9a @e +90

            @s 9a @e +1:30

            @s 9a @e +1h 30m

         However entered, etm would record the extent as '@e +1:30'.

      <b>Alerts</b>
        Alerts for an event or reminder are set by adding an '@a' field 
        to the event, e.g., 

            * phone conference with CR @d tue @s 2p @a (15, 5)

        would set alerts for 15 minutes before the event and 5 minutes 
        before the event. What happens when an alert is triggered is 
        determined by the setting for 'alertcmd' in '~/.etm/rc'. With 
        the default setting

            alertcmd = ''

        a message would be displayed with the current time and the text 
        of the message.

        Alternatively, on a mac (OS X leopard), you could use

            alertcmd = '''say -v "Alex" "%%(t)s. %%(m)s."'''

        to announce the message using the voice 'Alex' or

            alertcmd = '''say -v "Alex" "%%(t)s. %%(m)s."; 
                  growlnotify -t %%(T)s -m "%%(m)s"'''

        (all one one line) to have the message both spoken and displayed 
        using growl. On linux systems with 'notify-send' (provided by 
        libnotify-bin), a warning sound followed by a 5 second popup 
        display of the alert message could be produced with

            alertcmd = '''aplay /usr/share/sounds/purple/receive.wav 2> 
                   /dev/null; notify-send "%%(T)s" "%%(m)s" -t 5000'''

        With -t 0 the display would remain until you dismiss it. 

        Alternatively, you could receive an email reminder with

            alertcmd = '''echo "%%(m)s" | mail -s "etm alert at %%(T)s" 
                   your@email.address"'''

        if either sendmail or postfix is enabled on your system.

        More generally, any command that will accept a string argument 
        could be used.

      <b>Anniversary Substitutions</b>
         When items are processed whose titles contain a string of the 
         form !YYYY!, the string will be replaced with N[st|nd|th] 
         where N is the number of years from YYYY until the current 
         year. For example, on August 23, 2010, the event

            * Will's !1985! birthday @r y @M 8 @m 23 

         would be displayed as

            Will's 25th birthday

      <b>Limitations</b>
         In displays when grouping by date, etm will only display an item 
         or trigger an alert on the date of its starting time. This 
         means, for example, that the event

             * event @d 2010-12-01 @s 23:00 @e +2h

         would only display on December 1 even though the event lasts 
         until 1:00 on December 2. Similarly

             * event @d 2010-12-01 @s 1:00 @a (90, 30)

         would trigger an alert at 0:30 on 12/1 but would not trigger an 
         alert at 23:30 on 11/30.

  <a name="zone" href="#toc"><b>Time Zones</b></a>

    Dates and times in etm either have time zone information associated 
    with them or are treated as 'naive' date times. The treatment of 
    date times depends upon whether or not the item includes an @z ZONE 
    entry and, if it does, the value of ZONE. There are two 
    possibilities for lines in the etm data files:

    1) The item (i) does not have an @z ZONE entry or (ii) the item does 
       have an @z ZONE entry but the value of ZONE is 'none'. Item date 
       times are interpreted as naive and are displayed unchanged. 
       E.g., the events:

            * event @d 2010-11-28 @s 17:00

            * event @d 2010-11-28 @s 17:00 @z none

       would both be interpreted as starting at 17:00 on Nov 28 no 
       matter what the time zone of the user's system. Some calendar 
       applications refer to these as <em>floating events</em>. 

    2) The item has an @z ZONE entry and the value of ZONE is not 'none'.
       Item date times are interpreted as being in ZONE time and would 
       be displayed as local date times for the user's system time zone. 
       E.g., the event:

            * event @d 2010-12-01 @s 12:00 @z Pacific/Auckland 

       would be interpreted as starting at 12:00 December 1 
       Pacific/Auckland time and, if the users time zone were 
       US/Eastern, would be displayed as starting at 18:00 on November 
       30.

    When creating new items in etm, time zone information that you 
    enter, provided that it is valid, will be left unchanged. This 
    includes '@z none'. What happens when you create an item without an 
    '@z' entry depends upon the value of 'auto_set_timezone' in your rc 
    file. If 'True' (the default), then '@z ZONE' will automatically be 
    appended to your entry with ZONE equal to your local time zone. If 
    'False', then your entry will be saved as it is and its dates and 
    times will subsequently be interpreted as naive. 

    When entering zone information in etm, completion is available. 
    Suppose, for example, that you have entered

        * event @d 12/1 @s 12 @z US

    and press Ctrl-Space with the cursor after 'US'. Then etm would 
    display a pop-up list of possible completions from 'timezones' in 
    your rc file for your selection. With the default value of 
    'timezones', this list would include 'US/Alaska', 'US/Central', 
    'US/Eastern' and the others that begin with 'US'.

    Zone information in etm, other than 'none', automatically adjusts 
    for <em>daylight saving</em>. Consider, for example, the following two 
    events.

        * event @d 2010-08-01 @s 17:00 @z US/Eastern

        * event @d 2010-12-01 @s 17:00 @z US/Eastern

    While both would display on the given dates as starting at 17:00, 
    this would be EDT for the first and EST for the second. This means 
    that when you change the date for a task with zone information, you 
    do not need to concern yourself with making adjustments for daylight 
    saving.

    The values of 'due_hour' (0-23) and 'due_minute' (0-59) from your rc 
    file set the implicit times for tasks and other items without 
    starting times. These values have no effect unless '@z ZONE' is set 
    and ZONE is not equal to 'none'. The default, 'due_hour = 12' and 
    'due_minute = 0', corresponds to most expectations. E.g, a task that 
    is due at noon on November 30 in New York (12:00 11/30 EST), would 
    also be due on November 30 in London (17:00 11/30 GMT) and Honolulu 
    (7:00 11/30 HST), but would be due on December 1 on the other side 
    of the international date line in Tokyo (2:00 12/1 JST) and 
    Singapore (1:00 12/1 SGT).

  <a name="grouping" href="#toc"><b>Grouping and Sorting</b></a>

    Grouping and sorting is controlled by the '-g' outline display 
    option. The default

        -g ((y,m,d),)

    groups and sorts items by y)ear, m)onth and d)ay combined to 
    give output like

        Fri Jun 3 2011 (17) 3.5h
            3:00p Conference
            ...   16 other items for June 3
        Sat Jun 4 2011 (3) 1.0h
            ...   3 items for June 4

    As another example, suppose keywords have the format 'client 
    name:project name'. Then 

        -g (k1, (y,q,m), k2)

    would group and sort by k1 (client name), then y)ear, q)uarter 
    and m)onth combined and finally k2 (project name) to give output 
    such as:

        client 1 (3) 1.5h
            2nd quarter 2011 Jun (3) 1.5h
                project E (3) 1.5h
                    ...  3 items for 'client 1:project E'
        ...

    or

        -g (k1, (y,m,d), T) -d 0 -T 1

    would group by k1 (client), then y)ear, m)onth and d)ay combined and 
    finally T (item title). With details omitted, '-d 0' and totals 
    first, '-T 1', the output would be similar to

        3.6h) client 1
            2.0h) Fri Aug 5 2011 (2)
                1.0h) prepare order (1)
                0.5h) review invoice (1)
            ...
        ...

    Note in the examples above that the ledger information '(number 
    of included items) total time' is automatically included with each 
    node.

    Available group/sort keys include y (year), m (month), d (day), e 
    (extent), w (week number in year),  q (quarter number),  c 
    (context), l (location), u (user), s (item type), T (title) and k1, 
    k2, k3, (first, second and all remaining keywords). E.g. with '@k 
    A:B:C:D:E', k1 would be 'A', k2 would be 'B' and k3 would be 
    'C:D:E'.

    A final groupby option is F (project file). This option shows <em>all</em>
    items outlined by project F)ile ignoring other GROUPBY and BEGIN and 
    END date options.

    Examples:

        * show all your notes outlined by project file:

            -g F -o !n  

        * show all your unfinished tasks organized by project file:

            -g F -o !ut 


  <a name="regex" href="#toc"><b>Regular Expression Filters</b></a>

    Etm supports the use of case-insensitive, regular expression 
    matching when searching and when filtering by context, keyword,  
    location, priority, tag, user, file and/or project. E.g., an option 
    setting which included '-c errands' would limit the display to items 
    which contain 'errands' in the context field. 

    Alternatively, '-c !errands' would limit the display to items which 
    do not have contexts matching 'errands'. Note: when using the 
    command line the latter expression should be wrapped in single 
    quotes, e.g.,

        e.py o -c '!errands'

    An excellent introduction to the use of regular expressions is 
    provided by 

        www.duke.edu/~dgraham/ETM/LearningtoUseRegularExpressions.html

    Skip down to 'Matching Patterns in Text: The Basics' and note that 
    the surrounding '/' delimiters are not needed in etm.

<a name="key" href="#toc"><b>Key Bindings</b></a>

  <a name="commands" href="#toc"><b>General</b></a>
     space:  Group by year, month and day combined with nothing omitted 
             and with the current date selected.
    escape:  cancel the current operation and/or clear the current 
             selection and details panel. 
        F1:  Open the etm help display. This page, <em>ETM Overview</em> 
             is the first tab.
        F2:  Show etm 'about' information.
        F3:  Check for the availability of a newer etm version.
        F4:  Display a twelve month calendar.
        F5:  Open the date calculator.
        F6:  Show local weather information.
        F7:  Show local sun and moon information.
         .:  (period key) Change focus to the monthly calendar. 
         ,:  (comma key) Change focus to the outline panel. 
         a:  begin a new action or toggle the timer for an active action.
         A:  If an action timer is running or paused, prompt for changes 
             and record the action. Otherwise create a new action without 
             using the timer. 
         b:  set the options for a busy report.
         c:  begin a new item using a clone of the currently selected 
             item.
         d:  delete the currently selected task.
         e:  create a new event.
         f:  mark the currently selected task finished.
         g:  if the currently selected item contains a @g (goto) link, 
             then open it using the system default application.
         j:  set and jump to a (fuzzy parsed) date. 
         m:  move the currently selected item to a different project 
             file.
         n:  create a new note.
         o:  set the options for the outline display.
         p:  open an existing project using the editor specified in the 
             etm rc file.
         P:  create a new project.
         q:  Show the status of the background alert warnings queue for
             the current date.
         t:  create a new task.
         u:  un-finish the currently selected task.
    Ctrl-Q:  quit (close) etm.
    Ctrl-S:  begin a case insensitive, regular expression search for 
             items, regardless of date, which have matching field values.
       0-9:  set outline or busy display options using items 0 through 9 
             from <em>display_shortcuts</em> in etmrc.

  <a name="navigation" href="#toc"><b>Outline Panel</b></a>

     Home: Select the first item in the outline.
      End: Select the last item in the outline.
  Page Up: Scroll the outline up by one page.
Page Down: Scroll the outline down by one page.
       Up: (arrow cursor key)
           <em>No modifiers:</em> Select previous item.
           <em>Shift:</em> Select previous sibling.
           <em>Control:</em> Select first sibling.
           <em>Shift and Control:</em> Select first item.
     Down: (arrow cursor key)
           <em>No modifiers:</em> Select next item.
           <em>Shift:</em> Select next sibling.
           <em>Control:</em> Select last sibling.
           <em>Shift and Control:</em> Select last item.
     Left: (arrow cursor key)
           <em>No modifiers:</em> Select parent.
           <em>Shift:</em> Collapse current item and then select parent.
           <em>Control:</em> Select parent and then collapse it.
           <em>Shift and Control:</em> Collapse all branches recursively and
           then select ancestor.
    Right: (arrow cursor key)
           <em>No modifiers:</em> Select first child.
           <em>Shift:</em> Select first child and then expand it recursively.
           <em>Control:</em> Expand current item recursively and then select
           first child. 
           <em>Shift and Control:</em> Expand all branches recursively and
           then select first child.
   Ctrl-P: print
           Open a preview of the selection for printing. If nothing is 
           selected, then preview the entire outline. (Pressing Escape 
           in the outline panel clears the selection.) If a branch is 
           selected, then preview the branch. In both cases omit 
           portions of the outline which are collapsed. Finally, if a 
           leaf is selected then preview both the title and the details 
           of the item.
   Ctrl-V: vCal to clipboard
           Export the selection in vCal format to the system clipboard. 
           If nothing is selected, then export the entire outline. If a 
           branch is selected, then export the items in the branch. In 
           both cases omit portions of the outline which are collapsed. 
           Finally, if a leaf is selected then export only the single 
           item. 
   Ctrl-F: vCal to file
           Export the selection in vCal format to the file 'export.ics' 
           in the directory specified by 'export' in the etm rc file. If 
           nothing is selected, then export the entire outline. If a 
           branch is selected, then export the items in the branch. In 
           both cases omit portions of the outline which are collapsed. 
           Finally, if a leaf is selected then export only the single 
           item. 
    Ctrl-Y: yank to clipboard
           Copy the selection to the system clipboard. If nothing is 
           selected, then copy the entire outline. If a branch is 
           selected, then copy the items in the branch. In both cases 
           omit portions of the outline which are collapsed. Finally, if 
           a leaf is selected, then copy both the title and details of 
           the item.

   Return: (with an item selected or double clicking an item) Open the 
           item for editing in the details panel. 

  <a name="details" href="#toc"><b>Details Panel</b></a>

    Shift-Return: submit entry for processing.
          Escape: cancel entry (with confirmation).
        Ctrl-Tab: choose completion of current display or item entry 
                  from a selection list of matching items.
      Ctrl-Space: same as Ctrl-Tab.

    Also, built in keybindings for the stylized text control (wx.stc) 
    from http://www.yellowbrain.com/stc/keymap.html:
</pre>
<center>
<table width="90%%">
<tr><td>DOWN ARROW</td><td>Move caret down one line</td></tr>
<tr><td>SHIFT DOWN ARROW</td><td>Move caret down one line extending selection to new caret position</td></tr>

<tr><td>CTRL DOWN ARROW</td><td>Scroll the document down, keeping the caret visible</td></tr>
<tr><td>UP ARROW</td><td>Move caret up one line</td></tr>
<tr><td>SHIFT UP ARROW </td><td>Move caret up one line extending selection to new caret position</td></tr>
<tr><td>CTRL UP ARROW</td><td>Scroll the document up, keeping the caret visible</td></tr>

<tr><td>LEFT ARROW</td><td>Move caret left one character</td></tr>
<tr><td>SHIFT LEFT ARROW</td><td>Move caret left one character extending selection to new caret position</td></tr>
<tr><td>CTRL LEFT ARROW</td><td>Move caret left one word</td></tr>
<tr><td>CTRL+SHIFT LEFT ARROW</td><td>Move caret left one word extending selection to new caret position</td></tr>
<tr><td>ALT LEFT ARROW</td><td>Move to the previous change in capitalisation</td></tr>

<tr><td>ALT+SHIFT LEFT ARROW</td><td>Move to the previous change in capitalisation extending selection
to new caret position</td></tr>
<tr><td>RIGHT ARROW</td><td>Move caret right one character</td></tr>
<tr><td>SHIFT RIGHT ARROW</td><td>Move caret right one character extending selection to new caret position</td></tr>
<tr><td>CTRL RIGHT ARROW</td><td>Move caret right one word</td></tr>
<tr><td>CTRL+SHIFT RIGHT ARROW</td><td>Move caret right one word extending selection to new caret position</td></tr>

<tr><td>ALT RIGHT ARROW</td><td>Move to the next change in capitalisation</td></tr>
<tr><td>ALT+SHIFT RIGHT ARROW</td><td>Move to the next change in capitalisation extending selection
to new caret position.</td></tr>
<tr><td>HOME</td><td>Move caret to before first visible character on line.
If already there move to first character on line</td></tr>
<tr><td>SHIFT HOME</td><td>Like Home but extending selection to new caret position</td></tr>
<tr><td>CTRL HOME</td><td>Move caret to first position in document</td></tr>

<tr><td>CTRL+SHIFT HOME</td><td>Move caret to first position in document extending selection to new caret position</td></tr>
<tr><td>ALT HOME</td><td>Move caret to first position on display line</td></tr>
<tr><td>ALT+SHIFT HOME</td><td>Move caret to first position on display line extending selection to 
new caret position.</td></tr>
<tr><td>END</td><td>Move caret to last position on line</td></tr>
<tr><td>SHIFT END</td><td>Move caret to last position on line extending selection to new caret position</td></tr>

<tr><td>CTRL END</td><td>Move caret to last position in document</td></tr>
<tr><td>CTRL+SHIFT END</td><td>Move caret to last position in document extending selection to new caret position</td></tr>
<tr><td>ALT END</td><td>Move caret to last position on display line</td></tr>
<tr><td>ALT+SHIFT END</td><td>Move caret to last position on display line extending selection to new 
caret position</td></tr>
<tr><td>PRIOR</td><td>Move caret one page up</td></tr>

<tr><td>SHIFT PRIOR</td><td>Move caret one page up extending selection to new caret position</td></tr>
<tr><td>NEXT</td><td>Move caret one page down</td></tr>
<tr><td>SHIFT NEXT</td><td>Move caret one page down extending selection to new caret position</td></tr>
<tr><td>DELETE</td><td>Delete all text in the document</td></tr>
<tr><td>SHIFT DELETE</td><td>Cut the selection to the clipboard</td></tr>

<tr><td>CTRL DELETE</td><td>Delete the word to the right of the caret</td></tr>
<tr><td>CTRL+SHIFT DELETE</td><td>Delete forwards from the current position to the end of the line</td></tr>
<tr><td>INSERT</td><td>Switch from insert to overtype mode or the reverse</td></tr>
<tr><td>SHIFT INSERT</td><td>Paste the contents of the clipboard into the document replacing the selection</td></tr>
<tr><td>CTRL INSERT</td><td>Copy the selection to the clipboard</td></tr>

<tr><td>BACK</td><td>Delete the selection or if no selection, the character before the caret</td></tr>
<tr><td>SHIFT BACK</td><td>Delete the selection or if no selection, the character before the caret</td></tr>
<tr><td>CTRL BACK</td><td>Delete the word to the left of the caret</td></tr>
<tr><td>ALT BACK</td><td>Undo one action in the undo history</td></tr>

<tr><td>CTRL+SHIFT BACK</td><td>Delete back from the current position to the start of the line</td></tr>
<tr><td>CTRL 'Z'</td><td>Undo one action in the undo history</td></tr>
<tr><td>CTRL 'Y'</td><td>Redoes the next action on the undo history</td></tr>
<tr><td>CTRL 'X'</td><td>Cut the selection to the clipboard</td></tr>
<tr><td>CTRL 'C'</td><td>Copy the selection to the clipboard</td></tr>

<tr><td>CTRL 'V'</td><td>Paste the contents of the clipboard into the document replacing the selection</td></tr>
<tr><td>CTRL 'A'</td><td>Select all the text in the document</td></tr>
<tr><td>TAB</td><td>If selection is empty or all on one line replace the selection with a tab character.
 If more than one line selected, indent the lines.</td></tr>
<tr><td>SHIFT TAB</td><td>Dedent the selected lines</td></tr>
<tr><td>RETURN</td><td>Insert a new line, may use a CRLF, CR or LF depending on EOL mode</td></tr>

<tr><td>CTRL ADD</td><td>Magnify the displayed text by increasing the sizes by 1 point</td></tr>
<tr><td>CTRL SUBTRACT</td><td>Make the displayed text smaller by decreasing the sizes by 1 point</td></tr>
<tr><td>CTRL DIVIDE</td><td>Set the zoom level to 0. This returns the zoom to 'normal,' i.e., no zoom. </td></tr>
<tr><td>CTRL 'L'</td><td>Cut the line containing the caret</td></tr>

<tr><td>CTRL+SHIFT 'L'</td><td>Delete the line containing the caret</td></tr>
<tr><td>CTRL 'T'</td><td>Switch the current line with the previous</td></tr>
<tr><td>CTRL 'U'</td><td>Transform the selection to lower case</td></tr>
<tr><td>CTRL+SHIFT 'U'</td><td>Transform the selection to upper case</td></tr>
</table>
</center>
<pre>
  <a name="html" href="#toc"><b>HTML Windows</b></a>

    HTML windows include the help tabs, the 12-month calendar, busy 
    view and detail reports (generated with the outline '-d' option).

    Ctrl-Y: yank to clipboard
           Copy the selection to the system clipboard. 

<a name="files" href="#toc"><b>Files</b></a>

  <a name="config" href="#toc"><b>Configuration</b></a>

    All configuration settings are kept in <tt>~/.etm/rc</tt>. This file 
    will automatically be created if it doesn't already exist and 
    populated with default values. It is self-documented and can be 
    freely edited. If you make changes you don't like you can simply 
    erase the offending part, or even the entire file, and it will be 
    recreated with defaults the next time you run etm.

    If the current working directory contains a file named <tt>rc</tt> 
    when  etm is run, then all configuration settings will be taken from 
    that file instead of <tt>~/.etm/rc</tt>. 

  <a name="rotate" href="#toc"><b>Monthly</b></a>
    Many times it will be convenient to create a project file to hold a 
    collection of related items. Often, however, items will be created 
    that are independent of one another. There is no need to create 
    separate project files in such circumstances since etm automatically 
    creates files that can you can use for such independent items. 

    Suppose that it is currently July 2011. Then *etm* will 
    automatically create '2011/07_actns.text' in the path for 
    'etmActions' the first time it is run. By default, actions created 
    during July will be written to this file. Similar files will be 
    automatically be created and become the defaults as subsequent 
    months and years arrive.  A similar thing happens for events, notes 
    and tasks.
</pre>
"""

view_help = """
Press <em>&lt;Esc&gt; to cancel entry or &lt;Shift-Return&gt; to accept entry.<br>
Press <em>&lt;Tab&gt;  to complete option string from the current history list.<br>
Press <em>&lt;Shift-Space&gt; after entering -c [word] or -k [word] to complete [word] from a matching list of previously used contexts or keywords.</em>
"""

view_html = """
<title>etm</title>
<body text="%(colorMainFG)s" bgcolor="%(colorMainBG)s">
<center><font size="+1"><b>Options for Outline and Busy 
Views</b></font></center>
<p>
Press  'o' to set options for outline view or 'b' to set options for 
busy view. Allowed options are indicated below by 'O'. 
</p>
<hr width="100%%"/>
<table>
<tr><th>&nbsp;</th><th>&nbsp;</th><th>Outline</th><th>Busy</th><th>Description</th></tr>
<tr>
    <td valign="top">-b</td>
    <td valign="top">BEGIN</td>
    <td align="center" valign="top">O</td>
    <td align="center" valign="top">O</td>
    <td valign="top">Date. Display items on or after this (fuzzy parsed) 
    date. A relative date can be entered, e.g., <tt>'-b "-14"'</tt> 
    would set BEGIN to 14 days before the current date. Note that 
    "-14"<D-d> must be enclosed in quotes to prevent the minus sign from being interpreted as a field identifier.</td>
</tr>
<tr>
<td valign="top">-e</td>
<td valign="top">END</td>
<td align="center" valign="top">O</td>
<td align="center" valign="top">O</td> 
<td valign="top">Date or a plus sign followed by an integer. Display 
items before (but not on) this (fuzzy parsed) date or for this integer 
number of days starting with BEGIN. E.g., both '-b 7/1 -e 7/8' and '-b 
7/1 -e +7' would include items for the seven days from July 1 through 
July 7.</td>
</tr>
<tr>
<td valign="top">-g</td><td valign="top">GROUPBY</td><td align="center" valign="top">O</td>
<td align="center" valign="top">&nbsp;</td>

<td valign="top">Available group/sort keys include y (year), m (month), 
d (day), e (extent), w (week number in year),  q (quarter number),  c 
(context), l (location), u (user), s (item type), k1, k2 and k3, (first, 
second and all remaining  keywords) and T (title). Alternatively, F 
(file path) can be used to show <em>all</em> items organized by file 
path ignoring other GROUPBY and BEGIN and END date options. See the discussion under 'Grouping and Sorting' in the  ETM 
Overview tab for further details.</td>
</tr>

<tr>
<td valign="top">-o</td><td valign="top">OMIT</td>
<td align="center" valign="top">O</td>
<td align="center" valign="top">O</td>
<td valign="top">
String with characters from a (actions), b (task begin dates), e 
(events), n (notes), r (reminders), f (finished tasks),  u (undated 
tasks), w (waiting tasks), p (past due tasks) and t (other tasks),  If 
OMIT begins with '!', then only show items with types belonging to OMIT. 
Otherwise only show items with types not belonging to OMIT. Only items 
with extents (actions and events) are relevant to busy view.
</td>
</tr>

<tr>
<td valign="top">-d</td>
<td valign="top">DETAILS</td>
<td align="center" valign="top">O</td>
<td align="center" valign="top">&nbsp;</td>
<td valign="top">0, 1 or a string composed of item field keys or *. Display only group 
headings if 0. If 1 show items (leaves) as well. If a string of field 
keys, then open a print preview showing details corresponding to the 
keys as well as the items. E.g., '-d ln' would open a print preview 
showing items, their l)ocations and their n)otes. If '*' then show 
details corresponding to all field keys. Default: 1.</td>
</tr>

<tr>
<td valign="top">-T</td>
<td valign="top">TOTALSFIRST</td>
<td align="center" valign="top">O</td>
<td align="center" valign="top">&nbsp;</td>
<td valign="top">0 or 1. Display time totals before the group titles if 
1. Otherwise after the group titles. Default: 0.</td>
</tr>

<tr>
<td valign="top">-s</td><td valign="top">SEARCH</td>
<td align="center" valign="top">O</td>
<td align="center" valign="top">O</td>
<td valign="top">
Regular expression. include items, regardless of date, in which
<em>any</em> field value matches SEARCH (ignoring case). Prepend an 
exclamation mark to include items which do not have any field values 
matching SEARCH.
</td>
</tr>

<tr>
<td valign="top">-c</td><td valign="top">CONTEXT</td>
<td align="center" valign="top">O</td>
<td align="center" valign="top">O</td>
<td valign="top">
Regular expression. include items with contexts matching CONTEXT 
(ignoring case). Prepend an exclamation mark to include items which do 
not have contexts matching CONTEXT.
</td>
</tr>

<tr>
<td valign="top">-k</td><td valign="top">KEYWORD</td>
<td align="center" valign="top">O</td>
<td align="center" valign="top">O</td>
<td valign="top">
Regular expression. include items with keywords matching KEYWORD 
(ignoring case). Prepend an exclamation mark to include items which do 
not have keywords matching KEYWORD.
</td>
</tr>

<tr>
<td valign="top">-p</td><td valign="top">PRIORITY</td>
<td align="center" valign="top">O</td>
<td align="center" valign="top">O</td>
<td valign="top">
Regular expression. Include items with priorities matching PRIORITY.
For example, '-p [123]' would select items with the top three 
priorities and '-p ![123]' would select items with priorities lower than 
3 or no priority.
</td>
</tr>

<tr>
<td valign="top">-t</td><td valign="top">TAG</td>
<td align="center" valign="top">O</td>
<td align="center" valign="top">O</td>
<td valign="top">
Regular expression. include items with tags matching TAG (ignoring 
case). Prepend an exclamation mark to include items which do not have 
tags matching TAG. Multiple uses of this option are possible. E.g., use 
<tt>'-t tag 1 -t tag 2'</tt> to show items with both 'tag 1' and 'tag 
2'.
</td>
</tr>

<tr>
<td valign="top">-l</td><td valign="top">LOCATION</td>
<td align="center" valign="top">O</td>
<td align="center" valign="top">O</td>
<td valign="top">
Regular expression. include items with locations matching LOCATION 
(ignoring case). Prepend an exclamation mark to include items which do 
not have locations matching LOCATION.
</td>
</tr>

<tr>
<td valign="top">-u</td><td valign="top">USER</td>
<td align="center" valign="top">O</td>
<td align="center" valign="top">O</td>
<td valign="top">
Regular expression. include items with users matching USER 
(ignoring case). Prepend an exclamation mark to include items which do 
not have users matching USER.
</td>
</tr>

<tr>
<td valign="top">-P</td><td valign="top">PROJECT</td>
<td align="center" valign="top">O</td>
<td align="center" valign="top">O</td>
<td valign="top">
Regular expression. include items with projects matching PROJECT 
(ignoring case). Prepend an exclamation mark to include items which do 
not have projects matching PROJECT.
</td>
</tr>

<tr>
<td valign="top">-f</td><td valign="top">FILE</td>
<td align="center" valign="top">O</td>
<td align="center" valign="top">O</td>
<td valign="top">
Regular expression. include items with files matching FILE 
(ignoring case). Prepend an exclamation mark to include items which do 
not have projects matching FILE.
</td>
</tr>

<tr>
<td valign="top">-i</td><td valign="top">INCLUDE</td>
<td align="center" valign="top">&nbsp;</td>
<td align="center" valign="top">O</td>
<td valign="top">
String. One or more characters from B (busy time bars), b (busy times), 
F (free time bars), f (free times), and c (conflict times). Default: Bfc
</td>
</tr>

<tr>
<td valign="top">-O</td><td valign="top">OPENING</td>
<td align="center" valign="top">&nbsp;</td>
<td align="center" valign="top">O</td>
<td valign="top">
Time. The opening or earliest time (fuzzy parsed) to be considered when 
displaying free periods. Default: 8:00a.
</td>
</tr>

<tr>
<td valign="top">-C</td><td valign="top">CLOSING</td>
<td align="center" valign="top">&nbsp;</td>
<td align="center" valign="top">O</td>
<td valign="top">
Time. The closing or latest time (fuzzy parsed) to be considered when 
displaying free periods. Default: 10:00p.
</td>
</tr>

<tr>
<td valign="top">-m</td><td valign="top">MINIMUM</td>
<td align="center" valign="top">&nbsp;</td>
<td align="center" valign="top">O</td>
<td valign="top">
Positive integer. The minimum length in minutes for a free 
period to be displayed. Default: 60.
</td>
</tr>

<tr>
<td valign="top">-w</td><td valign="top">WRAP</td>
<td align="center" valign="top">&nbsp;</td>
<td align="center" valign="top">O</td>
<td valign="top">
Positive integer. Provide a buffer of WRAP minutes before and after busy  
periods when computing free periods. Default: 15.
</td>
</tr>

<tr>
<td valign="top">-v</td><td valign="top">VCAL</td>
<td align="center" valign="top">O</td>
<td align="center" valign="top">&nbsp;</td>
<td valign="top">
Export displayed items in vCal format to <tt>export.ics</tt> in the 
'export' directory specified in the etm rc file.
</td>
</tr>

<tr>
<td valign="top">-x</td><td valign="top">VALUES</td>
<td align="center" valign="top">O</td>
<td align="center" valign="top">&nbsp;</td>
<td valign="top">
Comma separated list of field keys. Export displayed items in CSV (comma
separted values) format to <tt>export.csv</tt> in the 'export' directory 
specified in the etm rc file. Values exported for each item include 1) 
item id, 2) item type number, 3) item description (title) and, in order, 
values corresponding to the keys in VALUES. Possible keys include 
y (year), m (month), d (day), e (extent minutes), p (priority), w (week 
number),  q (quarter number),  c (context),  k1, k2, k3, (keywords 1, 2 
and beyond), l (location), u (user), P (project name), t (tags) and n 
(note). 
</td>
</tr>

</table>
</body>
"""
info = {
        'copyright' : copyright,
        'etminfo' : etminfo(),
        'sysinfo' : sysinfo(),
        'extent' : 30,
        'etmrc' : etmrc,
        'version' : version,
        'editor' : editor,
        'export' : export,
        'etmdir' : etmdir,
        'etmdata' : etmdata,
        'colorMainFG' : colorMainFG,
        'colorMainBG' : colorMainBG,
        'view_help' : view_help,
        'wrap' : wrap,
        'minimum' : minimum,
        'opening' : opening,
        'closing' : closing,
        'slotsize' : slotsize,
        'entry_help' : entry_help,
        }

htmlPage = {
        'Help'   : helpview_html % info,
        'View'   : view_html % info,
        'Add'    : add_html % info,
        'Examples' : examples_html % info,
        }

# vim: tw=72