#! python

"""execsql.py

PURPOSE
=======
Read a sequence of SQL commands from a file and execute them against
a specified database (Access only, for now).  The reason for this script
is to allow execution of a SQL script--i.e., a sequence of SQL statements--because:
	*	Access only allows a single SQL command to be executed from the query window;
	*	The Jet engine can handle complex SQL statements that Access cannot, and 
		that therefore cannot be saved as queries in Access, so executing them
		directly from a text file is more convenient than copying and pasting them
		into the query design window.
The output of the last SQL statement in the script can be saved to a disk file
or directed to a new Excel workbook in memory.

AUTHOR
======
Dreas Nielsen (RDN)

COPYRIGHT AND LICENSE
=====================
Copyright (c) 2007, 2008, R.Dreas Nielsen
This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
The GNU General Public License is available at <http://www.gnu.org/licenses/>

NOTES
=====
1. Because of its interaction with Access and Excel, this is a Windows-only utility.
2. Mark Hammond's win32com package is required for interaction with MS applications.
3. The Access database must not have security enabled.
4. Redirecting output to an Excel worksheet in memory is slow.

HISTORY
=======
-------		--------------------------------------------------------------------
 Date		 Comments
-------		--------------------------------------------------------------------
11/11/2007	Created.  RDN
1/1/2008	Modified to allow writing of the output of the last SQL
			command to a CSV file.  RDN.
4/20/2008	Converted to use DAO to interact with Access
			instead of the dbconnect library.  RDN.
4/22/2008	Added regular expression to match 'create temp view...' preface.  RDN.
4/26/2008	Added creation and deletion of temporary views (queries) and
			export of final query to Excel.  RDN.
5/20/2008	Added unicode encoding for code page 1252 for data read from Access
			and written to CSV.  RDN.
12/18/2008	Added interpreter name to first line for distribution.  RDN.
12/19/2008	Added docstrings to two functions.  RDN.
==========================================================================
"""

#================================================================================
# TODO:
#	1. Add options to specify a database user name and password if appropriate.
#	2. Add a command language embedded in SQL comments, specifically to allow
#		commands to export queries other than the last.
#	3. Add loading of VB modules from a text file into an Access database
#		prior to executing the SQL.
#================================================================================

import sys
import csv
import re

import win32com.client

_version = "0.3.0.0"
_vdate = "2008-05-20"

__HELP_MSG = """Copyright (c) 2007-2008, R.Dreas Nielsen
Licensed under the GNU General Public License version 3.
Usage:
   execsql.py <sqlfile> <Access_db> [output_spec]
Arguments:
   <sqlfile>
      The name of a file of SQL commands to be executed.  Required argument.
   <Access_db>
      The name of the Access database against which to run the SQL.  Required argument.
   [output_spec]
      The name of a CSV file to which the output of the last SQL command should
      be written, or the literal string "Excel" to send output to a Microsoft Excel
      workbook in memory.  Note that Excel has row and column limitations, CSV does not.
      Optional argument."""


def execsql( sqllist, db, outfile=None ):
	"""Execute all SQL commands in the given list against the given Access database.
	Arguments:
	    'sqllist' is a list of SQL statements, to be executed in order.
	    'db' is an DAO database object, which must represent an existing database.
	    'outfile' is the name of a CSV file in which to dump the results of the last command.
	"""
	temp_rx = re.compile(r'^\s*create(?:\s+or\s+replace)?(\s+temp(?:orary)?)?\s+(?:(view|query))\s+(\w+) as\s+', re.I)
	tempquerynames = []
	for i in range(len(sqllist)):
		sqlcmd = sqllist[i]
		tqd = temp_rx.match(sqlcmd)
		if tqd:
			qn = tqd.group(3)
			qsql = sqlcmd[tqd.end():]
			try:
				db.QueryDefs.Delete(qn)
			except:
				pass
			db.CreateQueryDef(qn, qsql)
			if tqd.group(1) and tqd.group(1).strip().lower()[:4] == 'temp':
				tempquerynames.append(qn)
		else:
			if outfile and i == len(sqllist)-1:
				rs = db.OpenRecordset(sqlcmd)
				hdrs = [f.Name for f in rs.Fields]
				if outfile == "Excel":
					xl = win32com.client.Dispatch('Excel.Application')
					xl.Workbooks.Add()
					row = 1
					col = 1
					for hdr in hdrs:
						xl.ActiveSheet.Cells(row, col).Value = hdr
						col += 1
					while not rs.EOF:
						row += 1
						rec = rs.GetRows(1)
						vals = [fldval[0] for fldval in rec]
						col = 1
						for val in vals:
							xl.ActiveSheet.Cells(row, col).Value = val
							col += 1
					xl.Visible = True
				else:
					ofile = csv.writer(open(outfile, "wb"), quoting=csv.QUOTE_NONNUMERIC)
					ofile.writerow([h.encode("cp1252") for h in hdrs])
					while not rs.EOF:
						rec = rs.GetRows(1)
						ofile.writerow([(fldval[0].encode("cp1252") if isinstance(fldval[0], basestring) else fldval[0]) for fldval in rec])
					ofile = None
				rs.Close()
			else:
				db.Execute(sqlcmd)
	for qn in tempquerynames:
		db.QueryDefs.Delete(qn)


def read_sqlfile( fn ):
	"""Open the text file with the specified name, read it, and return a list of
	the SQL statements it contains.
	"""
	# Currently (11/11/2007) this routine knows only two things about SQL:
	#	1. Lines that start with "--" are comments.
	#	2. Lines that end with ";" terminate a SQL statement.
	sqlfile = open(fn, "rt")
	sqllist = []
	currcmd = ''
	for line in sqlfile:
		line = line.strip()
		if len(line) > 0 and not (len(line) > 1 and line[:2] == "--"):
			currcmd = "%s %s" % (currcmd, line)
			if line[-1:] == ';':
				sqllist.append(currcmd.strip())
				currcmd = ''
	return sqllist

def execsqlfile( fn, db, ofname=None ):
	"""Execute all the SQL commands in the file against the specified database."""
	sqlcmds = read_sqlfile(fn)
	execsql(sqlcmds, db, ofname)

def sql_to_Access( sql_fn, Access_fn, output_fn=None ):
	"""Opens the specified Access database and executes all SQL in the given script file."""
	daoEngine = win32com.client.Dispatch('DAO.DBEngine.36')
	db = daoEngine.OpenDatabase(Access_fn)
	execsqlfile(sql_fn, db, output_fn)

def print_help():
	"""Print a help message."""
	print "execsql %s %s -- Executes a file of SQL commands against an Access database" % (_version, _vdate)
	print __HELP_MSG

if __name__ == "__main__":
	if len(sys.argv) < 3 or len(sys.argv) > 4:
		print_help()
	else:
		if len(sys.argv) == 3:
			sql_to_Access(sys.argv[1], sys.argv[2])
		else:
			sql_to_Access(sys.argv[1], sys.argv[2], sys.argv[3])

