#! python

# execsql.py
# 
# PURPOSE
# =======
# Read a sequence of SQL commands from a file and execute them against
# a specified database (Access only, for now).  The reason for this script
# is to allow execution of a SQL script--i.e., a sequence of SQL statements--because:
# 	*	Access only allows a single SQL command to be executed from the query window;
# 	*	The Jet engine can handle complex SQL statements that Access cannot, and 
# 		that therefore cannot be saved as queries in Access, so executing them
# 		directly from a text file is more convenient than copying and pasting them
# 		into the query design window.
# The output of the last SQL statement in the script can be saved to a disk file
# or directed to a new Excel workbook in memory.
# 
# AUTHOR
# ======
# Dreas Nielsen (RDN)
# 
# COPYRIGHT AND LICENSE
# =====================
# Copyright (c) 2007-2009, R.Dreas Nielsen
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# The GNU General Public License is available at <http://www.gnu.org/licenses/>
# 
# NOTES
# =====
# 1. Because of its interaction with Access and Excel, this is a Windows-only utility.
# 2. Mark Hammond's win32com package is required for interaction with MS applications.
# 3. The Access database must not have security enabled.
# 4. Redirecting output to an Excel worksheet in memory is slow.
# 5. Meta-commands can be embedded in SQL comments.  These commands allow you to:
#		* Export a query or table to a CSV or tab-delimited file, or the console
#		* Write a message to the console or to a file
#		* Halt script processing
#		* Carry out any of these actions only if:
#			+ A specified query or table has a non-zero number of records
#			+ The preceding SQL statement produced an error.
# 
# HISTORY
# =======
# -------		--------------------------------------------------------------------
#  Date		 Comments
# -------		--------------------------------------------------------------------
# 11/11/2007	Created.  RDN
# 1/1/2008	Modified to allow writing of the output of the last SQL
# 			command to a CSV file.  RDN.
# 4/20/2008	Converted to use DAO to interact with Access
# 			instead of the dbconnect library.  RDN.
# 4/22/2008	Added regular expression to match 'create temp view...' preface.  RDN.
# 4/26/2008	Added creation and deletion of temporary views (queries) and
# 			export of final query to Excel.  RDN.
# 5/20/2008	Added unicode encoding for code page 1252 for data read from Access
# 			and written to CSV.  RDN.
# 12/18/2008	Added interpreter name to first line for distribution.  RDN.
# 12/19/2008	Added docstrings to two functions.  RDN.
# 5/2/2009	Improved error reporting.  RDN.
# 10/25/2009	Implemented simple metacommands embedded in comments that alllow
# 			export of data table and writing of simple text.  RDN.
# 10/28/2009	Eliminated saving (and attempted deletion) of duplicate temporary
# 			query names.  RDN.
# 10/31/2009	Implemented the 'sql_error()' conditional and the 'halt'
#			metacommand.  RDN.
# 11/1/2009	Debugging of 10/31/2009 changes.  RDN.
# 11/3/2009	Modifed to use both an odbc connection and a dao connection,
#			the former to allow errors to be caught, the latter to allow
#			temporary queries to be created and deleted.  RDN.
# 11/20/2009	Modified information extracted from exc_info() on line 479.  RDN.
# 11/21/2009	Modifed all occurrences of exc_info() to pull just item [1].  RDN.
# 11/28/2009	Modified to re-open ODBC connections before each statement so that
#				newly created queries will be seen by ODBC, and added SQL error
#				trapping.  RDN.
# 12/1/2009	Added a commit() after executing SQL in 'exec_one_sql()'.  RDN.
# 12/14/2009	Added a 'EXECUTE <query>' metacommand and a 'NOT' option for the
#				'HASROWS()' conditional.  Added 'INCLUDE' and 'SUB' metacommands.  RDN.
# ==========================================================================


#================================================================================
# TODO:
#	1. Add options to specify a database user name and password if appropriate.
#	2. [DONE] Trap database integrity errors encountered with .Execute() in
#		'exec_one_sql()' and set status.sql_error.
#	3. Add a real parser/exec for the command language embedded in SQL comments,
#      and add more functions.
#	4. Add loading of VB modules from a text file into an Access database
#		prior to executing the SQL.
#================================================================================

import os.path
import sys
import csv
import re
try:
	import win32com.client
except:
	sys.exit("The win32com module is required.\n    See http://sourceforge.net/projects/pywin32/\n")
try:
	import pyodbc
except:
	sys.exit("The pyodbc module is required.\n    See http://github.com/mkleehammer/pyodbc\n")

_version = "0.4.4.0"
_vdate = "2009-12-14"

__HELP_MSG = """Copyright (c) 2007-2009, R.Dreas Nielsen
Licensed under the GNU General Public License version 3.
Usage:
   execsql.py <sqlfile> <Access_db> [output_spec]
Arguments:
   <sqlfile>
      The name of a file of SQL commands to be executed.  Required argument.
   <Access_db>
      The name of the Access database against which to run the SQL.  Required argument.
   [output_spec]
      The name of a CSV file to which the output of the last SQL command should
      be written, or the literal string "Excel" to send output to a Microsoft Excel
      workbook in memory.  Note that Excel has row and column limitations, CSV does not.
      Optional argument.
Metacommands:
   Metacommands are embedded in SQL comment lines following the !x! token.
   INCLUDE <sql_command_file>
   SUB <match_str> <repl_str>
   EXECUTE <view_name>
   EXPORT <view_name> [APPEND] TO <filename> | stdout AS <CSV | TAB | TXT> 
   HALT 
   IF( [NOT] HASROWS(<view_name>) ) { <metacommand> }
   IF( SQL_ERROR() ) { <metacommand> } 
   WRITE "<text>" [TO <filename>]"""

class ExecCmdError(Exception):
	pass

class StatObj:
	pass

status = StatObj()
status.sql_error = False

# List of tuples indicating strings to match and their replacements.
substitutions = []

# Regex for the 'create temporary view' SQL extension
temp_rx = re.compile(r'^\s*create(?:\s+or\s+replace)?(\s+temp(?:orary)?)?\s+(?:(view|query))\s+(\w+) as\s+', re.I)

# Regexes for execsql commands
class Xcmd():
	sub_rx = re.compile(r'^SUB\s+(?P<match>\w+)\s+(?P<repl>.+)$')
	run_rx = re.compile(r'^(?P<cmd>RUN|EXECUTE)\s+(?P<queryname>\w+)$')
	export_rx = re.compile(r'^EXPORT\s+(?P<queryname>\w+)\s+(?P<append>APPEND\s+)?TO\s+(?P<outputname>[a-zA-Z0-9:._/\\]+)\s+AS\s*(?P<format>CSV|TAB|TXT)\s*$', re.I)
	#write_rx = re.compile(r'^WRITE\s+(?P<quote>["\'])(?P<text>\w+)(?P=quote)(?:\s+TO\s+(?P<outputname>[a-zA-Z0-9:._/\\]+))?\s*$', re.I)
	write_rx = re.compile(r'^WRITE\s+"(?P<text>.+)"(?:\s+TO\s+(?P<outputname>[a-zA-Z0-9:._/\\]+))?\s*$', re.I)
	halt_rx = re.compile(r'^halt$', re.I)
	if_rx = re.compile(r'IF\s*\((?P<condtest>.+)\)\s*{\s*(?P<condcmd>.+)}\s*$', re.I)
	condrows_rx = re.compile(r'^(?P<invert>NOT\s+)?HASROWS\((?P<queryname>.+)\)\s*$', re.I)
	conderr_rx = re.compile(r'^sql_error\(\s*\)\s*$', re.I)

class WriteHooks():
	def __init__(self, standard_output_func=None, error_output_func=None):
		"""Arguments should be functions that take a single string and
		write it to the desired destination.  Both stdout and stderr can be hooked.
		If a hook function is not specified, the default of stdout or stderr will
		be used."""
		self.write_func = standard_output_func
		self.err_func = error_output_func
	def reset(self):
		"""Resets output to stdout and stderr."""
		self.write_func = None
		self.err_func = None
	def redir_stdout(self, standard_output_func):
		self.write_func = standard_output_func
	def redir_stderr(self, error_output_func):
		self.err_func = error_output_func
	def redir(self, standard_output_func, error_output_func):
		self.redir_stdout(standard_output_func)
		self.redir_stderr(error_output_func)
	def write(self, str):
		if self.write_func:
			self.write_func("%s\n" % str)
		else:
			sys.stdout.write("%s\n" % str)
	def write_err(self, str):
		if self.err_func:
			self.err_func("%s\n" % str)
		else:
			sys.stderr.write("%s\n" % str)

output = WriteHooks()

class ErrInfo():
	SQL_ERR_MSG = "**** Error in SQL command starting on line %d of %s:\n     %s\n     SQL command:\n%s"
	SQL_SIMPLE_ERR_MSG = "**** Error in SQL command:\n%s\n**** SQL command:\n%s"
	EXEC_ERR_MSG = "**** Error in execsql command on line %d of %s:\n     %s\n     execsql command: %s"
	EXCEPT_ERR_MSG1 = "**** Error: %s\n     %s"
	EXCEPT_ERR_MSG2 = "**** Error: %s"
	OTHER_ERR_MSG = "**** Error: %s"
	def __init__(self, type, command_text=None, exception_msg=None, other_msg=None):
		"""Argument 'type' should be "db", "cmd", "exception", or "other".
		Arguments for each type are as follows:
			"db"	: command_text, exception_msg
			"cmd"	: command_text, <exception_msg | other_msg>
			"exception"	: exception_msg [, other_msg]
			"other"	: other_msg
		"""
		self.type = type
		self.command = command_text
		self.exception = exception_msg
		self.other = other_msg
	def write(self, cmdobj=None):
		if self.type=='db':
			if cmdobj:
				output.write_err(self.SQL_ERR_MSG % (cmdobj.line_no, cmdobj.source, self.exception, cmdobj.sql))
			else:
				output.write_err(self.SQL_SIMPLE_ERR_MSG % (self.exception, self.command))
		elif self.type=='cmd':
			if cmdobj:
				if self.exception:
					output.write_err(self.EXEC_ERR_MSG % (cmdobj.line_no, cmdobj.source, self.exception, self.command))
				else:
					output.write_err(self.EXEC_ERR_MSG % (cmdobj.line_no, cmdobj.source, self.other, self.command))
		elif self.type=='exception':
			if self.other:
				output.write_err(self.EXCEPT_ERR_MSG1 % (self.exception, self.other))
			else:
				output.write_err(self.EXCEPT_ERR_MSG2 % (self.exception))
		elif self.type=='other':
			output.write_err(self.OTHER_ERR_MSG % self.other)
		else:
			output.write_err("**** Error of unknown type")
			if self.exception:
				output.write_err("     Exception: %s" % self.exception)
			if self.command:
				output.write_err("     Command: %s" % self.command)
			if self.other:
				output.write_err("     Other information: %s" % self.other)

class Database():
	def __init__(self, Access_fn):
		self.name = Access_fn
		self.dao_conn = None
		self.odbc_conn = None
		# Create the DAO connection
		self.open_dao()
		# Create the ODBC connection
		self.open_odbc()
	def open_odbc(self):
		"""Open an ODBC connection."""
		# Because temporary queries are created using DAO, and the ODBC refresh interval
		# set in the database may be several minutes, those tempoary queries won't be seen
		# by routines accessing the database via ODBC unless the ODBC connection is re-created
		# immediately prior to use.  Therefore, this routine should be run before every
		# ODBC query that might reference a temporary query.
		if self.odbc_conn is not None:
			self.odbc_conn.close()
			self.odbc_conn = None
		try:
			self.odbc_conn = pyodbc.connect("DRIVER={Microsoft Access Driver (*.mdb)};DBQ=%s;" % self.name)
		except:
			ErrInfo("db", exception_msg=sys.exc_info()[1], command_text="Open database %s with ODBC" % self.name).write()
	def open_dao(self):
		if self.dao_conn is not None:
			self.dao_conn.Close
			self.dao_conn = None
		daoEngine = win32com.client.Dispatch('DAO.DBEngine.36')
		try:
			self.dao_conn = daoEngine.OpenDatabase(self.name)
		except:
			ErrInfo("db", exception_msg=sys.exc_info()[1], command_text="Open database %s with DAO" % self.name).write()
	def commit(self):
		self.odbc_conn.commit()

class SqlCmd():
	"""Define an SQL or execsql command and source file information."""
	def __init__(self, command_source_name, command_line_no, command_type, sql_text):
		self.source = command_source_name
		self.line_no = command_line_no
		self.command_type = command_type
		self.sql = sql_text


# Adapted from the pp() function by Aaron Watters,
# posted to gadfly-rdbms@egroups.com 1999-01-18.
# Formatting modified to work with win32api recordsets.
def prettyprint_query(select_stmt, db, outfile, append=False):
	status.sql_error = False
	db.open_odbc()
	curs = db.odbc_conn.cursor()
	try:
		curs.execute(select_stmt)
	except:
		status.sql_error = True
		return ErrInfo("db", select_stmt, sys.exc_info()[1])
	#try:
	#	rs = db.dao_conn.OpenRecordset(select_stmt)
	#except:
	#	status.sql_error = True
	#	return ErrInfo("db", select_stmt, sys.exc_info()[1])
	#rows = zip(*rs.GetRows(1000000)) 
	#names = [f.Name for f in rs.Fields]
	#rcols = range(len(names))
	#rrows = range(len(rows))
	rows = curs.fetchall()
	desc = curs.description
	names = [d[0] for d in desc]
	rcols = range(len(desc))
	rrows = range(len(rows))
	maxen = [max(0,len(names[j]),*(len(str(rows[i][j]))
			for i in rrows)) for j in rcols]
	names = ' '+' | '.join(
			[names[j].ljust(maxen[j]) for j in rcols])
	sep = '+'.join([ '-'*(maxen[j]+2) for j in rcols])
	rows = [names, sep] + [' '+' | '.join(
			[str(rows[i][j]).ljust(maxen[j])
			for j in rcols] ) for i in rrows]
	if outfile == 'stdout':
		ofile = sys.stdout
		margin = '    '
	else:
		margin = ''
		if append:
			ofile = open(outfile, "a")
		else:
			ofile = open(outfile, "w")
	for row in rows:
		ofile.write("%s%s\n" % (margin, row))
	return None

def set_sub(matchstr, replstr):
	"""Arguments are the string to match and the value to replace it with."""
	substitutions.append( (matchstr, replstr) )

def run_query(qry, db):
	"""Arguments are a query name, and the database to run it against."""
	try:
		db.open_odbc()
		curs = db.odbc_conn.cursor()
		curs.execute("select * from %s;" % qry)
	except:
		status.sql_error = True
		return ErrInfo("db", qry, sys.exc_info()[1])
	return None

def exec_query(qry, db):
	"""Arguments are a query name, and the database to run it against."""
	try:
		db.dao_conn.Execute(qry)
	except:
		status.sql_error = True
		return ErrInfo("db", command_text=qry, exception_msg=sys.exc_info()[1])


def write_query_to_file(select_stmt, db, outfile, format, append=False):
	"""Arguments are a SQL select statement, an open database object, the
	full name of the output file to use, and the format.  The format must be either
	'csv' or 'tab'."""
	status.sql_error = False
	if format.lower()=='csv':
		dialect = 'excel'
	elif format.lower()=='tab':
		dialect = 'excel-tab'
	elif format.lower()=='txt':
		return prettyprint_query(select_stmt, db, outfile, append)
	else:
		return ErrInfo("other", other_msg='Invalid output file format specification: %s' % format)
	try:
		#rs = db.dao_conn.OpenRecordset(select_stmt)
		db.open_odbc()
		curs = db.odbc_conn.cursor()
		curs.execute(select_stmt)
	except:
		status.sql_error = True
		return ErrInfo("db", select_stmt, sys.exc_info()[1])
	#hdrs = [f.Name for f in rs.Fields]
	hdrs = [c[0] for c in curs.description]
	if outfile == 'stdout':
		ofile = csv.writer(sys.stdout, dialect=dialect)
	else:
		if append:
			try:
				ofile = csv.writer(open(outfile, "ab"), dialect=dialect)
			except:
				return ErrInfo("exception", exception_msg=sys.exc_info()[1], other_msg="Can't open %s to write output." % outfile)
		else:
			try:
				ofile = csv.writer(open(outfile, "wb"), dialect=dialect)
			except:
				return ErrInfo("exception", exception_msg=sys.exc_info()[1], other_msg="Can't open %s to write output." % outfile)
	ofile.writerow([h.encode("cp1252") for h in hdrs])
	#while not rs.EOF:
	#	try:
	#		rec = rs.GetRows(1)
	#	except:
	#		return ErrInfo("db", select_stmt, sys.exc_info()[1])
	#	try:
	#		ofile.writerow([(fldval[0].encode("cp1252") if isinstance(fldval[0], basestring) else fldval[0]) for fldval in rec])
	#	except:
	#		return ErrInfo("exception", exception_msg=sys.exc_info()[1], other_msg="Can't write output to CSV file %s." % outfile)
	for rec in curs.fetchall():
		try:
			ofile.writerow([(fldval.encode("cp1252") if isinstance(fldval, basestring) else fldval) for fldval in rec])
		except:
			return ErrInfo("exception", exception_msg=sys.exc_info()[1], other_msg="Can't write output to CSV file %s." % outfile)
	if outfile != 'stdout':
		ofile = None
	#rs.Close()
	return None

def write_query_to_Excel(select_stmt, db):
	status.sql_error = False
	try:
		rs = db.dao_conn.OpenRecordset(select_stmt)
	except:
		status.sql_error = True
		return ErrInfo("db", command_text=select_stmt, exception_msg=sys.exc_info()[1])
	hdrs = [f.Name for f in rs.Fields]
	try:
		xl = win32com.client.Dispatch('Excel.Application')
	except:
		return ErrInfo("exception", exception_msg=sys.exc_info()[1], other_msg=("Can't instantiate Excel to write SQL output to it."))
	xl.Workbooks.Add()
	row = 1
	col = 1
	for hdr in hdrs:
		try:
			xl.ActiveSheet.Cells(row, col).Value = hdr
		except:
			return ErrInfo("exception", exception_msg=sys.exc_info()[1], other_msg=("Can't write column headers to Excel."))
		col += 1
	while not rs.EOF:
		row += 1
		rec = rs.GetRows(1)
		vals = [fldval[0] for fldval in rec]
		col = 1
		for val in vals:
			try:
				xl.ActiveSheet.Cells(row, col).Value = val
			except:
				return ErrInfo("exception", exception_msg=sys.exc_info()[1], other_msg=("Can't write data (%s) to Excel." % val))
			col += 1
	rs.Close()
	xl.Visible = True
	return None

def xf_hasrows( queryname, db, invert=False ):
	status.sql_error = False
	sql = "select count(*) from %s;" % queryname
	# Exceptions should be trapped by the caller, so are re-raised here after settting status
	try:
		# DAO version
		#rs = db.dao_conn.OpenRecordset(sql)
		#rec = rs.GetRows(1)
		# ODBC version
		db.open_odbc()
		curs = db.odbc_conn.cursor()
		curs.execute(sql)
		rec = curs.fetchall()
	except:
		status.sql_error = True
		raise
	nrows = rec[0][0]
	#rs.Close()
	if invert:
		return nrows == 0
	return nrows > 0

def xf_sqlerror():
	return status.sql_error

def xcmd_test( teststr, db ):
	"""Parse out an expression from the given string and evaluate it as a Boolean."""
	m = Xcmd.condrows_rx.match(teststr.strip())
	if m:
		q = m.group('queryname').strip()
		invert = m.group('invert') is not None
		return xf_hasrows(q, db, invert)
	else:
		m = Xcmd.conderr_rx.match(teststr.strip())
		if m:
			return xf_sqlerror()
		else:
			raise ExecCmdError("Unrecognized conditional: %s" % teststr )

def xcmd_exec( cmdstr, db ):
	"""Parse out a metacommand from the given string and
	execute it (against the database object supplied, if appropriate)."""
	cmdstr = cmdstr.strip()
	m = Xcmd.export_rx.match(cmdstr)
	if m:
		qry = m.group('queryname')
		op = m.group('outputname')
		fmt = m.group('format')
		app = m.group('append') is not None
		return write_query_to_file("select * from %s;" % qry, db, op, fmt, app)
	else:
		m = Xcmd.write_rx.match(cmdstr)
		if m:
			try:
				o = m.group('outputname')
				t = m.group('text')
				if o:
					open(o, 'a').write('%s\n' % t)
				else:
					sys.stdout.write('%s\n' % t)
			except:
				return ErrInfo("cmd", command_text=cmdstr, exception_msg=sys.exc_info()[1])
		else:
			m = Xcmd.halt_rx.match(cmdstr)
			if m:
				sys.exit(1)
			else:
				m = Xcmd.if_rx.match(cmdstr)
				if m:
					try:
						tst = m.group('condtest')
						cmd = m.group('condcmd')
						if xcmd_test( tst, db ):
							return xcmd_exec( cmd, db )
					except SystemExit:
						raise
					except:
						return ErrInfo("cmd", command_text=cmdstr, exception_msg=sys.exc_info()[1])
				else:
					m = Xcmd.run_rx.match(cmdstr)
					if m:
						try:
							cmd = m.group('cmd')
							qry = m.group('queryname')
							if cmd.lower() == "execute":
								return exec_query( qry, db )
							return run_query( qry, db )
						except SystemExit:
							raise
						except:
							return ErrInfo("cmd", command_text=cmdstr, exception_msg=sys.exc_info()[1])
					else:
						m = Xcmd.sub_rx.match(cmdstr)
						if m:
							try:
								set_sub(m.group('match'), m.group('repl'))
							except SystemExit:
								raise
							except:
								return ErrInfo("cmd", command_text=cmdstr, exception_msg=sys.exc_info()[1])
	return None

def xcmd_dispatch( cmd_obj, db ):
	"""Parse out an execsql-specific command from the 'sql' attribute of
	the command object and execute it (against the database object supplied,
	if appropriate).  Return a Bookean indicating success."""
	cmd = cmd_obj.sql.strip()
	try:
		e = xcmd_exec( cmd, db )
	except SystemExit:
		raise
	except: 
		ErrInfo("cmd", exception_msg=sys.exc_info()[1]).write(cmd_obj)
		return False
	if e:
		e.write( cmd_obj )
		return False
	else:
		return True
	

def exec_one_sql( sql_cmd_obj, db, tempquerynames, outfile=None ):
	# Use the DAO connection to make temporary queries.
	# Use the ODBC connection to execute other SQL commands, so that errors can be trapped.
	status.sql_error = False
	sqlcmd = sql_cmd_obj.sql
	for match, sub in substitutions:
		pat = "!!%s!!" % match
		sqlcmd = sqlcmd.replace(pat, sub)
	tqd = temp_rx.match(sqlcmd)
	if tqd:
		qn = tqd.group(3)
		qsql = sqlcmd[tqd.end():]
		try:
			db.dao_conn.QueryDefs.Delete(qn)
		except:
			pass
		try:
			db.dao_conn.CreateQueryDef(qn, qsql)
		except:
			status.sql_error = True
			ErrInfo("db", command_text=sqlcmd, exception_msg=sys.exc_info()[1]).write(sql_cmd_obj)
			return False
		if tqd.group(1) and tqd.group(1).strip().lower()[:4] == 'temp':
			if not (qn in tempquerynames):
				tempquerynames.append(qn)
	else:
		if outfile is not None:
			if outfile == "Excel":
				e = write_query_to_Excel(sqlcmd, db)
				if e:
					status.sql_error = True
					e.write(sql_cmd_obj)
					return False
			else:
				e = write_query_to_file(sqlcmd, db, outfile, 'csv')
				if e:
					status.sql_error = True
					e.write(sql_cmd_obj)
					return False
		else:
			try:
				db.open_odbc()
				curs = db.odbc_conn.cursor()
				curs.execute(sqlcmd)
				db.commit()
			except:
				status.sql_error = True
				ErrInfo("db", command_text=sqlcmd, exception_msg=sys.exc_info()[1]).write(sql_cmd_obj)
				return False
	return True

def exec_all_sql( sqllist, db, outfile=None, halt_on_err=False ):
	"""Execute all SQL commands in the given list of SqlCmd objects against
	the given Access database.
	Arguments:
		'sqllist' is a list of SqlCmd objects containing SQL statements and execsql
		commands, to be executed in order.
		'db' is an initialized Database object, which must represent an existing database.
		'outfile' is the name of a CSV file in which to dump the results of the last command.
	"""
	tempquerynames = []
	for i in range(len(sqllist)):
		sqlitem = sqllist[i]
		if sqlitem.command_type == 'sql':
			ok = exec_one_sql(sqlitem, db, tempquerynames, (None, outfile)[outfile is not None and i==len(sqllist)-1])
			if not ok:
				if halt_on_err:
					sys.exit(1)
		elif sqlitem.command_type == 'cmd':
			if not xcmd_dispatch(sqlitem, db):
				if halt_on_err:
					sys.exit(1)
	for qn in tempquerynames:
		try:
			db.dao_conn.QueryDefs.Delete(qn)
		except:
			pass


def read_sqlfile( fn ):
	"""Open the text file with the specified name, read it, and return a list of
	SqlCmd objects representing either SQL statements or 'execsql' command statements.
	"""
	# Lines containing execsql command statements must begin with "-- !x!"
	# Currently this routine knows only two things about SQL:
	#	1. Lines that start with "--" are comments.
	#	2. Lines that end with ";" terminate a SQL statement.'
	includeline = re.compile(r'^--\s+!x!\s*INCLUDE\s+(?P<filename>.+)$', re.I)
	execline = re.compile(r'^--\s+!x!\s*(?P<cmd>.+)$', re.I)
	cmtline = re.compile(r'^--')
	sqlfile = open(fn, "rt")
	sqllist = []
	currcmd = ''
	lno = 0
	for line in sqlfile:
		lno += 1
		line = line.strip()
		if len(line) > 0:
			m = includeline.match(line)
			if m:
				incfn = m.group('filename')
				if not os.path.exists(incfn):
					ErrInfo("other", other_msg="Command file %s not found." % incfn).write()
				else:
					sqllist.extend(read_sqlfile(incfn))
			else:
				m = execline.match(line)
				if m:
					sqllist.append(SqlCmd(fn, lno, 'cmd', m.group('cmd').strip()))
				elif not cmtline.match(line):
					if currcmd == '':
						sqlline = lno
						currcmd = line
					else:
						currcmd = "%s %s" % (currcmd, line)
					if line[-1:] == ';':
						sqllist.append(SqlCmd(fn, sqlline, 'sql', currcmd.strip()))
						currcmd = ''
	return sqllist

def execsqlfile( fn, db, ofname=None ):
	"""Execute all the SQL commands in the file against the specified database."""
	sqlcmds = read_sqlfile(fn)
	exec_all_sql(sqlcmds, db, ofname)

def sql_to_Access( sql_fn, Access_fn, output_fn=None ):
	"""Opens the specified Access database and executes all SQL in the given script file."""
	if not os.path.exists(sql_fn):
		ErrInfo("other", other_msg="SQL command file %s not found." % sql_fn).write()
		exit(1)
	if not os.path.exists(Access_fn):
		ErrInfo("other", other_msg="Access database file %s not found." % Access_fn).write()
	db = Database(Access_fn)
	execsqlfile(sql_fn, db, output_fn)

def print_help():
	"""Print a help message."""
	print "execsql %s %s -- Executes a file of SQL commands against an Access database" % (_version, _vdate)
	print __HELP_MSG

if __name__ == "__main__":
	if len(sys.argv) < 3 or len(sys.argv) > 4:
		print_help()
	else:
		if len(sys.argv) == 3:
			sql_to_Access(sys.argv[1], sys.argv[2])
		else:
			sql_to_Access(sys.argv[1], sys.argv[2], sys.argv[3])

