__version__ = '0.1.0'


def average(logs, inertia):
    """Calculate average using exponential smoothing

    :param logs: iterable contains (key, value) tuples
    :param inertia: weight of old average (between 0.0 to 1.0)
    """
    records = {}

    for key, value in logs:
        # create or find record
        if key in records:
            record = records[key]
        else:
            record = [value, value, 0]
            records[key] = record

        record[1] = (record[1] * inertia) + (value * (1 - inertia))
        record[2] += 1

    return dict(
        (key, (init, rate, count))
        for key, (init, rate, count) in records.items()
    )


def trial_success(logs, inertia):
    """Calculate success rate from asynchronous trial/success event log

    :param logs: iterable contains (key, event) tuples, where event is 'trial'
                 or 'success'
    :param inertia: weight of old average (between 0.0 to 1.0)
    """
    records = {}

    for key, event in logs:
        # create or find record
        if key in records:
            record = records[key]
        else:
            record = [0.0, 0.0, 0]
            records[key] = record

        # update record
        if event == 'trial':
            record[1] = record[1] * inertia
            record[2] += 1
        elif event == 'success':
            # if there's no trial at all, assume there was single trial
            if record[2] == 0:
                record[2] = 1
            # if it's the first success record, do not apply inertia
            if record[1] == 0:
                record[0] = 1. / record[2]
                record[1] = record[0]
            else:
                record[1] += 1 - inertia
        else:
            raise ValueError('Unknown event: %s' % event)

    return dict(
        (key, (init, rate, count))
        for key, (init, rate, count) in records.items()
    )


def merge(old_records, new_records, inertia):
    """Merge old records and new records using exponential smoothing

    :param old_records: a dictionary containing old records
    :param new_records: a dictionary containing new records
    :param inertia: an inertia value used to calculate old and new records
    """
    merged = old_records.copy()
    for key, (init, value, count) in new_records.items():
        if key in merged:
            old_weight = inertia ** count
            old_init, old_value, old_count = merged[key]
            merged[key] = (
                old_init,
                (old_value * old_weight) + (value * (1 - old_weight)),
                old_count + count
            )
        else:
            merged[key] = (init, value, count)
    return merged
