import os, sys
import time
import datetime
import argparse
import tempfile
import logging
import picamera
from conf import settings
from storage.s3 import S3Storage

logger = logging.getLogger(__name__)
logger.setLevel(logging.INFO)
handler = logging.FileHandler('eyespi.log')
handler.setLevel(logging.INFO)
formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
handler.setFormatter(formatter)
logger.addHandler(handler)


class Timelapse(object):

    def __init__(self, interval, maxtime, maxshots, rotation, filestore):
        logger.info('Initializing picamera instance')

        self.interval = interval
        self.maxtime = maxtime
        self.maxshots = maxshots
        self.format = 'jpg'

        self._filestore = filestore

        self._camera = picamera.PiCamera()
        self._camera.rotation = rotation

    def __del__(self):
        logger.info('Cleaning up picamera instance')
        self._camera.close()

    def start(self):
        self._camera.start_preview()
        # Camera warm-up time
        time.sleep(2)
        while True:
            ts = datetime.datetime.now().strftime('%Y-%m-%d-%H%M%S')
            frame_name = '%s.%s' % (ts, self.format,)
            with tempfile.TemporaryFile() as fp:
                self._camera.capture(fp, format=self.format)
                fp.seek(0)
                self.store(fp, frame_name)
            self.wait(self.interval)

    def store(self, image_file, path):
        self._filestore.set_file(image_file, path)

    def cleanup(self):
        logger.info('cleaning')

    def wait(self, interval):
        time.sleep(interval)


def main(argv):

    parser = argparse.ArgumentParser(description='Timelapse tool for the Raspberry Pi.')
    parser.add_argument( '-i', '--interval', default=15, type=int, help='Set photo interval in seconds. Recommended miniumum is 6.' )
    parser.add_argument( '-t', '--maxtime', default=-1, type=int, help='Maximum duration of timelapse in seconds.\nDefault is -1, for no maximum duration.' )
    parser.add_argument( '-n', '--maxshots', default=-1, type=int, help='Maximum number of photos to take.\nDefault is -1, for no maximum.' )
    parser.add_argument( '-r', '--rotation', default=0, type=int, help='Rotation to apply to images. Default is 0.' )
    parser.add_argument( '--name', default=None, type=str, help='Name to give the timelapse (for storage purposes). Default is the current timestamp.' )
    args=parser.parse_args()

    ts_name = None
    if args.name is not None:
        ts_name = args.name
    else:
        ts_name = datetime.datetime.now().strftime('%Y-%m-%d-%H%M%S')
    s3_store = S3Storage(settings.S3_FILESTORE_INFO)
    s3_store.set_prefix(ts_name)

    timelapse = Timelapse(args.interval, args.maxtime, args.maxshots, args.rotation, s3_store)
    timelapse.start()

    return True

if __name__ == "__main__":
   main(sys.argv[1:])
