# pagination for mongoengine
# credits: http://pastebin.com/zzL3pedt

class Paginator(object):

    def __init__(self, query, per_page):
        """Initialize a paginator.

        :param query: Iterable from pymongo or mongoengine
        :param per_page: Number of items per page
        :rtype: A ``Page`` instance, with a list of objects.

        Usage: ::

            latest_blog_posts = BlogPost.objects()
            paginator = Paginator(latest_blog_posts, 10)
            
            # get first page of blog posts
            page = paginator.page(1)
            
            for post in page.object_list:
                # do something with BlogPost instance
                print post.title

        """
        from math import ceil

        per_page = int(per_page)
        self.query = query
        self.per_page = per_page
        self._count = query.count()
        self.page_count = int(ceil(float(self._count) / per_page))

    def page_range(self):
        """Return a 1-based list of page numbers valid for this ``Paginator``.
        """
        return range(1, self.page_count + 1)

    def page(self, page_number):
        """1-based access to pages.
        """
        if not isinstance(page_number, int):
            try:
                page_number = int(page_number)
            except:
                raise Exception("Page numbers must be integers.")
        if page_number < 1:
            raise Exception("%s is less than 1." % page_number)
        if page_number > self.page_count:
            raise Exception("Page %s contains no results." \
                % page_number)

        # apply pagination offsets and limits
        start_offset = self.per_page * (page_number-1)
        end_offset = self.per_page + start_offset
        query = self.query[start_offset:end_offset]

        # return Page object
        object_list = list(query)
        return Page(object_list, page_number, self.page_count, self._count)


class Page(object):

    def __init__(self, object_list, page_number, page_count, 
                 all_objects_count):
        """A page of data generated by ``Paginator``.
        
        :param object_list: A ``list`` this page's objects
        :param page_number: 1-based index of this page
        :param page_count: Total number of pages
        :param all_objects_count: Total count of objects in original queryset.
        """
        self.object_list = object_list
        self.page_number = page_number
        self.page_count = page_count
        self.all_objects_count = all_objects_count

    def __repr__(self):
        return u"<Page %s of %s>" % (self.page_number, self.page_count)

    def has_previous(self):
        return self.page_number > 1

    def has_next(self):
        return self.page_number < self.page_count

    def has_other_pages(self):
        return self.has_next() or self.has_previous()

    def next_page_number(self):
        return self.page_number + 1

    def previous_page_number(self):
        return self.page_number - 1