# -*- coding: utf-8 -*-

"""
Tests for all number generators
"""

from fauxfactory import FauxFactory

import sys
import unittest


class TestNumbers(unittest.TestCase):
    """
    Test number generators
    """

    @classmethod
    def setUpClass(cls):
        """
        Instantiate our factory object
        """

        cls.factory = FauxFactory()

    def test_generate_integer_1(self):
        """
        @Test: Create a random integer with no range limits
        @Feature: Numbers Generator
        @Assert: A random integer is created
        """

        result = self.factory.generate_integer()
        self.assertTrue(
            isinstance(result, int), "A valid integer was not generated.")

    def test_generate_integer_2(self):
        """
        @Test: Create a random integer with set minimum limit
        @Feature: Numbers Generator
        @Assert: Integer is created and greater than minimum
        """

        try:
            # Change system max int to a smaller number
            old_sys_maxint = sys.maxint
            sys.maxint = 5

            for turn in xrange(10):
                result = self.factory.generate_integer(min_value=1)
                self.assertTrue(
                    result <= sys.maxint, "Integer is greater than max_value"
                )
                self.assertTrue(
                    result >= 1, "Integer is less than specified minimum"
                )
        finally:
            # Reset system max int back to original value
            sys.maxint = old_sys_maxint

    def test_generate_integer_3(self):
        """
        @Test: Create a random integer with set maximum limit
        @Feature: Numbers Generator
        @Assert: Integer is created and less than maximum value
        """

        try:
            # Change system max int to a smaller number
            old_sys_maxint = sys.maxint
            sys.maxint = 5
            min_value = - sys.maxint - 1

            for turn in xrange(10):
                result = self.factory.generate_integer(max_value=1)
                self.assertTrue(
                    result >= min_value, "Integer is less than min_value"
                )
                self.assertTrue(
                    result <= 1, "Integer is greater than specified maximum"
                )
        finally:
            # Reset system max int back to original value
            sys.maxint = old_sys_maxint

    def test_generate_integer_4(self):
        """
        @Test: Create a random integer with set min/max limits
        @Feature: Numbers Generator
        @Assert: An integer is created and falls within the specified range
        """

        for turn in xrange(10):
            result = self.factory.generate_integer(
                min_value=1, max_value=3)
            self.assertTrue(
                result >= 1, "Integer is less than min_value"
            )
            self.assertTrue(
                result <= 3, "Integer is greater than specified maximum"
            )

    def test_generate_integer_5(self):
        """
        @Test: Create a random integer with disallowed minimum limit
        @Feature: Numbers Generator
        @Assert: An integer number is not created due to value error
        """

        # This is lower than allowed platform minimum
        low_min = - sys.maxint - 2

        with self.assertRaises(ValueError):
            self.factory.generate_integer(min_value=low_min)

    def test_generate_integer_6(self):
        """
        @Test: Create a random integer with disallowed maximum limit
        @Feature: Numbers Generator
        @Assert: An integer number is not created due to value error
        """

        # This is greater than allowed platform maximum
        high_max = sys.maxint + 1

        with self.assertRaises(ValueError):
            self.factory.generate_integer(max_value=high_max)

    def test_generate_integer_7_0(self):
        """
        @Test: Create a random integer using empty strings as args
        @Feature: Numbers Generator
        @Assert: An integer number is not created due to value error
        """

        with self.assertRaises(ValueError):
            self.factory.generate_integer(min_value='')

    def test_generate_integer_7_1(self):
        """
        @Test: Create a random integer using empty strings as args
        @Feature: Numbers Generator
        @Assert: An integer number is not created due to value error
        """

        with self.assertRaises(ValueError):
            self.factory.generate_integer(max_value='')

    def test_generate_integer_7_2(self):
        """
        @Test: Create a random integer using empty strings as args
        @Feature: Numbers Generator
        @Assert: An integer number is not created due to value error
        """

        with self.assertRaises(ValueError):
            self.factory.generate_integer(min_value='', max_value='')

    def test_generate_integer_8_0(self):
        """
        @Test: Create a random integer using whitespace as args
        @Feature: Numbers Generator
        @Assert: An integer number is not created due to value error
        """

        with self.assertRaises(ValueError):
            self.factory.generate_integer(min_value=' ')

    def test_generate_integer_8_1(self):
        """
        @Test: Create a random integer using whitespace as args
        @Feature: Numbers Generator
        @Assert: An integer number is not created due to value error
        """

        with self.assertRaises(ValueError):
            self.factory.generate_integer(max_value=' ')

    def test_generate_integer_8_2(self):
        """
        @Test: Create a random integer using whitespace as args
        @Feature: Numbers Generator
        @Assert: An integer number is not created due to value error
        """

        with self.assertRaises(ValueError):
            self.factory.generate_integer(min_value=' ', max_value=' ')

    def test_generate_integer_9_0(self):
        """
        @Test: Create a random integer using alpha strings as args
        @Feature: Numbers Generator
        @Assert: An integer number is not created due to value error
        """

        with self.assertRaises(ValueError):
            self.factory.generate_integer(min_value='a')

    def test_generate_integer_9_1(self):
        """
        @Test: Create a random integer using alpha strings as args
        @Feature: Numbers Generator
        @Assert: An integer number is not created due to value error
        """

        with self.assertRaises(ValueError):
            self.factory.generate_integer(max_value='a')

    def test_generate_integer_9_2(self):
        """
        @Test: Create a random integer using alpha strings as args
        @Feature: Numbers Generator
        @Assert: An integer number is not created due to value error
        """

        with self.assertRaises(ValueError):
            self.factory.generate_integer(min_value='a', max_value='b')

    def test_generate_positive_integer_1(self):
        """
        @Test: Create a random positive integer
        @Feature: Numbers Generator
        @Assert: A positive number is created
        """

        result = self.factory.generate_positive_integer()

        self.assertTrue(result >= 0, "Generated integer is not positive")

    def test_generate_negative_integer_1(self):
        """
        @Test: Create a random negative integer
        @Feature: Numbers Generator
        @Assert: A negative number is created
        """

        result = self.factory.generate_negative_integer()

        self.assertTrue(result <= 0, "Generated integer is not negative")
