from ..configfile import ConfigFile
import gocept.net.ceph.cluster
import gocept.net.ceph.pools
import gocept.net.directory
import gocept.net.xmlrpc
import logging
import logging.handlers
import os
import os.path
import socket
import subprocess
import sys
import telnetlib
import tempfile


PROGNAME = os.path.basename(sys.argv[0])
KVM_HOST = socket.gethostname()
VERBOSE = os.environ.get('VERBOSE', False)

# A buffer to receive output from stdout and sterr of subprocesses
out_buf = tempfile.TemporaryFile()

logger = logging.getLogger(__name__)


def call(*cmd):
    out_buf.seek(0)
    out_buf.truncate(0)
    if VERBOSE:
        print('calling {}'.format(cmd))
    try:
        subprocess.check_call(
            cmd, stdout=out_buf, stderr=subprocess.STDOUT)
    except subprocess.CalledProcessError, e:
        out_buf.seek(0)
        sys.stdout.write(out_buf.read())
        sys.exit(e.returncode)


class VM(object):

    root = ''
    pidfile = '{root}/run/kvm.{name}.pid'
    confdfile = '{root}/etc/conf.d/kvm.{name}'
    configfile = '{root}/etc/kvm/{name}.cfg'
    optfile = '{root}/etc/kvm/{name}.opt'
    initfile = '{root}/etc/init.d/kvm.{name}'
    runlevelfile = '{root}/etc/runlevels/default/kvm.{name}'
    chroot = '{root}/srv/vm/{name}'

    def __init__(self, enc, this_kvm_host=KVM_HOST):
        self.name = enc['name']
        self.enc = enc
        self.parameters = enc['parameters']
        self.parameters['monitor_port'] = self.parameters['id'] + 20000
        self.this_kvm_host = this_kvm_host

        for attr in ['pidfile', 'confdfile', 'initfile', 'runlevelfile',
                     'optfile', 'configfile', 'chroot']:
            setattr(self, attr, getattr(self, attr).format(
                root=self.root,**enc))

    def ensure(self):
        self.write_confd_file()
        self.write_config_file()
        self.write_opt_file()
        self.ensure_init_script_exists()

        # The VM should be offline
        if not self.parameters['online']:
            self.ensure_not_startup_config()
            self.ensure_not_running()
            return

        # The VM should be online...
        # ... but not *here*, so we outmigrate it.
        if self.parameters['kvm_host'] != self.this_kvm_host:
            self.ensure_not_startup_config()
            self.outmigrate()
            return

        # ... should be online here.
        self.ensure_startup_config()
        self.inmigrate()

    def outmigrate(self):
        if not self.is_running():
            return
        subprocess.check_call(
            ['/usr/bin/fc-livemig', self.name, 'outgoing',
             self.parameters['kvm_host']])

    def inmigrate(self):
        if self.is_running():
            return
        subprocess.check_call(
            ['/usr/bin/fc-livemig', self.name, 'incoming',
             self.this_kvm_host])

    def write_confd_file(self):
        confd = ConfigFile(self.confdfile)
        confd.write("""\
# configure-kvm: do not edit this file directly. It will be overwritten!
VMRG="{resource_group}"
VMDISK="{disk}"
CHROOT="{chroot}"
# picked up here by live migration ... o_O
MONITOR_PORT="{monitor_port}"
""".format(chroot=self.chroot, **self.parameters))

        confd.commit()

    def write_opt_file(self):
        # Careful: changing anything in those files will cause maintenance w/
        # reboot of all VMs in the infrastructure.
        config = ConfigFile(self.optfile)
        config.write("""\
# qemu command line argument file
# generated by localconfig. do not edit.
KVM_OPTIONS="\
      -daemonize -pidfile {pidfile}\
      -nodefaults\
      -chroot {chroot}\
      -runas nobody\
      -serial file:/var/log/vm/{name}.log\
      -display vnc=${{HOSTNAME}}.mgm.${{SUFFIX3}}:{id}\
      -vga cirrus\
      -m {memory}\
      -watchdog i6300esb\
      -watchdog-action reset\
      -readconfig /run/kvm.{name}.cfg\
"
SWAPSIZE="{swap}"
TMPSIZE="{tmp}"
""".format(chroot=self.chroot, pidfile=self.pidfile, name=self.name,
           swap=self.swapsize(), tmp=self.tmpsize(),
           configfile=self.configfile,
           **self.parameters))
        config.commit()

    def swapsize(self):
        if self.parameters['memory'] > 2048:
            return self.parameters['memory'] / 2
        if self.parameters['memory'] < 768:
            return self.parameters['memory'] + 256
        return 1024

    def tmpsize(self):
        if self.parameters['disk'] < 50:
            return 5120
        return self.parameters['disk'] * 1024 / 10


    def write_config_file(self):
        # Careful: changing anything in those files will cause maintenance w/
        # reboot of all VMs in the infrastructure.
        config = ConfigFile(self.configfile)
        config.write("""\
# qemu config file
# generated by localconfig. do not edit.

[machine]
  accel = "kvm"
  type = "pc-q35-2.1"

[smp-opts]
  cpus = "{cores}"

[name]
  guest = "{name}"
  process = "kvm.{name}"

[drive]
  index = "0"
  media = "disk"
  if = "virtio"
  format = "rbd"
  file = "rbd:{resource_group}/{name}.root:id=${{HOSTNAME}}"
  aio = "native"
  cache = "writeback"

[drive]
  index = "1"
  media = "disk"
  if = "virtio"
  format = "rbd"
  file = "rbd:{resource_group}/{name}.swap:id=${{HOSTNAME}}"
  aio = "native"
  cache = "writeback"

[drive]
  index = "2"
  media = "disk"
  if = "virtio"
  format = "rbd"
  file = "rbd:{resource_group}/{name}.tmp:id=${{HOSTNAME}}"
  aio = "native"
  cache = "writeback"

[device]
  driver = "virtio-rng-pci"

# Guest agent support
[device]
  driver = "virtio-serial"

[device]
  driver = "virtserialport"
  chardev = "qga0"
  name = "org.qemu.guest_agent.0"

[chardev "qga0"]
  backend = "socket"
  path = "/run/kvm.{name}.gqa.sock"
  server = "on"
  wait = "off"

# QMP monitor support via Unix socket

[mon "qmp_monitor"]
  mode = "control"
  chardev = "ch_qmp_monitor"
  default = "on"

[chardev "ch_qmp_monitor"]
  backend = "socket"
  path = "/run/kvm.{name}.qmp.sock"
  server = "on"
  wait = "off"

# Human monitor support via Unix socket

[chardev "ch_readline_socket_monitor"]
  backend = "socket"
  path = "/run/kvm.{name}.monitor.sock"
  server = "on"
  wait = "off"

[mon "readline_socket_monitor"]
  mode = "readline"
  chardev = "ch_readline_socket_monitor"

# Human monitor support via localhost IP

[chardev "ch_readline_telnet_monitor"]
  backend = "socket"
  host = "localhost"
  port = "{monitor_port}"
  server = "on"
  wait = "off"

[mon "readline_telnet_monitor"]
  mode = "readline"
  chardev = "ch_readline_telnet_monitor"

# Network interfaces

""".format(name=self.name, **self.parameters))

        for net, net_config in sorted(self.parameters['interfaces'].items()):
            config.write("""\
[device]
  driver = "virtio-net-pci"
  netdev = "{ifname}"
  mac = "{mac}"

[netdev "{ifname}"]
  type = "tap"
  ifname = "{ifname}"
  script = "/etc/kvm/kvm-ifup"
  downscript = "/etc/kvm/kvm-ifdown"
  vhost = "on"

""".format(ifname='t{}{}'.format(net, self.parameters['id']),
           mac=net_config['mac']))

        config.commit()

    def ensure_init_script_exists(self):
        if os.path.exists(self.initfile):
            return
        if VERBOSE:
            print('creating init script for {}'.format(self.name))
        try:
            # get rid of dangling symlinks
            os.unlink(self.initfile)
        except OSError:
            pass
        os.symlink('kvm', self.initfile)

    def ensure_startup_config(self):
        # STARTUP CONFIG
        if not os.path.exists(self.runlevelfile):
            if VERBOSE:
                print "registering init script for {}".format(self.name)
            call('rc-update', 'add', 'kvm.{}'.format(self.name), 'default')

    def ensure_not_startup_config(self):
        # STARTUP CONFIG
        if os.path.exists(self.runlevelfile):
            if VERBOSE:
                print "un-registering init script for {}".format(self.name)
            call('rc-update', 'del', 'kvm.{}'.format(self.name), 'default')

    def ensure_not_running(self):
        # RUNTIME ENVIRONMENT
        if self.is_running():
            if VERBOSE:
                print('Stopping {}'.format(self.name))
                sys.stdout.flush()
            try:
                self.init('stop')
            except subprocess.CalledProcessError:
                self.init('zap')
                raise

    def init(self, cmd):
        subprocess.check_call([self.initfile, cmd])

    def is_running(self):
        if not os.path.exists(self.pidfile):
            return False
        try:
            with open(self.pidfile) as f:
                pid = int(f.read())
        except ValueError:
            return False
        return pid > 1 and os.path.exists('/proc/%s' % pid)

    def sorted_networks(self):
        """Return list network names in the right order.

        The 'srv' network is always listed first to ensure that it gets
        mapped to eth0 during bootstrap. It is an error if no srv address is
        configured.
        """
        nets = self.parameters['interfaces'].keys()
        nets.remove('srv')
        return ['srv'] + nets


def ensure_vms():
    exitcode = 0
    directory = gocept.net.directory.Directory()
    location = os.environ['PUPPET_LOCATION']

    with gocept.net.directory.exceptions_screened():
        vms = directory.list_virtual_machines(location)

    for vm_data in vms:
        vm = VM(vm_data)
        try:
            vm.ensure()
        except Exception:
            raise
            exitcode = 1
    sys.exit(exitcode)


def block_resize(vm):
    """Trigger block resize action via Qemu monitor."""
    print('{}: resizing disk for VM {} to {} GiB'.format(
        PROGNAME, vm['name'], vm['parameters']['disk']))
    try:
        monitor_port = int(vm['parameters']['id']) + 20000
        kvm_monitor = telnetlib.Telnet('localhost', monitor_port)
        kvm_monitor.read_until('(qemu) ')
        disk_mib = vm['parameters']['disk'] * 1024
        kvm_monitor.write('block_resize virtio0 {}\n'.format(disk_mib))
        output = kvm_monitor.read_until('(qemu) ')
        if VERBOSE:
            print(output)
        kvm_monitor.close()
    except socket.error as e:
        print('{}: Error communicating with KVM instance {}. Offline?'.format(
            PROGNAME, vm['name']))
        print(e)
        return


def resize_disks():
    directory = gocept.net.directory.Directory()
    location = os.environ['PUPPET_LOCATION']
    cluster = gocept.net.ceph.cluster.Cluster()
    pools = gocept.net.ceph.pools.Pools(cluster)

    with gocept.net.directory.exceptions_screened():
        vms = directory.list_virtual_machines(location, KVM_HOST)

    for vm in vms:
        try:
            image = pools[vm['parameters']['resource_group']].get(
                vm['name'] + '.root')
        except (KeyError, RuntimeError):
            if VERBOSE:
                print('{}: Could not get Ceph volume info for {}. '.format(
                    PROGNAME, vm['name']))
            continue
        if (image.size_gb >= vm['parameters']['disk']):
            # large enough
            continue
        block_resize(vm)
