from gocept.net.configure.kvm import ensure_vms, VM
import mock
import os
import pytest
import shutil
import tempfile
import unittest


@pytest.fixture(autouse=True)
def prefix_kvm_paths(tmpdir, monkeypatch):
    os.mkdir(str(tmpdir/'etc'))
    os.mkdir(str(tmpdir/'etc/kvm'))
    os.mkdir(str(tmpdir/'run'))
    monkeypatch.setattr(VM, 'root',  str(tmpdir))


def make_vm(**kw):
    result = {'name': 'test00',
              'parameters': {'id': 1000,
                             'online': False,
                             'memory': 512,
                             'cores': 1,
                             'resource_group': 'test',
                             'disk': 10,
                             'interfaces': {'srv': {'mac': 'foo'}}}}
    result['parameters'].update(kw)
    return result


class KVMConfigTest(unittest.TestCase):

    def setUp(self):
        os.environ['PUPPET_LOCATION'] = 'dev'
        self.p_directory = mock.patch('gocept.net.directory.Directory')
        self.fake_directory = self.p_directory.start()

    def tearDown(self):
        del os.environ['PUPPET_LOCATION']
        self.p_directory.stop()

    @mock.patch('gocept.net.configure.kvm.VM.ensure')
    def test_no_actions_on_empty_list(self, ensure):
        with self.assertRaises(SystemExit) as e:
            ensure_vms()
        assert e.exception.code == 0
        self.assertEquals(ensure.call_count, 0)

    @mock.patch('gocept.net.configure.kvm.VM.ensure')
    def test_listed_vms_get_ensure_called(self, ensure):
        self.fake_directory().list_virtual_machines.return_value = [
            make_vm()]
        with self.assertRaises(SystemExit) as e:
            ensure_vms()
        assert e.exception.code == 0
        self.assertEquals(ensure.call_count, 1)


class VMTest(unittest.TestCase):

    def setUp(self):
        self.tmpdir = tempfile.mkdtemp()
        open(self.tmpdir+'/kvm', 'w').close()
        VM.confdfile = '{}/confd.{{name}}'.format(self.tmpdir)
        VM.initfile = '{}/init.{{name}}'.format(self.tmpdir)
        VM.runlevelfile = '{}/runlevel.default.{{name}}'.format(self.tmpdir)

        self.p_call = mock.patch('gocept.net.configure.kvm.call')
        self.call = self.p_call.start()

        self.p_subpcall = mock.patch('subprocess.check_call')
        self.subpcall = self.p_subpcall.start()

    def tearDown(self):
        shutil.rmtree(self.tmpdir)
        self.p_call.stop()
        self.p_subpcall.stop()

    def test_vm_not_running_but_should_be(self):
        vm = VM(make_vm(online=True, kvm_host='foo'), this_kvm_host='foo')
        vm.ensure()
        assert """\
# configure-kvm: do not edit this file directly. It will be overwritten!
VMRG="test"
VMDISK="10"
CHROOT="{}/srv/vm/test00"
# picked up here by live migration ... o_O
MONITOR_PORT="21000"
""".format(VM.root) == open(vm.confdfile, 'r').read()
        assert 'kvm' == os.readlink(vm.initfile)

        assert len(self.subpcall.mock_calls) == 1
        assert (self.subpcall.mock_calls[0] ==
                mock.call(['/usr/bin/fc-livemig', 'test00', 'incoming', 'foo']))

        assert len(self.call.mock_calls) == 1
        assert (self.call.mock_calls[0] ==
                mock.call('rc-update', 'add', 'kvm.test00', 'default'))

    def test_vm_running_and_should_be(self):
        vm = VM(make_vm(online=True, kvm_host='foo'), this_kvm_host='foo')
        vm.is_running = lambda: True
        vm.ensure()
        assert """\
# configure-kvm: do not edit this file directly. It will be overwritten!
VMRG="test"
VMDISK="10"
CHROOT="{}/srv/vm/test00"
# picked up here by live migration ... o_O
MONITOR_PORT="21000"
""".format(VM.root) == open(vm.confdfile, 'r').read()
        assert os.readlink(vm.initfile) == 'kvm'

        assert len(self.subpcall.mock_calls) == 0

        assert len(self.call.mock_calls) == 1
        assert (self.call.mock_calls[0] ==
                mock.call('rc-update', 'add', 'kvm.test00', 'default'))

    def test_vm_running_but_should_not_be(self):
        vm = VM(make_vm(online=False, kvm_host='foo'), this_kvm_host='foo')
        open(vm.runlevelfile, 'w').close()
        vm.is_running = lambda: True
        vm.ensure()
        assert """\
# configure-kvm: do not edit this file directly. It will be overwritten!
VMRG="test"
VMDISK="10"
CHROOT="{}/srv/vm/test00"
# picked up here by live migration ... o_O
MONITOR_PORT="21000"
""".format(VM.root) == open(vm.confdfile, 'r').read()
        assert os.readlink(vm.initfile) == 'kvm'

        assert len(self.subpcall.mock_calls) == 1
        assert (self.subpcall.mock_calls[0] ==
                mock.call([vm.initfile, 'stop']))

        assert len(self.call.mock_calls) == 1
        assert (self.call.mock_calls[0] ==
                mock.call('rc-update', 'del', 'kvm.test00', 'default'))

    def test_vm_stopped_and_should_be(self):
        vm = VM(make_vm(online=False, kvm_host='foo'), this_kvm_host='foo')
        vm.is_running = lambda: False
        vm.ensure()
        assert """\
# configure-kvm: do not edit this file directly. It will be overwritten!
VMRG="test"
VMDISK="10"
CHROOT="{}/srv/vm/test00"
# picked up here by live migration ... o_O
MONITOR_PORT="21000"
""".format(VM.root) == open(vm.confdfile, 'r').read()

        assert """\
# qemu command line argument file
# generated by localconfig. do not edit.
KVM_OPTIONS="\
      -daemonize -pidfile {pid}\
      -nodefaults\
      -chroot {chroot}\
      -runas nobody\
      -serial file:/var/log/vm/test00.log\
      -display vnc=${{HOSTNAME}}.mgm.${{SUFFIX3}}:1000\
      -vga cirrus\
      -m 512\
      -watchdog i6300esb\
      -watchdog-action reset\
      -readconfig /run/kvm.test00.cfg"
SWAPSIZE="768"
TMPSIZE="5120"
""".format(chroot=vm.chroot, pid=vm.pidfile) == open(vm.optfile, 'r').read()

        assert os.readlink(vm.initfile) == 'kvm'

        assert len(self.subpcall.mock_calls) == 0
        assert len(self.call.mock_calls) == 0



def test_vm_swapsize():
    vm = VM(make_vm())
    vm.parameters['memory'] = 256
    assert vm.swapsize() == 512
    vm.parameters['memory'] = 512
    assert vm.swapsize() == 768
    vm.parameters['memory'] = 768
    assert vm.swapsize() == 1024
    vm.parameters['memory'] = 1024
    assert vm.swapsize() == 1024
    vm.parameters['memory'] = 2048
    assert vm.swapsize() == 1024
    vm.parameters['memory'] = 4096
    assert vm.swapsize() == 2048


def test_vm_tmpsize():
    vm = VM(make_vm())
    vm.parameters['disk'] = 5
    assert vm.tmpsize() == 5120
    vm.parameters['disk'] = 10
    assert vm.tmpsize() == 5120
    vm.parameters['disk'] = 50
    assert vm.tmpsize() == 5120
    vm.parameters['disk'] = 100
    assert vm.tmpsize() == 10240
    vm.parameters['disk'] = 200
    assert vm.tmpsize() == 20480
