# Copyright (c) gocept gmbh & co. kg
# See also LICENSE.txt

"""Configure specific Postfix transports tied to service users.

For each service user, a new pipe transport named `user-{USER}` is
created. The user-specific transport is invoked with the command line
`$HOME/bin/deliver -f {ENVELOPE_FROM} {NEXTHOP} {RECIPIENT}
[{RECIPIENT}...]`. These parameters are the same as other transports'
parameters (think bsmtp etc.).
"""

from __future__ import unicode_literals, print_function
from gocept.net.configfile import ConfigFile
import argparse
import grp
import pwd
import subprocess
import sys


class MasterCf(object):
    """Append user-specific transport to an initial master.cf file.

    The main body of master.cf is expected to reside in `infile`. We
    append a transport line for each service user found in the passwd
    database.

    To restart Postfix, call the `reload()` method.
    """

    SERVICE_GID = grp.getgrnam('service').gr_gid

    def __init__(self, infile, outfile):
        self.infile = infile
        self.outfile = ConfigFile(outfile)
        self.changed = False

    def list_serviceusers(self):
        for pwent in pwd.getpwall():
            if pwent.pw_gid == self.SERVICE_GID:
                yield pwent

    def user_transport(self, username, home):
        """Create a single per-user transport for master.cf."""
        return """\
user-{user} unix - n n - - pipe
  flags=Ruhq user={user} directory={home}
  argv={home}/bin/deliver -f $sender $nexthop $recipient
""".format(user=username, home=home)

    def update(self):
        with open(self.infile) as f:
            print("# Auto-generated by {}. Don't edit!".format(
                sys.argv[0]), file=self.outfile)
            self.outfile.write(f.read())
        for user in self.list_serviceusers():
            self.outfile.write(self.user_transport(user.pw_name, user.pw_dir))
        self.changed = self.outfile.commit()

    def restart(self, cmd):
        if not self.changed:
            return
        subprocess.check_call([cmd], shell=True)


def master():
    """Main entry point for master.cf update."""
    argp = argparse.ArgumentParser(description=__doc__)
    argp.add_argument('-r', '--restart', metavar='CMD',
                      default='/etc/init.d/postfix restart',
                      help='command to restart postfix (default: %(default)s)')
    argp.add_argument('INFILE', help='static master.cf main body')
    argp.add_argument('OUTFILE', help='generated master.cf location')
    args = argp.parse_args()
    m = MasterCf(args.INFILE, args.OUTFILE)
    m.update()
    if args.restart:
        m.restart(args.restart)
