from abc import ABCMeta, abstractmethod
import beanstalkc
import json
from sys import exc_info
import logging


RESULT_OK = 101
RESULT_CORRUPT = 102
RESULT_MUMBLE = 103
RESULT_DOWN = 104
RESULT_CHECKER_ERROR = 110


logging.basicConfig(level=logging.INFO,
                    format='[%(asctime)s] - %(levelname)s - %(message)s')
logger = logging.getLogger(__name__)


class FemidaChecker(object):
    """Checker interface. You should implement _push and _pull methods"""
    __metaclass__ = ABCMeta

    def push(self, endpoint, flag_id, flag):
        result = (RESULT_CHECKER_ERROR, flag_id)
        try:
            result = self._push(endpoint, flag_id, flag)
        except:
            logger.exception('An exception occurred', exc_info=exc_info())
        return result

    def pull(self, endpoint, flag_id, flag):
        result = RESULT_CHECKER_ERROR
        try:
            result = self._pull(endpoint, flag_id, flag)
        except:
            logger.exception('An exception occurred', exc_info=exc_info())
        return result

    @abstractmethod
    def _push(self, endpoint, flag_id, flag):
        """Push <flag> into the <endpoint> service. Return tuple containing
        result from one of RESULT_* constants and flag_id"""
        pass

    @abstractmethod
    def _pull(self, endpoint, flag_id, flag):
        """Check if the flag that can be pulled from <endpoint> service by
        using some data in <flag_id> equals given <flag>. Return result from
        one of RESULT_* constants"""
        pass

    def run(self):
        """This method will be final. Don't override it"""
        beanstalk = beanstalkc.Connection(host='127.0.01', port=11300)
        beanstalk.watch('push')

        sink = beanstalkc.Connection(host='127.0.0.1', port=11300)
        sink.use('result')
        while True:
            job = beanstalk.reserve()
            try:
                data = json.loads(job.body)
                logger.info('Received job with body {0}'.format(
                    json.dumps(data)))
                if data['operation'] == 'push':
                    res = self.push(data['endpoint'], data['flag_id'],
                                    data['flag'])
                elif data['operation'] == 'pull':
                    res = self.pull(data['endpoint'], data['flag_id'],
                                    data['flag'])
                logger.info('Finished job with result {0}'.format(
                    json.dumps(res)))
                sink.put(json.dumps(res))
            except:
                logger.exception('An exception occurred', exc_info=exc_info())
            job.delete()
        sink.close()
        beanstalk.close()
