from abc import ABCMeta, abstractmethod
import beanstalkc
import json
from sys import exc_info, stdout
import logging
from os import getenv
from enum import Enum


class Result(Enum):
    ok = 101
    corrupt = 102
    mumble = 103
    down = 104
    checker_error = 110


class Checker(object):
    """Checker interface. You should implement _push and _pull methods"""
    __metaclass__ = ABCMeta

    def __init__(self):
        self.setup_logging()

    def setup_logging(self):
        console_handler = logging.StreamHandler(stdout)
        log_format = '[%(asctime)s] - %(levelname)s - %(message)s'
        formatter = logging.Formatter(log_format)
        console_handler.setFormatter(formatter)
        self.logger = logging.Logger(__name__)
        self.logger.setLevel(logging.DEBUG)
        self.logger.addHandler(console_handler)

    def push(self, endpoint, flag_id, flag):
        result, new_flag_id = Result.checker_error, flag_id
        try:
            result, new_flag_id = self._push(endpoint, flag_id, flag)
        except:
            self.logger.exception('An exception occurred', exc_info=exc_info())
        return result.value, new_flag_id

    def pull(self, endpoint, flag_id, flag):
        result = Result.checker_error
        try:
            result = self._pull(endpoint, flag_id, flag)
        except:
            self.logger.exception('An exception occurred', exc_info=exc_info())
        return result.value

    @abstractmethod
    def _push(self, endpoint, flag_id, flag):
        """Push <flag> into the <endpoint> service. Return tuple containing
        result from one of RESULT_* constants and flag_id"""
        pass

    @abstractmethod
    def _pull(self, endpoint, flag_id, flag):
        """Check if the flag that can be pulled from <endpoint> service by
        using some data in <flag_id> equals given <flag>. Return result from
        one of RESULT_* constants"""
        pass

    def run(self):
        """This method will be final. Don't override it"""
        host, port = getenv('BEANSTALKD_URI').split(':')
        beanstalk = beanstalkc.Connection(host=host, port=int(port))
        self.logger.info('Established connection with beanstalk server')

        beanstalk.watch(getenv('TUBE_PUSH'))
        beanstalk.watch(getenv('TUBE_PULL'))

        while True:
            job = beanstalk.reserve()
            try:
                data = json.loads(job.body)
                tube = job.stats()['tube']
                service_id, operation = tube.split('.')
                if operation == 'push':
                    status, flag_id = self.push(data['endpoint'],
                                                data['flag_id'],
                                                data['flag'])
                    res = dict(status=status,
                               flag=data['flag'],
                               flag_id=flag_id,
                               endpoint=data['endpoint'])
                    s = 'PUSH flag {0} to {1}: result {2}, flag_id {3}'
                    self.logger.info(s.format(data['flag'],
                                              data['endpoint'],
                                              res['status'],
                                              res['flag_id']))
                    beanstalk.use(getenv('TUBE_REPORT_PUSH'))
                    beanstalk.put(json.dumps(res))
                elif operation == 'pull':
                    status = self.pull(data['endpoint'],
                                       data['flag_id'],
                                       data['flag'])
                    res = dict(request_id=data['request_id'],
                               status=status)
                    s = 'PULL flag {0} from {1} with flag_id {2}: result {3}'
                    self.logger.info(s.format(data['flag'],
                                              data['endpoint'],
                                              data['flag_id'],
                                              res['status']))
                    beanstalk.use(getenv('TUBE_REPORT_PULL'))
                    beanstalk.put(json.dumps(res))
            except KeyboardInterrupt:
                beanstalk.close()
                self.logger.info('Closed connection to beanstalk server')
                break
            except:
                self.logger.exception('An exception occurred',
                                      exc_info=exc_info())
            finally:
                job.delete()
