from sys import exc_info
import warnings

from talljosh import Function
from twisted.internet import defer, reactor
from twisted.python import failure
from twisted.python.util import mergeFunctionMetadata

def background(value):
    '''
    Converts the given into a deferred, whether it is a generator, a deferred,
    or another value.
    '''
    if is_runnable_generator(value):
        value = run_generator(value)
    if not isinstance(value, defer.Deferred):
        value = defer.succeed(value)
    return value

def async(fn):
    '''
    Marks this function as being potentially asynchronous. If the function uses
    "yield", the function will be decorated as having inline callbacks.

    An advantage of marking a function as @async rather than as @inlineCallbacks
    is that removing the last yield won't break everything horribly.
    '''
    def wrapper(*args, **kwargs):
        result = fn(*args, **kwargs)
        if is_runnable_generator(result):
            result = run_generator(result)
        return result
    return mergeFunctionMetadata(fn, wrapper)

def true_generator(fn):
    '''
    Explicitly marks this generator function as a function whose generators
    should not be run.

    Useful if for some reason you want to return a generator from a function
    whose result would normally be waited for. (Alternatively you could wrap the
    result in defer.Succeed before returning).
    '''
    def wrapper(*args, **kwargs):
        g = fn(*args, **kwargs)
        g._fibre_do_not_run = True
        return g
    return mergeFunctionMetadata(fn, wrapper)

def is_runnable_generator(g):
    return hasattr(g, 'send') and not getattr(g, '_fibre_do_not_run', False)

def wait(t):
    '''
    Waits for t seconds then fires the callback.
    '''
    def cancel(d):
        call.cancel()

    d = defer.Deferred(canceller=cancel)
    call = reactor.callLater(t, d.callback, None)
    return d

class first(Function):
    '''
    Succeeds with (task, result) when the first of the given deferreds calls
    back, fails with the first errback.
    '''
    def run(self, *tasks):
        self.deferred = defer.deferred(canceller=self.cancel)

        for task in tasks:
            task.addCallbacks(self.taskSucceeded, self.taskFailed, callbackArgs=[task])
        return self.deferred()

    def taskSucceeded(self, task, result):
        if self.deferred.called:
            return
        self.deferred.callback((task, result))

    def taskFailed(self, reason):
        if self.deferred.called:
            return
        self.deferred.errback(reason)

class run_generator(Function):
    '''
    Based on twisted.internet.defer.inlineCallbacks.

    Has the additional consequence that if the generator yields a generator, it
    will also be treated as an inline callback unless it's decorated with
    @true_generator.
    '''

    def run(self, g):
        self.g = g
        self.deferred = defer.Deferred(canceller=self.cancel)
        self.waiting = True
        self.result = None
        self.pending_deferred = None
        return self.proceed()

    def cancel(self, d=None):
        '''
        Called by self.deferred.cancel().
        '''
        if self.deferred.called:
            return

        if self.pending_deferred:
            self.pending_deferred.cancel()
        else:
            # Must be called from within the generator itself.
            raise defer.CancelledError()

    def proceed(self, result=None):
        # This function is complicated by the need to prevent unbounded
        # recursion arising from repeatedly yielding immediately ready
        # deferreds. This while loop and the waiting variable solve that by
        # manually unfolding the recursion.

        g = self.g
        while 1:
            try:
                # Send the last result back as the result of the yield expression.
                isFailure = isinstance(result, failure.Failure)
                if isFailure:
                    result = result.throwExceptionIntoGenerator(g)
                else:
                    result = g.send(result)

            except StopIteration:
                # fell off the end, or "return" statement
                self.deferred.callback(None)
                return self.deferred

            except defer._DefGen_Return, e:
                # returnValue() was called
                self._check_for_returnValue_at_incorrect_level(isFailure)
                self.deferred.callback(e.value)
                return self.deferred

            except:
                self.deferred.errback()
                return self.deferred

            if is_runnable_generator(result):
                # a generator function was yielded, run it.
                result = run_generator(result)

            if isinstance(result, defer.Deferred):
                # a deferred was yielded, get the result.
                self.pending_deferred = result
                result.addBoth(self.gotResult)
                if self.waiting:
                    # Haven't called back yet, set flag so that we get reinvoked
                    # and return from the loop
                    self.waiting = False
                    return self.deferred

                result = self.result
                # Reset waiting to initial values for next loop.  gotResult uses
                # waiting, but this isn't a problem because gotResult is only
                # executed once, and if it hasn't been executed yet, the return
                # branch above would have been taken.

                self.waiting = True
                self.result = None

    def gotResult(self, r):
        self.pending_deferred = None
        if self.waiting:
            self.waiting = False
            self.result = r
        else:
            self.waiting = True
            self.result = None
            self.proceed(r)

    def _check_for_returnValue_at_incorrect_level(self, isFailure):
            # returnValue() was called; time to give a result to the original
            # Deferred.  First though, let's try to identify the potentially
            # confusing situation which results when returnValue() is
            # accidentally invoked from a different function, one that wasn't
            # decorated with @inlineCallbacks.

            # The traceback starts in this frame (the one for self.proceed); the
            # next one down should be the application code.
            appCodeTrace = exc_info()[2].tb_next
            if isFailure:
                # If we invoked this generator frame by throwing an exception
                # into it, then throwExceptionIntoGenerator will consume an
                # additional stack frame itself, so we need to skip that too.
                appCodeTrace = appCodeTrace.tb_next
            # Now that we've identified the frame being exited by the
            # exception, let's figure out if returnValue was called from it
            # directly.  returnValue itself consumes a stack frame, so the
            # application code will have a tb_next, but it will *not* have a
            # second tb_next.
            if appCodeTrace.tb_next.tb_next:
                # If returnValue was invoked non-local to the frame which it is
                # exiting, identify the frame that ultimately invoked
                # returnValue so that we can warn the user, as this behavior is
                # confusing.
                ultimateTrace = appCodeTrace
                while ultimateTrace.tb_next.tb_next:
                    ultimateTrace = ultimateTrace.tb_next
                filename = ultimateTrace.tb_frame.f_code.co_filename
                lineno = ultimateTrace.tb_lineno
                warnings.warn_explicit(
                    "returnValue() in %r causing %r to exit: "
                    "returnValue should only be invoked by functions decorated "
                    "with inlineCallbacks" % (
                        ultimateTrace.tb_frame.f_code.co_name,
                        appCodeTrace.tb_frame.f_code.co_name),
                    DeprecationWarning, filename, lineno)
