import logging

import talljosh
from twisted.internet import reactor

from fibre.asynchronous import background

DEFAULT_LOG_FORMAT = '%(asctime)s %(levelname)s %(filename)s:%(funcName)s %(message)s'

class run_application(talljosh.Function):
    init_logging = False
    logging_args = None

    def run(self):
        self.initialise()
        d = background(self.fn())
        self.loop(d)
        self.finalise()

    def initialise(self):
        if self.init_logging:
            initialise_logging(self.logging_args)

    def loop(self, d):
        @d.addCallback
        def done(result):
            reactor.callWhenRunning(reactor.stop)
        @d.addErrback
        def error(reason):
            if not reason.check(SystemExit):
                reason.printTraceback()
            reactor.callWhenRunning(reactor.stop)

        reactor.run()

    def finalise(self):
        pass

class entry_point(talljosh.Function):
    '''
    If the module in which entry_point() is called is being executed rather than
    imported (i.e. __name__ == '__main__'), a call to this function will call
    the fn parameter and run the reactor until the call is complete.
    '''
    main_function = staticmethod(run_application)
    def run(self, fn, stack_levels=1, init_logging=None, logging_args=None):
        class main_function(self.main_function):
            pass
        main_function.fn = staticmethod(fn)
        if init_logging is not None:
            main_function.init_logging = init_logging
        if logging_args is not None:
            main_function.logging_args = logging_args
        talljosh.entry_point(main_function, stack_levels=stack_levels+2)

def initialise_logging(args):
    '''
    Sets up python logging and redirects twisted logging there too.
    '''
    if not args:
        args = {}
    args.setdefault('format', DEFAULT_LOG_FORMAT)

    import twisted.python.log
    logging.basicConfig(**args)
    observer = twisted.python.log.PythonLoggingObserver()
    twisted.python.log.startLoggingWithObserver(observer.emit)
