from __future__ import print_function

import os
import sys
import termios

from twisted.internet import protocol, stdio, defer
from twisted.internet.defer import returnValue
from twisted.protocols import basic

_line_reader = None
def read_raw_line():
    '''
    Waits until a line has been entered on stdin and returns the line as a
    raw byte string.
    '''
    global _line_reader
    if not _line_reader:
        _line_reader = LineReader()
        stdio.StandardIO(_line_reader)

    d = defer.Deferred()
    _line_reader.attach(d)

    return d

def read_line(prompt=None, encoding='utf-8'):
    '''
    Waits until a line has been entered on stdin and returns the line as a
    unicode string.

    @param excoding: the encoding of the input stream.
    '''
    if prompt:
        print(prompt, end='')
        sys.stdout.flush()

    line = yield read_raw_line()
    returnValue(line.decode(encoding))

_char_reader = None
def read_character(encoding='utf-8'):
    '''
    Waits until a unicode character has been entered on stdin and returns that
    value.

    Note that there is only ever one default UnicodeCharReader on stdin, and if
    there are multiple calls to read_character() operating at the same time, the
    encoding of the most recent call is used.

    If you ever have a reason to have stdin being interpretted as two different
    encodings at the same time, you must manually construct separate
    UnicodeCharReaders for each case.
    '''
    global _char_reader
    if not _char_reader:
        _char_reader = UnicodeCharReader()
        stdio.StandardIO(_char_reader)

    d = defer.Deferred()
    _char_reader.attach(d)
    _char_reader.encoding = encoding

    return d

class LineReader(basic.LineReceiver):
    '''
    Allows callbacks to be attached, fires all callbacks when a line is
    received.
    '''
    delimiter = os.linesep

    def __init__(self):
        self.deferreds = []

    def attach(self, d):
        self.deferreds.append(d)

    def lineReceived(self, line):
        deferreds, self.deferreds = self.deferreds, []
        for d in deferreds:
            d.callback(line)

class UnicodeCharReader(protocol.Protocol, basic._PauseableMixin):
    '''
    Fires deferreds for every unicode character read from a stream. Useful on
    stdin with non-canonical input mode set.
    '''
    MAX_LENGTH = 99999
    recvd = ''
    deferreds = ()
    encoding = 'utf-8'

    def dataReceived(self, recd):
        self.recvd = self.recvd + recd
        if self.paused:
            return

        data = self.recvd.decode(self.encoding, 'ignore')
        while data and not self.paused:
            char, data = data[0], data[1:]
            self.recvd = self.recvd[len(char.encode(self.encoding)):]
            self.charReceived(char)

    def attach(self, d):
        if not self.deferreds:
            self.deferreds = []
        self.deferreds.append(d)

    def charReceived(self, byte):
        deferreds, self.deferreds = self.deferreds, ()
        for d in deferreds:
            d.callback(byte)

_input_mode_manager = None
class InputModeManager(object):
    def __init__(self):
        self.original_mode = self.get_input_mode()
        from twisted.internet import reactor
        reactor.addSystemEventTrigger('before', 'shutdown', self.finalise)

    def finalise(self):
        self.set_input_mode(self.original_mode)

    def get_input_mode(self):
        '''
        @returns [iflag, oflag, cflag, lflag, ispeed, ospeed, cc]
        '''
        return termios.tcgetattr(sys.stdin)

    def modify_input_lflags(self, set=0, clear=0):
        '''
        Modifies the input mode by setting and clearing the given bits.
        '''
        data = self.get_input_mode()
        data[3] |= set
        data[3] &= ~clear
        self.set_input_mode(data)

    def set_input_mode(self, mode):
        termios.tcsetattr(sys.stdin, termios.TCSANOW, mode)

def get_input_mode_manager():
    global _input_mode_manager
    if not _input_mode_manager:
        _input_mode_manager = InputModeManager()
    return _input_mode_manager
