from collections import defaultdict
import functools
import logging
import types

import pygame
from talljosh import Function, evaluate
from twisted.internet import defer

log = logging.getLogger(__name__)
keyboard_status = defaultdict(lambda: False)

@evaluate
def key_names():
    result = dict((name[len('K_'):], getattr(pygame.constants, name))
            for name in dir(pygame.constants)
            if name.startswith('K_'))
    for i in xrange(10):
        result[i] = result[str(i)]
    return result

class EventKind(object):
    pass

class Event(object):
    kind = None
    propagating = True

    def __init__(self, kind, **kwargs):
        self.kind = kind
        for key, value in kwargs.iteritems():
            setattr(self, key, value)

    def caught(self):
        self.propagating = False

class EventQueue(object):
    def __init__(self):
        self.deferreds = []
        self.subscribers = []

    def next(self, kinds=None):
        d = defer.Deferred()
        self.deferreds.append((d, kinds))
        return d

    def subscribe(self, callback):
        self.subscribers.append(callback)

    def unsubscribe(self, callback):
        self.subscribers.remove(callback)

    def push(self, event):
        '''
        Pushes the given event to this event queue.
        @returns: True if the event has been handled, otherwise False.
        '''
        deferreds = iter(self.deferreds[:])
        self.deferreds[:] = []
        while event.propagating:
            try:
                d, kinds = deferreds.next()
            except StopIteration:
                break

            if event.kind in kinds:
                try:
                    d.callback(event)
                except:
                    log.error('Error in event processing', exc_info=True)
            else:
                self.deferreds.append((d, kinds))

        subscribers = list(self.subscribers)
        while event.propagating and subscribers:
            callback = subscribers.pop(0)
            try:
                callback(event)
            except:
                log.error('Error in event processing', exc_info=True)

        return not event.propagating

# Global event queue.
queue = EventQueue()

class EventHandlerMeta(type):
    def __new__(class_, name, bases, dict_):
        handlers = {}
        for base in reversed(bases):
            handlers.update(getattr(base, '_handlers', {}))

        for key, value in dict_.iteritems():
            for event_kind in getattr(value, 'handles', ()):
                handlers[event_kind] = value

            if isinstance(value, types.FunctionType):
                new_value = staticmethod(value)
                dict_[key] = new_value

        dict_['_handlers'] = handlers
        result = super(EventHandlerMeta, class_).__new__(class_, name, bases, dict_)
        return result

    def __get__(self, instance, class_):
        if instance is None:
            return self
        return functools.partial(self, instance)

class EventHandler(object):
    __metaclass__ = EventHandlerMeta
    def __new__(class_, *args):
        event = args[-1]
        if event.kind in class_._handlers:
            class_._handlers[event.kind](*args)

QuitRequest = EventKind()
WindowActive = EventKind()
KeyUp = EventKind()
KeyDown = EventKind()
MouseMotion = EventKind()
MouseUp = EventKind()
MouseDown = EventKind()
JoystickAxisMotion = EventKind()
JoystickBallMotion = EventKind()
JoystickHatMotion = EventKind()
JoystickUp = EventKind()
JoystickDown = EventKind()
WindowResize = EventKind()
WindowExpose = EventKind()

class event_from_pygame(Function):
    type_mapping = {
        pygame.QUIT: (QuitRequest, ()),
        pygame.ACTIVEEVENT: (WindowActive, ('gain', 'state')),
        pygame.KEYDOWN: (KeyDown, ('unicode', 'key', 'mod')),
        pygame.KEYUP: (KeyUp, ('key', 'mod')),
        pygame.MOUSEMOTION: (MouseMotion, ('pos', 'rel', 'buttons')),
        pygame.MOUSEBUTTONUP: (MouseUp, ('pos', 'button')),
        pygame.MOUSEBUTTONDOWN: (MouseDown, ('pos', 'button')),
        pygame.JOYAXISMOTION: (JoystickAxisMotion, ('joy', 'axis', 'value')),
        pygame.JOYBALLMOTION: (JoystickBallMotion, ('joy', 'ball', 'rel')),
        pygame.JOYHATMOTION : (JoystickHatMotion, ('joy', 'hat', 'value')),
        pygame.JOYBUTTONUP  : (JoystickUp, ('joy', 'button')),
        pygame.JOYBUTTONDOWN: (JoystickDown, ('joy', 'button')),
        pygame.VIDEORESIZE  : (WindowResize, ('size', 'w', 'h')),
        pygame.VIDEOEXPOSE  : (WindowExpose, ()),
    }

    def run(self, pygame_event):
        event_kind, attrs = self.type_mapping[pygame_event.type]
        result = Event(event_kind)
        for attr in attrs:
            setattr(result, attr, getattr(pygame_event, attr))

        return result
