import os
import platform
import re
import shutil
import sys

from setuptools.command.build_ext import build_ext
from setuptools.command.install_lib import install_lib
from setuptools import Extension
from setuptools import setup
from subprocess import call

generated_lib = None


class wurf_install_lib(install_lib):
    def install(self):
        target = os.path.abspath(self.install_dir)
        if not os.path.isdir(target):
            os.makedirs(target)

        global generated_lib
        output_file = os.path.join(target, os.path.basename(generated_lib))
        if os.path.exists(output_file):
            os.remove(output_file)
        shutil.move(generated_lib, target)
        print('WRITING LIBRARY: {}'.format(output_file))
        return [output_file]


class wurf_build_ext(build_ext):
    def run(self):

        # Clone project
        call([
            'git',
            'clone',
            'git@github.com:steinwurf/fifi-python.git',
            'fifi-python'])

        cwd = os.getcwd()
        os.chdir(os.path.join(cwd, 'fifi-python'))

        # Checkout the right version
        call(['git', 'checkout', '2.0.1'])

        # We should always build the python bindings with the architecture
        # of the used python executable.
        python_architecture = platform.architecture()[0]

        # Mapping from the python architecture to a suitable mkspec.
        cxx_mkspec = {
            '32bit': 'cxx_default_x86',
            '64bit': 'cxx_default_x64',
        }[python_architecture]

        install_path = 'installed_binaries'
        build_cmd = [
            sys.executable,
            'waf',
            'configure',
            '--options=install_path={},cxx_mkspec={}'.format(
                install_path,
                cxx_mkspec),
            'build', 'install']

        call(build_cmd)
        # Find the created library file
        # This can be either:
        #   [python_lib].so
        #   [python_lib].pyd
        #   [python_lib].cpython-34m.so
        # and possibly more.
        regex = '{}\.(so|pyd|.*\.so)$'.format('fifi')

        global generated_lib
        for filename in os.listdir(install_path):
            if re.match(regex, filename):
                generated_lib = os.path.join(
                    'fifi-python', install_path, filename)
        os.chdir(cwd)

setup(
    name='fifi',
    version='2.0.1',
    ext_modules=[Extension('fifi', sources=[])],
    description='Finite Field Library',
    long_description=open('README.txt').read(),
    license='Steinwurf Research License',
    author='Steinwurf ApS',
    author_email='pip@steinwurf.com',
    url='https://github.com/steinwurf/fifi-python',
    keywords='steinwurf finite field ',
    platforms=[
        'Windows',
        'Linux',
        'MacOS X',
        ],
    classifiers=[
        'Development Status :: 4 - Beta',
        'Intended Audience :: Developers',
        'Intended Audience :: Science/Research',
        'License :: Free For Educational Use',
        'Operating System :: MacOS :: MacOS X',
        'Operating System :: Microsoft :: Windows',
        'Operating System :: POSIX :: Linux',
        'Programming Language :: C++',
        'Programming Language :: Python',
        'Programming Language :: Python :: 2',
        'Programming Language :: Python :: 2.7',
        'Programming Language :: Python :: 3',
        'Programming Language :: Python :: 3.4',
        ],
    cmdclass={'build_ext': wurf_build_ext, 'install_lib': wurf_install_lib}
)
