# Firelet - Distributed firewall management.
# Copyright (C) 2010 Federico Ceratto
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from firelet import mailer
from firelet.flcore import *
from firelet.flssh import SSHConnector, MockSSHConnector
from firelet.flmap import draw_svg_map
from firelet.flutils import flag, Bunch, get_rss_channels
from nose.tools import assert_raises, with_setup, raises
from nose.plugins.skip import SkipTest

from firelet import cli
from firelet.cli import main as cli_main

from logging import getLogger
log = getLogger(__name__)
deb = log.debug

#TODO: migration to network objects
#TODO: parallel SSH
#TODO: SSH check and deployment

# Summary of tests:
#
# User Management
# File load/save
# GitFireSet
#
# CLI
# IP address manipulation
# cartesian product
#

from testingutils import *
import testingutils

def debug(s, o=None):
    """Log an object representation"""
    try:
        d = dumps(o, indent=2)
    except:
        d = repr(o)
    li = d.split('\n')
    if len(li) < 3:
        log.debug("%s: %s" % (s, repr(o)))
    else:
        indented = "\n    ".join(li)
        log.debug("-------- [%s] ---------\n    %s" % (s, indented))
        log.debug("----- [end of %s] -----" % s)


# #  Testing Confreader  # #

@with_setup(setup_dir, teardown_dir)
def test_confreader():
    from firelet.confreader import ConfReader
    conf = ConfReader('firelet.ini')
    assert conf.title == 'Firelet'
    assert conf.ssh_username == 'firelet'

# #  Testing misc  # #

def test_validc():
    """Test invalid characters"""
    for c in (30, 34, 39, 60, 62, 96, 128):
        assert validc(chr(c)) == False

def test_clean():
    """Test user input cleanup"""
    s = clean(' !"#$%&\'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\]^_')
    assert s == ' !#$%&()*+,-./0123456789:;=?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\]^_'

# #  Testing Mailer  # #
from firelet.mailer import Mailer
from mock import Mock

def test_send_msg():
    m = Mailer()
    m._send = Mock()
    m.send_msg()
    assert m._send.call_count == 1
    msg = m._send.call_args[0][3] # 'msg' passed to _send()
    assert 'Subject: [Firelet] Message' in msg
    assert 'DOCTYPE html' in msg

def test_send_diff():
    m = Mailer()
    m._send = Mock()
    m.send_diff({'items':[]})
    assert m._send.call_count == 1
    msg = m._send.call_args[0][3] # 'msg' passed to _send()
    assert 'Subject: [Firelet] Diff' in msg
    assert 'DOCTYPE html' in msg

#FIXME
def disabledtest_send_html():
    m = Mailer()
    m._send = Mock()
    m.send_html(sbj='sbj', body='body')
    assert m._send.call_count == 1
    msg = m._send.call_args[0][3] # 'msg' passed to _send()
    assert 'Subject: [Firelet] Message' in msg
    assert 'DOCTYPE html' in msg

# #  Testing flssh module without network interaction # #

#TODO: test SSHConnector instead of MockSSHConnector where possible

def test_parse_iptables_save_1():
    sx = MockSSHConnector(targets={'localhost':['127.0.0.1']})
    assert_raises(Exception, sx.parse_iptables_save, '')

def test_parse_iptables_save_2():
    sx = MockSSHConnector(targets={'localhost':['127.0.0.1']})
    ret = sx.parse_iptables_save("""
# Generated by iptables-save v1.4.9 on Sun Feb 20 15:04:36 2011
*filter
:INPUT ACCEPT [36:4504]
:FORWARD ACCEPT [0:0]
:OUTPUT ACCEPT [36:4504]
-A INPUT -s 3.3.3.3/32 -j ACCEPT
-A INPUT -d 3.3.3.3/32 -p tcp -m tcp --dport 44 -j ACCEPT
COMMIT
# Completed on Sun Feb 20 15:04:36 2011
    """)
    assert repr(ret) == """{'filter': ['-A INPUT -s 3.3.3.3/32 -j ACCEPT', '-A INPUT -d 3.3.3.3/32 -p tcp -m tcp --dport 44 -j ACCEPT'], 'nat': []}""", \
        "Returned: %s" % repr(ret)

def test_parse_iptables_save_3():
    sx = MockSSHConnector(targets={'localhost':['127.0.0.1']})
    ret = sx.parse_iptables_save("""*nat
:PREROUTING ACCEPT [7:600]
:POSTROUTING ACCEPT [987:59895]
:OUTPUT ACCEPT [987:59895]
-A PREROUTING -d 1.2.3.4/32 -p tcp -m tcp --dport 44 -j ACCEPT
COMMIT
*filter
:INPUT ACCEPT [36:4504]
:FORWARD ACCEPT [0:0]
:OUTPUT ACCEPT [36:4504]
-A INPUT -d 3.3.3.3/32 -p tcp -m tcp --dport 44 -j ACCEPT
COMMIT""")
    assert repr(ret) == """{'filter': ['-A INPUT -d 3.3.3.3/32 -p tcp -m tcp --dport 44 -j ACCEPT'], 'nat': ['-A PREROUTING -d 1.2.3.4/32 -p tcp -m tcp --dport 44 -j ACCEPT']}""", \
        "Returned: %s" % repr(ret)


#def test_gen_iptables_restore_1():
#    sx = SSHConnector(targets={'localhost':['127.0.0.1']})
#    block = sx._gen_iptables_restore('localhost', [])
#    assert block == [
#        '# Created by Firelet for host localhost',
#        '*filter',
#        ':INPUT ACCEPT',
#        ':FORWARD ACCEPT',
#        ':OUTPUT ACCEPT',
#        'COMMIT'
#    ], "Incorrect empty iptables-restore block created: %s" % repr(block)

@SkipTest
@with_setup(setup_dir, teardown_dir)
def test_MockSSHConnector_get_confs():
    sshconn = MockSSHConnector(targets={'localhost':['127.0.0.1']})
    sshconn.repodir = testingutils.repodir
    d  = sshconn.get_confs( )
    assert 'localhost' in d, repr(d)
    assert 'iptables' in d['localhost'], repr(d)
    assert 'ip_a_s' in d['localhost'], repr(d)
    assert d['localhost'].iptables != None
    assert d['localhost'].ip_a_s != None

    ok = {'localhost': {'iptables': ['# Generated by iptables-save v1.4.8 on Sun Jul  4 09:28:19 2010', '*nat', ':PREROUTING ACCEPT [8:3712]', ':POSTROUTING ACCEPT [32:3081]', ':OUTPUT ACCEPT [32:3081]', '-A POSTROUTING -o eth3 -j MASQUERADE', 'COMMIT', '# Completed on Sun Jul  4 09:28:19 2010', '# Generated by iptables-save v1.4.8 on Sun Jul  4 09:28:19 2010', '*filter', ':INPUT ACCEPT [4304:2534591]', ':FORWARD ACCEPT [0:0]', ':OUTPUT ACCEPT [4589:2195434]', '-A INPUT -s 10.0.0.0/8 -p tcp -m tcp --dport 80 -j ACCEPT', '-A FORWARD -s 1.2.3.4/32 -d 5.6.7.8/32 -p tcp -m multiport --dports 22,80,443 -j ACCEPT', '-A OUTPUT -d 10.10.10.10/32 -p udp -m udp --dport 123 -j ACCEPT', 'COMMIT', '# Completed on Sun Jul  4 09:28:19 2010'], 'ip_a_s': {'lo': ('127.0.0.1/8', '::1/128'), 'wlan0': ('192.168.1.1/24', 'fe80::219:d2ff:fe26:fb8e/64'), 'eth0': (None, None)}}}

    for x in d:
        for y in d[x]:
            assert d[x][y] == ok[x][y], """Incorrect conf retrieved:
                \n%s\n\nversus\n\n%s""" % (d[x][y], ok[x][y])


#def setup_dummy_flssh():
#    """Patch the pxssh module to use files instead of performing network interaction"""
#    import pxssh
#    setup_dir()
#    def dummy_sl(self, a):
#        n = self.my_hostname
#        log.debug( "Sending '%s' to bogus '%s'" % (a, n))
#        if 'save' in a:
#            self.before = open('test/iptables-save-%s' % n).read()
#        else:
#            self.before = open('test/ip-addr-show-%s' % n).read()
#
#    pxssh.login = pxssh.isalive = pxssh.prompt = pxssh.logout = lambda *x: True
#    pxssh.sendline = dummy_sl
#    globals()['pxssh'] = pxssh
#
#def teardown_flssh():
#    teardown_dir()
#
#
#@with_setup(setup_dummy_flssh)
#def test_get_confs_local_dummy():
#    from firelet.flssh import SSHConnector, MockSSHConnector
#
#    sshconn = SSHConnector(targets={'localhost':['127.0.0.1']} )
#    d  = sshconn.get_confs( )
#    assert 'localhost' in d
#    assert d['localhost']
#    assert d == {'localhost': [None, '127.0.0.1', {'filter': '-A INPUT -s 10.0.0.0/8 -p tcp -m tcp --dport 80 -j ACCEPT\n-A FORWARD -s 1.2.3.4/32 -d 5.6.7.8/32 -p tcp -m multiport --dports 22,80,443 -j ACCEPT\n-A OUTPUT -d 10.10.10.10/32 -p udp -m udp --dport 123 -j ACCEPT', 'nat': '-A POSTROUTING -o eth3 -j MASQUERADE'}, {'lo': ('127.0.0.1/8', '::1/128'), 'teredo': (None, 'fe80::ffff:ffff:ffff/64'), 'wlan0': ('192.168.1.1/24', 'fe80::219:d2ff:fe26:fb8e/64'), 'eth0': (None, None)}]}




#@with_setup(setup_dummy_flssh, teardown_dir)
#def test_get_confs3():
#    fs = DumbFireSet(repodir=testingutils.repodir)
#    fs._get_confs()
#    assert fs._remote_confs == {'InternalFW': [None, '10.66.2.1', {'filter': '-A INPUT -s 10.0.0.0/8 -p tcp -m tcp --dport 80 -j ACCEPT\n-A FORWARD -s 1.2.3.4/32 -d 5.6.7.8/32 -p tcp -m multiport --dports 22,80,443 -j ACCEPT\n-A OUTPUT -d 10.10.10.10/32 -p udp -m udp --dport 123 -j ACCEPT', 'nat': '-A POSTROUTING -o eth3 -j MASQUERADE'}, {'lo': ('127.0.0.1/8', '::1/128'), 'eth1': ('10.66.2.1/24', 'fe80::3939:3939:3939:3939/64'), 'eth0': ('10.66.1.2/24', 'fe80::3939:3939:3939:3939/64')}], 'Server001': [None, '10.66.2.2', {'filter': '-A INPUT -s 10.0.0.0/8 -p tcp -m tcp --dport 80 -j ACCEPT\n-A FORWARD -s 1.2.3.4/32 -d 5.6.7.8/32 -p tcp -m multiport --dports 22,80,443 -j ACCEPT\n-A OUTPUT -d 10.10.10.10/32 -p udp -m udp --dport 123 -j ACCEPT', 'nat': '-A POSTROUTING -o eth3 -j MASQUERADE'}, {'lo': ('127.0.0.1/8', '::1/128'), 'eth0': ('10.66.2.2/24', 'fe80::3939:3939:3939:3939/64')}], 'BorderFW': [None, '10.66.1.1', {'filter': '-A INPUT -s 10.0.0.0/8 -p tcp -m tcp --dport 80 -j ACCEPT\n-A FORWARD -s 1.2.3.4/32 -d 5.6.7.8/32 -p tcp -m multiport --dports 22,80,443 -j ACCEPT\n-A OUTPUT -d 10.10.10.10/32 -p udp -m udp --dport 123 -j ACCEPT', 'nat': '-A POSTROUTING -o eth3 -j MASQUERADE'}, {'lo': ('127.0.0.1/8', '::1/128'), 'eth1': ('10.66.1.1/24', 'fe80::3939:3939:3939:3939/64'), 'eth0': ('172.16.2.223/24', 'fe80::3939:3939:3939:3939/64')}], 'Smeagol': [None, '10.66.1.3', {'filter': '-A INPUT -s 10.0.0.0/8 -p tcp -m tcp --dport 80 -j ACCEPT\n-A FORWARD -s 1.2.3.4/32 -d 5.6.7.8/32 -p tcp -m multiport --dports 22,80,443 -j ACCEPT\n-A OUTPUT -d 10.10.10.10/32 -p udp -m udp --dport 123 -j ACCEPT', 'nat': '-A POSTROUTING -o eth3 -j MASQUERADE'}, {'lo': ('127.0.0.1/8', '::1/128'), 'eth0': ('10.66.1.3/24', 'fe80::3939:3939:3939:3939/64')}]}



#@with_setup(setup_dummy_flssh, teardown_dir)
#def test_get_confs4():
#    fs = DumbFireSet(repodir=testingutils.repodir)
#    fs._get_confs()
#    fs._check_ifaces()
#    rd = fs.compile_dict(hosts=fs.hosts)


from paramiko import SSHClient
import mock

# mock the paramiko.SSHClient.connect method to test _connect_one
# _connect_one is normally running in a dedicated thread
@mock.patch.object(SSHClient, 'connect')
def test_flssh_connect_one(mocked_connect):

    sx = SSHConnector(targets={})
    sx._connect_one('bogusfirewall', (
        '0.0.0.1',
        '0.0.0.2',
    ))
    assert sx._pool['bogusfirewall'].ip_addr == '0.0.0.1'


# #  User management testing  # #

@with_setup(setup_dir, teardown_dir)
def test_user_management():
    u = Users(d=testingutils.repodir)
    u.create('Totoro', 'admin', 'rawr', 'totoro@nowhere.forest')
    assert_raises(Exception,  u.create, 'Totoro', '', '', '')
    u.validate('Totoro', 'rawr')
    assert_raises(Exception, u.validate, 'Totoro', 'booo')
    u.update('Totoro', role='user')
    assert u._users['Totoro'][0] == 'user'
    u.update('Totoro', pwd='')
    u.update('Totoro', email='')
    assert u._users['Totoro'][2] == ''
    assert_raises(Exception, u.update, 'Totoro2', 'email=""')
    u.delete('Totoro')
    assert_raises(Exception,  u.delete, 'Totoro')


# # File save/load # #

@with_setup(setup_dir, teardown_dir)
def test_load_save_hosts():
    lines = open(testingutils.repodir + '/hosts.csv', 'r').readlines()
    content = [x.strip() for x in lines]
    content = filter(None, content)
    h = Hosts(d=testingutils.repodir)
    h.save()
    lines = open(testingutils.repodir + '/hosts.csv', 'r').readlines()
    content2 = [x.strip() for x in lines]
    content2 = filter(None, content2)
    h2 = Hosts(d=testingutils.repodir)
    assert content == content2, "load/save hosts loop failed:\n\n%s\n\n%s\n\n" \
        % (repr(content), repr(content2))
    assert repr(h) == repr(h2), "load/save hosts loop failed"

@with_setup(setup_dir, teardown_dir)
def test_load_save_csv():
    h = readcsv('rules', d=testingutils.repodir)
    h = tuple(h)
    savecsv('rules', h, d=testingutils.repodir)
    h2 = readcsv('rules', d=testingutils.repodir)
    h2 = tuple(h2)
    assert h == h2, "load/save hosts loop failed:\n%s\n!=\n%s" % \
        (h, h2)

# #  FireSet testing # #

@with_setup(setup_dir, teardown_dir)
def test_gitfireset_otp():
    fs = GitFireSet(repodir=testingutils.repodir)
    otp = fs.generate_otp()
    assert isinstance(otp, str)
    assert len(otp) == 10

@with_setup(setup_dir, teardown_dir)
def test_gitfireset_simple():
    fs = GitFireSet(repodir=testingutils.repodir)
    assert fs.save_needed() == False
    fs.save('test')
    assert fs.save_needed() == False
    fs.reset()
    assert fs.save_needed() == False

@with_setup(setup_dir, teardown_dir)
def test_gitfireset_long():
    fs = GitFireSet(repodir=testingutils.repodir)
    # Delete first item in every table
    for t in ('rules', 'hosts', 'hostgroups', 'services', 'networks'):
        fs.delete(t, 1)
        assert fs.save_needed() == True, "save_needed non set when deleting item 1 from %s" % t
        fs.save("%s: n.1 deleted" % t)
        assert fs.save_needed() == False

    # Perform changes
    fs.rules.disable(2)
    assert not fs.rules.enabled(2), "Rule 2 should be flagged as disabled"
    fs.rules.enable(2)
    assert fs.rules.enabled(2), "Rule 2 should be flagged as enabled"
    fs.rules.moveup(2)
    assert fs.save_needed() == True
    fs.rules.movedown(1)
    fs.save('movedown1')
    fs.rules.movedown(2)
    fs.save('movedown2')
    fs.rules.movedown(3)
    fs.save('movedown3')

    # Check version list
    vl = fs.version_list()
    assert zip(*vl)[2] == (['movedown3'],
        ['movedown2'],
        ['networks: n.1 deleted'],
        ['services: n.1 deleted'],
        ['hostgroups: n.1 deleted'],
        ['hosts: n.1 deleted'],
        ['rules: n.1 deleted'])
    dup = duplicates(zip(*vl)[3])
    assert not dup, "Some commit IDs are duplicate: \
    %s" % repr(dup)

    # Check version_diff
    last_commit_id = vl[-1][-1]
    diff = fs.version_diff(last_commit_id)
    assert ('1 http_ok InternalFW:eth1 * Server001:eth0 HTTP ACCEPT 0 "Web server"',
        'add') in diff
    assert ('1 http_ok InternalFW:eth1 * Server001:eth0 HTTP ACCEPT 0 "Web server"',
        'del') in diff
    assert len(diff) == 52

    # Rollback and check again
    fs.rollback(2)
    assert fs.save_needed() == False
    vl = fs.version_list()
    log.debug('version_list: %s' % repr(vl))
    assert zip(*vl)[2] == (['networks: n.1 deleted'],
        ['services: n.1 deleted'],
        ['hostgroups: n.1 deleted'],
        ['hosts: n.1 deleted'],
        ['rules: n.1 deleted'])


@with_setup(setup_dir, teardown_dir)
def test_gitfireset_smarttable_methods():
    fs = GitFireSet(repodir=testingutils.repodir)

    h = fs.fetch('hosts', 0)
    fs.delete('hosts', 0)
    #TODO: add other methods

#        d = {'name': pg('name'),
#                    'childs': childs}
#            if rid == None:     # new item
#                fs.hostgroups.add(d)


@with_setup(setup_dir, teardown_dir)
def test_gitfireset_check_ifaces_1():
    fs = GitFireSet(repodir=testingutils.repodir)
    fs._remote_confs = None
    assert_raises(AssertionError, fs._check_ifaces)

@with_setup(setup_dir, teardown_dir)
def test_gitfireset_check_ifaces_20():
    fs = GitFireSet(repodir=testingutils.repodir)
    d = {'InternalFW': {'filter': [], 'ip_a_s': {'eth1': ('10.66.2.1/24',None),
                'eth0': ('10.66.1.2/24', None)}},
            'Server001': {'filter': [], 'ip_a_s': {'eth0': ('10.66.2.2/24', None)}},
            'BorderFW': {'filter': [], 'ip_a_s': {
                'eth1': ('10.66.1.1/24', None),
                'eth2': ('88.88.88.88/24', None),
                'eth0': ('172.16.2.223/24', None)}},
            'Smeagol': {'filter': [], 'ip_a_s': {'eth0': ('10.66.1.3/24', None)}} }
    fs._remote_confs = {}
    for n, v in d.iteritems():
        fs._remote_confs[n] = Bunch(filter=v['filter'], ip_a_s=v['ip_a_s'])
    fs._check_ifaces()

@with_setup(setup_dir, teardown_dir)
def test_gitfireset_check_ifaces_wrong_value():
    fs = GitFireSet(repodir=testingutils.repodir)
    fs._remote_confs = {'bogus': 'not a bunch'} # value should be a Bunch
    assert_raises(AssertionError, fs._check_ifaces)

@with_setup(setup_dir, teardown_dir)
def test_gitfireset_check_ifaces_wrong_bunch_size():
    fs = GitFireSet(repodir=testingutils.repodir)
    fs._remote_confs = {'bogus': Bunch()} # len(Bunch(...)) should be 2 (ip_addr_v4, ip_addr_v6)
    assert_raises(AssertionError, fs._check_ifaces)

@with_setup(setup_dir, teardown_dir)
def test_gitfireset_check_ifaces_missing_iface():
    fs = GitFireSet(repodir=testingutils.repodir)
    fs.hosts = [
        Bunch(hostname='host1', iface='lo'),
        Bunch(hostname='host2', iface='lo')
    ]
    fs._remote_confs = {
        'host1': Bunch(ip_a_s = {'lo': ()}),
        'host2': Bunch(ip_a_s = {}) # missing iface
    }
    assert_raises(AssertionError, fs._check_ifaces)

@with_setup(setup_dir, teardown_dir)
def test_gitfireset_check_ifaces_wrong_ipaddr_string():
    """_check_ifaces should raise AssertionError on incorrect IPaddr strings"""
    fs = GitFireSet(repodir=testingutils.repodir)
    fs.hosts = [
        Bunch(hostname='host1', iface='lo'),
        Bunch(hostname='host2', iface='lo')
    ]
    fs._remote_confs = {
        'host1': Bunch(ip_a_s = {'lo': ('bogus', 'bogus')}),
        'host2': Bunch(ip_a_s = {'lo': ('bogus', 'bogus') })
    }
    assert_raises(AssertionError, fs._check_ifaces)

@with_setup(setup_dir, teardown_dir)
def test_gitfireset_check_ifaces_wrong_ipaddr():
    """_check_ifaces should raise AssertionError on incorrect IPaddr"""
    fs = GitFireSet(repodir=testingutils.repodir)
    fs.hosts = [
        Bunch(hostname='host1', iface='lo', ip_addr='1.2.3.4'),
    ]
    fs._remote_confs = {
        'host1': Bunch(ip_a_s = {'lo': ('1.2.3.5/32', None)}),
    }
    assert_raises(AssertionError, fs._check_ifaces)

@with_setup(setup_dir, teardown_dir)
def test_gitfireset_check_ifaces_correct():
    fs = GitFireSet(repodir=testingutils.repodir)
    fs.hosts = [
        Bunch(hostname='host1', iface='lo', ip_addr='1.2.3.4', mng='1'),
    ]
    fs._remote_confs = {
        'host1': Bunch(
            ip_a_s = {'lo': ('1.2.3.4/32', None)},
            iptables_p = Bunch()
        ),
    }
    fs._check_ifaces(stop_on_extra_interfaces=False)

@with_setup(setup_dir, teardown_dir)
def test_gitfireset_check_ifaces_correct2():
    """gitfireset_check_ifaces_correct2 has stop_on_extra_interfaces = True"""
    fs = GitFireSet(repodir=testingutils.repodir)
    fs.hosts = [
        Bunch(hostname='host1', iface='lo', ip_addr='1.2.3.4', mng=1),
    ]
    fs._remote_confs = {
        'host1': Bunch(
            ip_a_s = {'lo': ('1.2.3.4/32', None)},
            iptables_p = Bunch()
        ),
    }
    fs._check_ifaces(stop_on_extra_interfaces=True)

@with_setup(setup_dir, teardown_dir)
def test_gitfireset_check_ifaces_alert():
    """gitfireset_check_ifaces_correct3 should raise Alert
    'Alert: One or more firewalls have extra interfaces: host1: eth0'
    """
    fs = GitFireSet(repodir=testingutils.repodir)
    fs.hosts = [
        Bunch(hostname='host1', iface='lo', ip_addr='1.2.3.4', mng=1),
    ]
    fs._remote_confs = {
        'host1': Bunch(
            ip_a_s = {
                'lo': ('1.2.3.4/32', None),
                'eth0': ('7.7.7.7/32', None), #extra iface
            },
            iptables_p = Bunch()
        ),
    }
    assert_raises(Alert, fs._check_ifaces, stop_on_extra_interfaces=True)


@with_setup(setup_dir, teardown_dir)
def test_gitfireset_sibling_names():
    fs = GitFireSet(repodir=testingutils.repodir)
    names = ['AllSystems', 'BorderFW:eth0', 'BorderFW:eth1', 'BorderFW:eth2', 'Clients', 'InternalFW:eth0', \
        'InternalFW:eth1', 'SSHnodes', 'Server001:eth0', 'Servers', 'Smeagol:eth0', 'Tester:eth1', 'WebServers']

    sn = fs.list_sibling_names()
    assert sorted(sn) == sorted(names), "list_sibling_names generating incorrect output: %s" % repr(sorted(sn))

@with_setup(setup_dir, teardown_dir)
def test_gitfireset_get_firewalls():
    fs = GitFireSet(repodir=testingutils.repodir)
    hosts = fs._get_firewalls()
    hostnames = sorted((h.hostname, h.iface) for h in hosts)
    ok = [('BorderFW', 'eth0'), ('BorderFW', 'eth1'), ('BorderFW', 'eth2'), ('InternalFW', 'eth0'),
        ('InternalFW', 'eth1'), ('Server001', 'eth0'), ('Smeagol', 'eth0')]
    assert hostnames == ok, "Wrong hosts selected as firewalls: %s" % repr(hostnames)

#@with_setup(setup_dir, teardown_dir)
#def test_dumbfireset():
#    fs = DumbFireSet(repodir=testingutils.repodir)
#    assert fs.save_needed() == False
#    fs.save('save')
#    assert fs.save_needed() == False
#    fs.reset()
#    assert fs.save_needed() == False
#    fs.rollback(2)
#    assert fs.save_needed() == False
#    vl = fs.version_list()
#    # assert
#    for t in ('rules', 'hosts', 'hostgroups', 'services', 'networks'):
#        tmp = len(fs.__dict__[t])
#        fs.delete(t, 0)
#        assert fs.save_needed() == True, t
#        assert tmp == len(fs.__dict__[t]) + 1, t
#    fs.save('test')
#    assert fs.save_needed() == False
#    orig_rules = fs.rules[:] # copy
#    fs.rules.moveup(2)
#    assert fs.save_needed() == True
#    assert orig_rules != fs.rules
#    fs.rules.movedown(1)
#    assert orig_rules == fs.rules
#
#    fs.rules.movedown(1)
#    assert orig_rules != fs.rules
#    assert fs.save_needed() == True
#    fs.reset()
#    assert fs.save_needed() == False
#    assert orig_rules == fs.rules



@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_get_confs():
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    fs._get_confs(keep_sessions=False)
    for hostname, v in fs._remote_confs.iteritems():
        assert isinstance(v, Bunch)
    for h in fs._get_firewalls():
        assert h.hostname in fs._remote_confs, "Missing host %s" % h.hostname

@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_deployment():
    """Deploy confs, then check"""
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    fs.deploy()
    diff = fs.check()
    assert diff == {}, repr(diff)[:400]


# # Rule compliation and deployment testing # #

@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_compile_rules_basic():
    """Compile rules and perform basic testing"""
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    rset = fs.compile_rules()
    for hn, d in rset.iteritems():
        for chain,  rules in d.iteritems():
            assert string_in_list('-j DROP', rules), "-j DROP not in %s" % repr(rules)

@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_compile_rules_full():
    fs = GitFireSet(repodir=testingutils.repodir)
    rd = fs.compile_rules()
    ok = {
        "InternalFW": {
        "FORWARD": [
          "-j ACCEPT",
          "-m state --state RELATED,ESTABLISHED -j ACCEPT",
          "-s 10.66.1.1/32 -d 10.66.2.0/24 -p tcp  -m tcp  --dport 22 -j LOG  --log-prefix \"f_ssh_mgmt\" --log-level 2",
          "-s 10.66.1.1/32 -d 10.66.2.0/24 -p tcp  -m tcp  --dport 22 -j ACCEPT",
          "-s 10.66.2.2/32 -d 10.66.1.3/32 -p tcp  -m tcp  --dport 6660:6669 -j LOG  --log-prefix \"f_irc\" --log-level 0",
          "-s 10.66.2.2/32 -d 10.66.1.3/32 -p tcp  -m tcp  --dport 6660:6669 -j ACCEPT",
          "-s 10.66.1.3/32 -d 172.16.2.223/32 -p udp  -m udp  --dport 123 -j LOG  --log-prefix \"f_ntp\" --log-level 0",
          "-s 10.66.1.3/32 -d 172.16.2.223/32 -p udp  -m udp  --dport 123 -j ACCEPT",
          "-s 10.66.2.2/32 -d 172.16.2.223/32 -p udp  -m udp  --dport 123 -j LOG  --log-prefix \"f_ntp\" --log-level 0",
          "-s 10.66.2.2/32 -d 172.16.2.223/32 -p udp  -m udp  --dport 123 -j ACCEPT",
          "-s 10.66.2.2/32 -d 10.66.1.3/32 -p udp  -m udp  --dport 123 -j LOG  --log-prefix \"f_ntp\" --log-level 0",
          "-s 10.66.2.2/32 -d 10.66.1.3/32 -p udp  -m udp  --dport 123 -j ACCEPT",
          " -j LOG  --log-prefix \"f_default\" --log-level 1",
          " -j DROP",
          " -j LOG  --log-prefix \"f_default\" --log-level 1",
          " -j DROP"
        ],
        "INPUT": [
        "-j ACCEPT",
          "-m state --state RELATED,ESTABLISHED -j ACCEPT",
          "-i lo -j ACCEPT",
          "-s 10.66.1.1/32 -d 10.66.2.0/24 -i eth1  -p tcp  -m tcp  --dport 22 -j LOG --log-prefix \"i_ssh_mgmt\" --log-level 2",
          "-s 10.66.1.1/32 -d 10.66.2.0/24 -i eth1  -p tcp  -m tcp  --dport 22 -j ACCEPT",
          "-s 10.66.1.3/32 -d 10.66.1.2/32 -i eth0  -p tcp  -m multiport --dports 143,585,993 -j LOG --log-prefix \"i_imap\" --log-level 2",
          "-s 10.66.1.3/32 -d 10.66.1.2/32 -i eth0  -p tcp  -m multiport --dports 143,585,993 -j ACCEPT",
          " -i eth0  -j LOG --log-prefix \"i_default\" --log-level 1",
          " -i eth0  -j DROP",
          " -i eth1  -j LOG --log-prefix \"i_default\" --log-level 1",
          " -i eth1  -j DROP"
        ],
        "OUTPUT": [
          "-m state --state RELATED,ESTABLISHED -j ACCEPT",
          "-o lo -j ACCEPT",
          "-s 10.66.1.2/32 -d 10.66.1.1/32 -o eth0  -p tcp  -m tcp  --dport 443 -j ACCEPT",
          "-s 10.66.2.1/32 -d 10.66.2.2/32 -o eth1  -p tcp  -m tcp  --dport 80 -j ACCEPT",
          " -o eth0  -j LOG --log-prefix \"o_default\" --log-level 1",
          " -o eth0  -j DROP",
          " -o eth1  -j LOG --log-prefix \"o_default\" --log-level 1",
          " -o eth1  -j DROP"
        ]
      },
      "BorderFW": {
        "FORWARD": [
            '-j ACCEPT',  #FIXME
          "-m state --state RELATED,ESTABLISHED -j ACCEPT",
          "-s 88.88.88.1/32 -d 172.16.2.223/32 -p tcp  -m tcp  --dport 22 -j LOG  --log-prefix \"f_ssh_all\" --log-level 0",
          "-s 88.88.88.1/32 -d 172.16.2.223/32 -p tcp  -m tcp  --dport 22 -j ACCEPT",
          "-s 88.88.88.1/32 -d 10.66.1.3/32 -p tcp  -m tcp  --dport 22 -j LOG  --log-prefix \"f_ssh_all\" --log-level 0",
          "-s 88.88.88.1/32 -d 10.66.1.3/32 -p tcp  -m tcp  --dport 22 -j ACCEPT",
          "-s 88.88.88.1/32 -d 10.66.2.2/32 -p tcp  -m tcp  --dport 22 -j LOG  --log-prefix \"f_ssh_all\" --log-level 0",
          "-s 88.88.88.1/32 -d 10.66.2.2/32 -p tcp  -m tcp  --dport 22 -j ACCEPT",
          "-s 10.66.1.3/32 -d 172.16.2.223/32 -p udp  -m udp  --dport 123 -j LOG  --log-prefix \"f_ntp\" --log-level 0",
          "-s 10.66.1.3/32 -d 172.16.2.223/32 -p udp  -m udp  --dport 123 -j ACCEPT",
          " -j LOG  --log-prefix \"f_default\" --log-level 1",
          " -j DROP",
          " -j LOG  --log-prefix \"f_default\" --log-level 1",
          " -j DROP",
          " -j LOG  --log-prefix \"f_default\" --log-level 1",
          " -j DROP"
        ],
        "INPUT": [
            '-j ACCEPT', #FIXME
          "-m state --state RELATED,ESTABLISHED -j ACCEPT",
          "-i lo -j ACCEPT",
          "-s 88.88.88.1/32 -d 172.16.2.223/32 -i eth0  -p tcp  -m tcp  --dport 22 -j ACCEPT",
          "-s 10.66.1.2/32 -d 10.66.1.1/32 -i eth1  -p tcp  -m tcp  --dport 443 -j ACCEPT",
          "-s 10.66.1.3/32 -d 10.66.1.1/32 -i eth1  -j LOG --log-prefix \"i_NoSmeagol\" --log-level 3",
          "-s 10.66.1.3/32 -d 10.66.1.1/32 -i eth1  -j DROP",
          "-s 10.66.1.3/32 -d 172.16.2.223/32 -i eth0  -p udp  -m udp  --dport 123 -j ACCEPT",
          "-s 10.66.2.2/32 -d 172.16.2.223/32 -i eth0  -p udp  -m udp  --dport 123 -j ACCEPT",
          " -i eth0  -j LOG --log-prefix \"i_default\" --log-level 1",
          " -i eth0  -j DROP",
          " -i eth1  -j LOG --log-prefix \"i_default\" --log-level 1",
          " -i eth1  -j DROP",
          " -i eth2  -j LOG --log-prefix \"i_default\" --log-level 1",
          " -i eth2  -j DROP"
        ],
        "OUTPUT": [
          "-m state --state RELATED,ESTABLISHED -j ACCEPT",
          "-o lo -j ACCEPT",
          "-s 10.66.1.1/32 -d 10.66.2.0/24 -o eth1  -p tcp  -m tcp  --dport 22 -j LOG --log-prefix \"o_ssh_mgmt\" --log-level 2",
          "-s 10.66.1.1/32 -d 10.66.2.0/24 -o eth1  -p tcp  -m tcp  --dport 22 -j ACCEPT",
          "-s 172.16.2.223/32 -d 10.66.1.3/32 -o eth0  -p udp  -m udp  --dport 123 -j ACCEPT",
          " -o eth0  -j LOG --log-prefix \"o_default\" --log-level 1",
          " -o eth0  -j DROP",
          " -o eth1  -j LOG --log-prefix \"o_default\" --log-level 1",
          " -o eth1  -j DROP",
          " -o eth2  -j LOG --log-prefix \"o_default\" --log-level 1",
          " -o eth2  -j DROP"
        ]
      },
      "Server001": {
        "FORWARD": [
          "-j DROP"
        ],
        "INPUT": [
          "-m state --state RELATED,ESTABLISHED -j ACCEPT",
          "-i lo -j ACCEPT",
          "-s 88.88.88.1/32 -d 10.66.2.2/32 -i eth0  -p tcp  -m tcp  --dport 22 -j ACCEPT",
          "-s 10.66.2.1/32 -d 10.66.2.2/32 -i eth0  -p tcp  -m tcp  --dport 80 -j ACCEPT",
          "-d 10.66.2.2/32 -i eth0  -p tcp  -m tcp  --dport 80 -j ACCEPT",
          "-s 10.66.1.1/32 -d 10.66.2.0/24 -i eth0  -p tcp  -m tcp  --dport 22 -j LOG --log-prefix \"i_ssh_mgmt\" --log-level 2",
          "-s 10.66.1.1/32 -d 10.66.2.0/24 -i eth0  -p tcp  -m tcp  --dport 22 -j ACCEPT",
          " -i eth0  -j LOG --log-prefix \"i_default\" --log-level 1",
          " -i eth0  -j DROP"
        ],
        "OUTPUT": [
          "-m state --state RELATED,ESTABLISHED -j ACCEPT",
          "-o lo -j ACCEPT",
          "-s 10.66.2.2/32 -d 10.66.1.3/32 -o eth0  -p tcp  -m tcp  --dport 6660:6669 -j ACCEPT",
          "-s 10.66.2.2/32 -d 172.16.2.223/32 -o eth0  -p udp  -m udp  --dport 123 -j ACCEPT",
          "-s 10.66.2.2/32 -d 10.66.1.3/32 -o eth0  -p udp  -m udp  --dport 123 -j ACCEPT",
          " -o eth0  -j LOG --log-prefix \"o_default\" --log-level 1",
          " -o eth0  -j DROP"
        ]
      },
      "Tester": {
        "FORWARD": [
          "-j DROP"
        ],
        "INPUT": [
          "-m state --state RELATED,ESTABLISHED -j ACCEPT",
          "-i lo -j ACCEPT",
          " -i eth1  -j LOG --log-prefix \"i_default\" --log-level 1",
          " -i eth1  -j DROP"
        ],
        "OUTPUT": [
          "-m state --state RELATED,ESTABLISHED -j ACCEPT",
          "-o lo -j ACCEPT",
          "-s 88.88.88.1/32 -d 172.16.2.223/32 -o eth1  -p tcp  -m tcp  --dport 22 -j ACCEPT",
          "-s 88.88.88.1/32 -d 10.66.1.3/32 -o eth1  -p tcp  -m tcp  --dport 22 -j ACCEPT",
          "-s 88.88.88.1/32 -d 10.66.2.2/32 -o eth1  -p tcp  -m tcp  --dport 22 -j ACCEPT",
          " -o eth1  -j LOG --log-prefix \"o_default\" --log-level 1",
          " -o eth1  -j DROP"
        ]
      },
      "Smeagol": {
        "FORWARD": [
          "-j DROP"
        ],
        "INPUT": [
          "-m state --state RELATED,ESTABLISHED -j ACCEPT",
          "-i lo -j ACCEPT",
          "-s 88.88.88.1/32 -d 10.66.1.3/32 -i eth0  -p tcp  -m tcp  --dport 22 -j ACCEPT",
          "-s 10.66.2.2/32 -d 10.66.1.3/32 -i eth0  -p tcp  -m tcp  --dport 6660:6669 -j ACCEPT",
          "-s 172.16.2.223/32 -d 10.66.1.3/32 -i eth0  -p udp  -m udp  --dport 123 -j ACCEPT",
          "-s 10.66.2.2/32 -d 10.66.1.3/32 -i eth0  -p udp  -m udp  --dport 123 -j ACCEPT",
          " -i eth0  -j LOG --log-prefix \"i_default\" --log-level 1",
          " -i eth0  -j DROP"
        ],
        "OUTPUT": [
          "-m state --state RELATED,ESTABLISHED -j ACCEPT",
          "-o lo -j ACCEPT",
          "-s 10.66.1.3/32 -d 10.66.1.1/32 -o eth0  -j LOG --log-prefix \"o_NoSmeagol\" --log-level 3",
          "-s 10.66.1.3/32 -d 10.66.1.1/32 -o eth0  -j DROP",
          "-s 10.66.1.3/32 -d 10.66.1.2/32 -o eth0  -p tcp  -m multiport --dports 143,585,993 -j LOG --log-prefix \"o_imap\" --log-level 2",
          "-s 10.66.1.3/32 -d 10.66.1.2/32 -o eth0  -p tcp  -m multiport --dports 143,585,993 -j ACCEPT",
          "-s 10.66.1.3/32 -d 172.16.2.223/32 -o eth0  -p udp  -m udp  --dport 123 -j ACCEPT",
          " -o eth0  -j LOG --log-prefix \"o_default\" --log-level 1",
          " -o eth0  -j DROP"
        ]
      }
    }

    for hostname in ok:
        for chain in ok[hostname]:
            for n, my_line in enumerate(rd[hostname][chain]):
                ok_line = ok[hostname][chain][n]
                assert my_line.strip() == ok_line.strip(), "Incorrect rules in %s chain %s:\ngot:      [%s]\nexpected: [%s]\nline: %d" % (
                    hostname, chain, my_line,  ok_line , n )

    #FIXME: review rule, ensure they are really correct



@SkipTest
@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_build_ipt_restore():
    """Run diff between compiled rules and empty remote confs"""
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    rset = fs.compile_rules()
    m = map(fs._build_ipt_restore, rset.iteritems())
    m = dict(m)
    ok = {
      "InternalFW": [
        "# Created by Firelet for host InternalFW",
        "*filter",
        "-A INPUT -m state --state RELATED,ESTABLISHED -j ACCEPT",
        "-A INPUT -i lo -j ACCEPT",
        "-A INPUT -s 10.66.1.1/32 -d 10.66.2.0/24 -i eth1  -p tcp  -m tcp  --dport 22 -j LOG --log-prefix \"ssh_mgmt\" --log-level 2",
        "-A INPUT -s 10.66.1.1/32 -d 10.66.2.0/24 -i eth1  -p tcp  -m tcp  --dport 22 -j ACCEPT",
        "-A INPUT -s 10.66.1.3/32 -d 10.66.1.2/32 -i eth0  -p tcp  -m multiport --dports 143,585,993 -j LOG --log-prefix \"imap\" --log-level 2",
        "-A INPUT -s 10.66.1.3/32 -d 10.66.1.2/32 -i eth0  -p tcp  -m multiport --dports 143,585,993 -j ACCEPT",
        "-A INPUT  -i eth0  -j LOG --log-prefix \"default\" --log-level 1",
        "-A INPUT  -i eth0  -j DROP",
        "-A INPUT  -i eth1  -j LOG --log-prefix \"default\" --log-level 1",
        "-A INPUT  -i eth1  -j DROP",
        "-A FORWARD -m state --state RELATED,ESTABLISHED -j ACCEPT",
        "-A FORWARD -s 10.66.1.1/32 -d 10.66.2.0/24 -p tcp  -m tcp  --dport 22 -j LOG  --log-prefix \"ssh_mgmt\" --log-level 2",
        "-A FORWARD -s 10.66.1.1/32 -d 10.66.2.0/24 -p tcp  -m tcp  --dport 22 -j ACCEPT",
        "-A FORWARD -s 10.66.2.2/32 -d 10.66.1.3/32 -p tcp  -m tcp  --dport 6660:6669 -j LOG  --log-prefix \"irc\" --log-level 0",
        "-A FORWARD -s 10.66.2.2/32 -d 10.66.1.3/32 -p tcp  -m tcp  --dport 6660:6669 -j ACCEPT",
        "-A FORWARD -s 10.66.1.3/32 -d 172.16.2.223/32 -p udp  -m udp  --dport 123 -j LOG  --log-prefix \"ntp\" --log-level 0",
        "-A FORWARD -s 10.66.1.3/32 -d 172.16.2.223/32 -p udp  -m udp  --dport 123 -j ACCEPT",
        "-A FORWARD -s 10.66.2.2/32 -d 172.16.2.223/32 -p udp  -m udp  --dport 123 -j LOG  --log-prefix \"ntp\" --log-level 0",
        "-A FORWARD -s 10.66.2.2/32 -d 172.16.2.223/32 -p udp  -m udp  --dport 123 -j ACCEPT",
        "-A FORWARD -s 10.66.2.2/32 -d 10.66.1.3/32 -p udp  -m udp  --dport 123 -j LOG  --log-prefix \"ntp\" --log-level 0",
        "-A FORWARD -s 10.66.2.2/32 -d 10.66.1.3/32 -p udp  -m udp  --dport 123 -j ACCEPT",
        "-A FORWARD  -j LOG  --log-prefix \"default\" --log-level 1",
        "-A FORWARD  -j DROP",
        "-A FORWARD  -j LOG  --log-prefix \"default\" --log-level 1",
        "-A FORWARD  -j DROP",
        "-A OUTPUT -m state --state RELATED,ESTABLISHED -j ACCEPT",
        "-A OUTPUT -o lo -j ACCEPT",
        "-A OUTPUT -s 10.66.1.2/32 -d 10.66.1.1/32 -o eth0  -p tcp  -m tcp  --dport 443 -j ACCEPT",
        "-A OUTPUT -s 10.66.2.1/32 -d 10.66.2.2/32 -o eth1  -p tcp  -m tcp  --dport 80 -j ACCEPT",
        "-A OUTPUT  -o eth0  -j LOG --log-prefix \"default\" --log-level 1",
        "-A OUTPUT  -o eth0  -j DROP",
        "-A OUTPUT  -o eth1  -j LOG --log-prefix \"default\" --log-level 1",
        "-A OUTPUT  -o eth1  -j DROP",
        "COMMIT"
      ],
      "Server001": [
        "# Created by Firelet for host Server001",
        "*filter",
        "-A INPUT -m state --state RELATED,ESTABLISHED -j ACCEPT",
        "-A INPUT -i lo -j ACCEPT",
        "-A INPUT -s 88.88.88.1/32 -d 10.66.2.2/32 -i eth0  -p tcp  -m tcp  --dport 22 -j ACCEPT",
        "-A INPUT -s 10.66.2.1/32 -d 10.66.2.2/32 -i eth0  -p tcp  -m tcp  --dport 80 -j ACCEPT",
        "-A INPUT -s 0.0.0.0/0 -d 10.66.2.2/32 -i eth0  -p tcp  -m tcp  --dport 80 -j ACCEPT",
        "-A INPUT -s 10.66.1.1/32 -d 10.66.2.0/24 -i eth0  -p tcp  -m tcp  --dport 22 -j LOG --log-prefix \"ssh_mgmt\" --log-level 2",
        "-A INPUT -s 10.66.1.1/32 -d 10.66.2.0/24 -i eth0  -p tcp  -m tcp  --dport 22 -j ACCEPT",
        "-A INPUT  -i eth0  -j LOG --log-prefix \"default\" --log-level 1",
        "-A INPUT  -i eth0  -j DROP",
        "-A FORWARD -j DROP",
        "-A OUTPUT -m state --state RELATED,ESTABLISHED -j ACCEPT",
        "-A OUTPUT -o lo -j ACCEPT",
        "-A OUTPUT -s 10.66.2.2/32 -d 10.66.1.3/32 -o eth0  -p tcp  -m tcp  --dport 6660:6669 -j ACCEPT",
        "-A OUTPUT -s 10.66.2.2/32 -d 172.16.2.223/32 -o eth0  -p udp  -m udp  --dport 123 -j ACCEPT",
        "-A OUTPUT -s 10.66.2.2/32 -d 10.66.1.3/32 -o eth0  -p udp  -m udp  --dport 123 -j ACCEPT",
        "-A OUTPUT  -o eth0  -j LOG --log-prefix \"default\" --log-level 1",
        "-A OUTPUT  -o eth0  -j DROP",
        "COMMIT"
      ],
      "Smeagol": [
        "# Created by Firelet for host Smeagol",
        "*filter",
        "-A INPUT -m state --state RELATED,ESTABLISHED -j ACCEPT",
        "-A INPUT -i lo -j ACCEPT",
        "-A INPUT -s 88.88.88.1/32 -d 10.66.1.3/32 -i eth0  -p tcp  -m tcp  --dport 22 -j ACCEPT",
        "-A INPUT -s 10.66.2.2/32 -d 10.66.1.3/32 -i eth0  -p tcp  -m tcp  --dport 6660:6669 -j ACCEPT",
        "-A INPUT -s 172.16.2.223/32 -d 10.66.1.3/32 -i eth0  -p udp  -m udp  --dport 123 -j ACCEPT",
        "-A INPUT -s 10.66.2.2/32 -d 10.66.1.3/32 -i eth0  -p udp  -m udp  --dport 123 -j ACCEPT",
        "-A INPUT  -i eth0  -j LOG --log-prefix \"default\" --log-level 1",
        "-A INPUT  -i eth0  -j DROP",
        "-A FORWARD -j DROP",
        "-A OUTPUT -m state --state RELATED,ESTABLISHED -j ACCEPT",
        "-A OUTPUT -o lo -j ACCEPT",
        "-A OUTPUT -s 10.66.1.3/32 -d 10.66.1.1/32 -o eth0  -j LOG --log-prefix \"NoSmeagol\" --log-level 3",
        "-A OUTPUT -s 10.66.1.3/32 -d 10.66.1.1/32 -o eth0  -j DROP",
        "-A OUTPUT -s 10.66.1.3/32 -d 10.66.1.2/32 -o eth0  -p tcp  -m multiport --dports 143,585,993 -j LOG --log-prefix \"imap\" --log-level 2",
        "-A OUTPUT -s 10.66.1.3/32 -d 10.66.1.2/32 -o eth0  -p tcp  -m multiport --dports 143,585,993 -j ACCEPT",
        "-A OUTPUT -s 10.66.1.3/32 -d 172.16.2.223/32 -o eth0  -p udp  -m udp  --dport 123 -j ACCEPT",
        "-A OUTPUT  -o eth0  -j LOG --log-prefix \"default\" --log-level 1",
        "-A OUTPUT  -o eth0  -j DROP",
        "COMMIT"
      ],
      "Tester": [
        "# Created by Firelet for host Tester",
        "*filter",
        "-A INPUT -m state --state RELATED,ESTABLISHED -j ACCEPT",
        "-A INPUT -i lo -j ACCEPT",
        "-A INPUT  -i eth1  -j LOG --log-prefix \"default\" --log-level 1",
        "-A INPUT  -i eth1  -j DROP",
        "-A FORWARD -j DROP",
        "-A OUTPUT -m state --state RELATED,ESTABLISHED -j ACCEPT",
        "-A OUTPUT -o lo -j ACCEPT",
        "-A OUTPUT -s 88.88.88.1/32 -d 172.16.2.223/32 -o eth1  -p tcp  -m tcp  --dport 22 -j ACCEPT",
        "-A OUTPUT -s 88.88.88.1/32 -d 10.66.1.3/32 -o eth1  -p tcp  -m tcp  --dport 22 -j ACCEPT",
        "-A OUTPUT -s 88.88.88.1/32 -d 10.66.2.2/32 -o eth1  -p tcp  -m tcp  --dport 22 -j ACCEPT",
        "-A OUTPUT  -o eth1  -j LOG --log-prefix \"default\" --log-level 1",
        "-A OUTPUT  -o eth1  -j DROP",
        "COMMIT"
      ],
      "BorderFW": [
        "# Created by Firelet for host BorderFW",
        "*filter",
        "-A INPUT -m state --state RELATED,ESTABLISHED -j ACCEPT",
        "-A INPUT -i lo -j ACCEPT",
        "-A INPUT -s 88.88.88.1/32 -d 172.16.2.223/32 -i eth0  -p tcp  -m tcp  --dport 22 -j ACCEPT",
        "-A INPUT -s 10.66.1.2/32 -d 10.66.1.1/32 -i eth1  -p tcp  -m tcp  --dport 443 -j ACCEPT",
        "-A INPUT -s 10.66.1.3/32 -d 10.66.1.1/32 -i eth1  -j LOG --log-prefix \"NoSmeagol\" --log-level 3",
        "-A INPUT -s 10.66.1.3/32 -d 10.66.1.1/32 -i eth1  -j DROP",
        "-A INPUT -s 10.66.1.3/32 -d 172.16.2.223/32 -i eth0  -p udp  -m udp  --dport 123 -j ACCEPT",
        "-A INPUT -s 10.66.2.2/32 -d 172.16.2.223/32 -i eth0  -p udp  -m udp  --dport 123 -j ACCEPT",
        "-A INPUT  -i eth0  -j LOG --log-prefix \"default\" --log-level 1",
        "-A INPUT  -i eth0  -j DROP",
        "-A INPUT  -i eth1  -j LOG --log-prefix \"default\" --log-level 1",
        "-A INPUT  -i eth1  -j DROP",
        "-A INPUT  -i eth2  -j LOG --log-prefix \"default\" --log-level 1",
        "-A INPUT  -i eth2  -j DROP",
        "-A FORWARD -m state --state RELATED,ESTABLISHED -j ACCEPT",
        "-A FORWARD -s 88.88.88.1/32 -d 172.16.2.223/32 -p tcp  -m tcp  --dport 22 -j LOG  --log-prefix \"ssh_all\" --log-level 0",
        "-A FORWARD -s 88.88.88.1/32 -d 172.16.2.223/32 -p tcp  -m tcp  --dport 22 -j ACCEPT",
        "-A FORWARD -s 88.88.88.1/32 -d 10.66.1.3/32 -p tcp  -m tcp  --dport 22 -j LOG  --log-prefix \"ssh_all\" --log-level 0",
        "-A FORWARD -s 88.88.88.1/32 -d 10.66.1.3/32 -p tcp  -m tcp  --dport 22 -j ACCEPT",
        "-A FORWARD -s 88.88.88.1/32 -d 10.66.2.2/32 -p tcp  -m tcp  --dport 22 -j LOG  --log-prefix \"ssh_all\" --log-level 0",
        "-A FORWARD -s 88.88.88.1/32 -d 10.66.2.2/32 -p tcp  -m tcp  --dport 22 -j ACCEPT",
        "-A FORWARD -s 10.66.1.3/32 -d 172.16.2.223/32 -p udp  -m udp  --dport 123 -j LOG  --log-prefix \"ntp\" --log-level 0",
        "-A FORWARD -s 10.66.1.3/32 -d 172.16.2.223/32 -p udp  -m udp  --dport 123 -j ACCEPT",
        "-A FORWARD  -j LOG  --log-prefix \"default\" --log-level 1",
        "-A FORWARD  -j DROP",
        "-A FORWARD  -j LOG  --log-prefix \"default\" --log-level 1",
        "-A FORWARD  -j DROP",
        "-A FORWARD  -j LOG  --log-prefix \"default\" --log-level 1",
        "-A FORWARD  -j DROP",
        "-A OUTPUT -m state --state RELATED,ESTABLISHED -j ACCEPT",
        "-A OUTPUT -o lo -j ACCEPT",
        "-A OUTPUT -s 10.66.1.1/32 -d 10.66.2.0/24 -o eth1  -p tcp  -m tcp  --dport 22 -j LOG --log-prefix \"ssh_mgmt\" --log-level 2",
        "-A OUTPUT -s 10.66.1.1/32 -d 10.66.2.0/24 -o eth1  -p tcp  -m tcp  --dport 22 -j ACCEPT",
        "-A OUTPUT -s 172.16.2.223/32 -d 10.66.1.3/32 -o eth0  -p udp  -m udp  --dport 123 -j ACCEPT",
        "-A OUTPUT  -o eth0  -j LOG --log-prefix \"default\" --log-level 1",
        "-A OUTPUT  -o eth0  -j DROP",
        "-A OUTPUT  -o eth1  -j LOG --log-prefix \"default\" --log-level 1",
        "-A OUTPUT  -o eth1  -j DROP",
        "-A OUTPUT  -o eth2  -j LOG --log-prefix \"default\" --log-level 1",
        "-A OUTPUT  -o eth2  -j DROP",
        "COMMIT"
      ]
    }

    debug('m', m)

    for hostname in m:
        for ok_line, my_line in zip(ok[hostname], m[hostname]):
            assert my_line == ok_line, "Incorrect rule built for %s:\ngot [%s]\nexpected [%s]" % (hostname, my_line,  ok_line )




#@with_setup(setup_dir, teardown_dir)
#def test_DemoGitFireSet_diff_table_simple():
#    """Run diff between compiled rules and empty remote confs"""
#    fs = DemoGitFireSet(repodir=testingutils.repodir)
#    new_confs = fs.compile_rules()
#    remote_confs = {}
#    dt = fs._diff(remote_confs, new_confs)
#    assert dt == '<p>The firewalls are up to date. No deployment needed.</p>'
    #FIXME:  deployment IS needed


@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_extract_iptables_rules():
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    #FIXME: the _get_confs implementation in DemoGitFireSet is broken
    fs._get_confs(keep_sessions=False)
    rules_d = fs._extract_ipt_filter_rules(fs._remote_confs)
    for hn, rules in rules_d.iteritems():
        assert len(rules) > 12,  rules
        assert len(rules) < 34,  rules
        for rule in rules:
            assert rule not in ('COMMIT', '*filter', '*nat')

#TODO: review this test, ensure it's using real data
@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_extract_iptables_rules_2():
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    remote_confs = {
        'InternalFW': {
            'iptables': {'filter': [], 'nat': []},
            'ip_a_s': {},
        },
        'BorderFW': {
            'iptables': {'filter': ['a', 'b', 'c'], 'nat': []},
            'ip_a_s': {},
        }
    }
    rules_d = fs._extract_ipt_filter_rules(remote_confs)
    assert rules_d == {'InternalFW': [], 'BorderFW': ['a', 'b', 'c']}, repr(rules_d)



@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_diff_table_generation_1():
    """Test diff with no changes"""
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    diff_dict = fs._diff({}, {})
    assert diff_dict == {}

@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_diff_table_generation_2():
    """Test diff with no changes"""
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    diff_dict = fs._diff({'InternalFW':['']}, {'InternalFW':['']})
    assert diff_dict == {}

@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_diff_table_generation_3():
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    diff_dict = fs._diff({'InternalFW':['old item', 'static item', 'old item2']},
                                   {'InternalFW':['static item', 'new item', 'new item2']})
    assert diff_dict == {'InternalFW': (['new item', 'new item2'], ['old item', 'old item2'])}

@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_diff_table_generation_all_fw_removed():
    """Test diff where all the firewalls has been removed.
    An empty diff should be generated."""
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    fs._get_confs(keep_sessions=False)
    existing_rules = fs._extract_ipt_filter_rules(fs._remote_confs)
    diff_dict = fs._diff(existing_rules,   {})
    assert diff_dict == {}, "An empty diff should be generated."

@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_diff_table_generation_all_fw_added():
    """Test diff right after all the firewalls has been added.
    An empty diff should be generated."""
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    comp_rules = fs.compile_rules()
    new_rules = {}
    for hn, b in comp_rules.iteritems():
        li = fs._build_ipt_restore_blocks((hn, b))
        new_rules[hn] = li
    diff_dict = fs._diff({}, new_rules)
    assert diff_dict == {}, "An empty diff should be generated."


# Used during development with test/rebuild.sh #
# to generate new sets of  test files #
#
#@with_setup(setup_dir, teardown_dir)
#def test_DemoGitFireSet_rebuild():
#    fs = DemoGitFireSet(repodir=testingutils.repodir)
#    comp_rules = fs.compile_rules()
#    for hn, b in comp_rules.iteritems():
#        li = fs._build_ipt_restore((hn, b))[1]
#        open("test/new-iptables-save-%s" % hn, 'w').write('\n'.join(li)+'\n')


@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_check():
    """Run diff between complied rules and remote confs using DemoGitFireSet
    Given the test files, the check should be ok and require no deployment"""
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    diff_dict = fs.check()
#    assert diff_dict == {},  repr(diff_dict)[:300]

    #FIXME: enable the test again




@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_deploy():
    """Run diff between complied rules and remote confs.
    Given the test files, the check should be ok and require no deployment"""
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    log.debug("Test deployment in %s" % repodir)
    fs.deploy()
    for h in fs._get_firewalls():
        ok = open(testingutils.repodir + '/iptables-save-%s' % h.hostname).readlines()
        r = open(testingutils.repodir + '/iptables-save-%s-x' % h.hostname).readlines()
        assert ok == r

    assert not fs.save_needed()
    diff_dict = fs.check()
    assert diff_dict == {}, "Check should be giving empty result instead of: %s" \
        % repr(diff_dict)[:300]


@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_deploy_then_check():
    """Deploy conf then run check again"""
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    assert not fs.save_needed()
    log.debug("Running deployment using repository in %s" % repodir)
    fs.deploy()
    log.debug("Deployed.")
    assert not fs.save_needed()
    log.debug("Running check...")

    diff_dict = fs.check()
    assert diff_dict == {}, "Check should be giving empty result instead of: %s" \
        % repr(diff_dict)[:300]
    assert not fs.save_needed()



#@with_setup(setup_dir, teardown_dir)
#def test_GitFireSet_deployment():
#    fs = GitFireSet(repodir=testingutils.repodir)
#    fs.deploy()

#@with_setup(setup_dir, teardown_dir)
#def test_DemoGitFireSet_deploy():
#    fs = DemoGitFireSet(repodir=testingutils.repodir)
#    dt = fs.deploy()
#    for h in fs.hosts:
#        r = map(str.rstrip, open(repodir + '/iptables-save-%s' % h.hostname))
#        ok = map(str.rstrip, open(repodir + '/iptables-save-%s-correct' % h.hostname))
#        for a, b in zip(r, ok):
#            assert a == b, "%s differs from %s in iptables-save-%s" % (a, b, h.hostname)
#
#

#@with_setup(setup_dummy_flssh)
#def test_get_confs_local_dummy():
#    from firelet.flssh import SSHConnector, MockSSHConnector
#
#    sshconn = SSHConnector(targets={'localhost':['127.0.0.1']} )
#    d  = sshconn.get_confs( )
#    assert 'localhost' in d
#    assert d['localhost']
#    assert d == {'localhost': [None, '127.0.0.1', {'filter': '-A INPUT -s 10.0.0.0/8 -p tcp -m tcp --dport 80 -j ACCEPT\n-A FORWARD -s 1.2.3.4/32 -d 5.6.7.8/32 -p tcp -m multiport --dports 22,80,443 -j ACCEPT\n-A OUTPUT -d 10.10.10.10/32 -p udp -m udp --dport 123 -j ACCEPT', 'nat': '-A POSTROUTING -o eth3 -j MASQUERADE'}, {'lo': ('127.0.0.1/8', '::1/128'), 'teredo': (None, 'fe80::ffff:ffff:ffff/64'), 'wlan0': ('192.168.1.1/24', 'fe80::219:d2ff:fe26:fb8e/64'), 'eth0': (None, None)}]}

# fs.services.update() testing

@raises(AssertionError)
@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_service_update_error1():
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    fs.services.update({})

@raises(Alert)
@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_service_update_incorrect_tcp_ports():
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    fs.services.update(dict(protocol='TCP', ports='foo,foo'), rid=0)

@raises(Alert)
@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_service_update_missing_tcp_ports():
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    fs.services.update(dict(protocol='TCP', ports=','), rid=0)

@raises(AssertionError)
@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_service_update_reversed_tcp_ports():
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    fs.services.update(dict(protocol='TCP', ports='10:1'), rid=0)

@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_service_update_tcp():
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    fs.services.update(dict(protocol='TCP', ports='8888', name='HTTP'), rid=0)
    assert fs.services[0].ports == '8888'

@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_service_update_ip():
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    fs.services.update(dict(protocol='IP', ports='', name='IP'), rid=0)
    assert fs.services[0].ports == ''

@raises(Alert)
@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_service_update_incorrect_icmp_type():
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    fs.services.update(dict(protocol='ICMP', ports='foo'), rid=0)

@raises(Alert)
@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_service_update_incorrect_protocol():
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    fs.services.update(dict(protocol='foo', ports=''), rid=0)

@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_service_update_icmp():
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    fs.services.update(dict(protocol='ICMP', ports='8', name='NewName'), rid=0)
    assert fs.services[0].ports == '8'
    assert fs.services[0].name == 'NewName'

# fs.services.update() testing

@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_service_add():
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    fs.services.add(dict(protocol='ICMP', ports='8', name='NewName'))

@raises(AssertionError)
@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_service_add_duplicate():
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    fs.services.add(dict(protocol='ICMP', ports='8', name='HTTP'))

# fs.rules.update() testing

@raises(AssertionError)
@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_rules_update_missing_rid():
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    fs.rules.update({})

@raises(Alert)
@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_rules_update_missing_rule():
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    fs.rules.update({}, rid=1000)

@raises(KeyError)
@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_rules_update_missing_param():
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    fs.rules.update({}, rid=0)

@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_rules_update():
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    d = dict(
        action='',
        desc='desc_foo',
        dst='',
        dst_serv='',
        enabled='',
        log_level='',
        name='Rule0',
        src='',
        src_serv='',
    )
    fs.rules.update(d, rid=0)

# fs.rules.add() testing

@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_rules_add():
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    d = dict(
        action='',
        desc='desc_foo',
        dst='',
        dst_serv='',
        enabled='',
        log_level='',
        name='Rule0',
        src='',
        src_serv='',
    )
    fs.rules.add(d, rid=0)

@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_rules_add_empty():
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    fs.rules.add({}, rid=0)

@raises(Alert)
@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_rules_add_duplicate():
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    d = dict(
        action='',
        desc='desc_foo',
        dst='',
        dst_serv='',
        enabled='',
        log_level='',
        name='ssh_all', # duplicate
        src='',
        src_serv='',
    )
    fs.rules.add(d, rid=0)

# fs.hosts.add() testing

@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_hosts_add():
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    d = dict(
        hostname='',
        iface='',
        ip_addr='',
        masklen='',
        local_fw='',
        network_fw='',
        mng=[],
        routed=[],
    )
    fs.hosts.add(d)

@raises(AssertionError)
@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_hosts_add_duplicate():
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    d = dict(
        hostname='InternalFW',
        iface='eth0',
        ip_addr='',
        masklen='',
        local_fw='',
        network_fw='',
        mng=[],
        routed=[],
    )
    fs.hosts.add(d)

# fs.hostgroups.add() testing

@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_hostgroups_add():
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    d = dict(
        name='',
        childs=[],
    )
    fs.hostgroups.add(d)

@raises(AssertionError)
@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_hostgroups_add_duplicate():
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    d = dict(
        name='AllSystems',
        childs=[],
    )
    fs.hostgroups.add(d)

# fs.hostgroups.update() testing

@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_hostgroups_update():
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    d = dict(name='foo', childs=[])
    fs.hostgroups.update(d, rid=0)

@raises(Alert)
@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_hostgroups_update_missing():
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    d = dict(name='foo', childs=[])
    fs.hostgroups.update(d, rid=1000)

# fs.networks.add() testing

@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_networks_add():
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    d = dict(
        name='',
        ip_addr='1.2.0.0',
        masklen=16,
    )
    fs.networks.add(d)

@raises(AssertionError)
@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_networks_add_duplicate():
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    d = dict(
        name='Internet',
        ip_addr='1.2.0.0',
        masklen=16,
    )
    fs.networks.add(d)

@raises(Exception)
@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_networks_add_incorrect_ipaddr():
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    d = dict(
        name='',
        ip_addr='foo',
        masklen=16,
    )
    fs.networks.add(d)

@raises(Exception)
@with_setup(setup_dir, teardown_dir)
def test_DemoGitFireSet_networks_add_incorrect_netmask():
    fs = DemoGitFireSet(repodir=testingutils.repodir)
    d = dict(
        name='',
        ip_addr='1.2.0.0',
        masklen='foo',
    )
    fs.networks.add(d)


# #  IP address handling  # #

def test_network_update():
    assert Network(['','255.255.255.255',8]).ip_addr == '255.0.0.0'
    assert Network(['','255.255.255.255',16]).ip_addr == '255.255.0.0'
    assert Network(['','255.255.255.255',24]).ip_addr == '255.255.255.0'
    assert Network(['','255.255.255.255',27]).ip_addr == '255.255.255.224'
    assert Network(['','255.255.255.255',28]).ip_addr == '255.255.255.240'
    assert Network(['','255.255.255.255',29]).ip_addr == '255.255.255.248'
    assert Network(['','255.255.255.255',30]).ip_addr == '255.255.255.252'


def test_network_contains_networks():
    assert Network(['', '255.255.255.255', 16]) in Network(['', '255.255.255.255', 8])
    assert Network(['', '255.255.255.255', 16]) in Network(['', '255.255.255.255', 16])
    assert Network(['', '255.255.255.255', 8]) not in Network(['', '255.255.255.255', 16])
    assert Network(['', '1.0.0.0', 17]) in Network(['', '1.0.0.0', 16])
    assert Network(['', '1.0.0.0', 16]) in Network(['', '1.0.0.0', 16])
    assert Network(['', '1.0.0.0', 15]) not in Network(['', '1.0.0.0', 16])
    assert Network(['', '42.42.42.42', 15]) not in Network(['','42.42.42.42', 16])
    assert Network(['', '42.42.42.42', 16]) in Network(['','42.42.42.42', 16])
    assert Network(['', '42.42.42.42', 17]) in Network(['','42.42.42.42', 16])

def test_network_contains_hosts():
    assert Host(['h', 'eth0', '1.1.1.1', 24, '1', '1', '1', [] ]) \
        in Network(['h', '1.1.1.0', 28])
    assert Host(['h', 'eth0', '1.1.1.15',24, '1', '1', '1', [] ]) \
        in Network(['h', '1.1.1.0', 28])
    assert Host(['h', 'eth0', '1.1.1.16',24, '1', '1', '1', [] ]) \
        not in Network(['h', '1.1.1.0', 28])
    assert Host(['h', 'eth0', '1.1.1.1',24, '1', '1', '1', [] ]) \
        in Network(['h', '1.1.1.0', 24])
    assert Host(['h', 'eth0', '1.1.1.1',24, '1', '1', '1', [] ]) \
        in Network(['h', '1.1.1.0', 8])
    assert Host(['h', 'eth0', '1.1.1.1',24, '1', '1', '1', [] ]) \
        not in Network(['h', '1.1.2.0', 24])
    assert Host(['h', 'eth0', '1.1.1.1',24, '1', '1', '1', [] ]) \
        not in Network(['h', '10.1.1.0', 8])

def test_host_contains_host():
    assert Host(['h', 'eth0', '1.1.1.1', 24, '1', '1', '1', [] ]) \
        in Host(['h', 'eth0', '1.1.1.1', 24, '1', '1', '1', [] ])

def test_host_contains_network():
    # only an Host can be in a Host, otherwise raise an Exception
    def tester():
        Network(['h', '1.1.1.0', 8]) in \
        Host(['h', 'eth0', '1.1.1.1', 24, '1', '1', '1', [] ])
    assert_raises(Exception, tester)

def test_compare():
    from netaddr import IPNetwork
    for x in xrange(0, 32):
        n=IPNetwork('255.1.1.1/%d' % x)
        ok = n.network
        mine = Network(['','255.1.1.1', x]).ip_addr
        log.debug( 'ok: %s mine: %s len: %d' % (ok,  mine, x))
        assert str(mine) == str(ok)


#def test_flattening():
#    hg2 = HostGroup(['name', [Host(['h', 'eth0', '1.1.1.1',24, '1', '1', '1', [] ])], ])
#    hg3 = HostGroup(['name2', [Network(['n', '2.2.2.0', 24]), hg2]])
#    hg = HostGroup(childs=[hg2, hg3])
#    assert ['h', 'h'] == [h.hostname for h in hg.hosts()]
#    assert ['n'] == [h.name for h in hg.networks()], repr(hg.networks())


@with_setup(setup_dir, teardown_dir)
def test_svg_map():
    fs = GitFireSet(repodir=testingutils.repodir)
    svg = draw_svg_map(fs)
    assert 'DOCTYPE svg PUBLIC' in svg, "No SVG output?"
    assert 'rivendell' in svg, "No rivendell in the map"


# #  Test JSON lib  # #

def json_loop(obj):
    return json.loads(json.dumps(obj, sort_keys=True))

def test_json1():
    d = {'string':'string', 's2':6, 's3':7.7, 's4':True, 's5':False}
    assert d == json_loop(d)

def test_json2():
    d = {'string':'string', 's2':6, 's3':7, 's4':True, 's5':False}
    assert d == json_loop(d)
    assert json.dumps(d) == """{"s3": 7, "s2": 6, "string": "string", \
"s5": false, "s4": true}"""

def test_json3():
    d = {'d1':{'d2':{'d3':{'d4':{'d5':{'this is getting':'boring'}}}}}}
    assert d == json_loop(d)
    assert json.dumps(d) == """{"d1": {"d2": {"d3": {"d4": {"d5": \
{"this is getting": "boring"}}}}}}"""

def test_json4():
    d = [x for x in xrange(42)]
    assert d == json_loop(d)

def test_json5():
    """Keys are casted to strings, integers are not preserved"""
    d = {1:1, 2:2, 3:3}
    assert d != json_loop(d)

@with_setup(setup_dir, teardown_dir)
def test_json_files():
    d = {'d1':{'d2':{'d3':{'d4':{'d5':{'this is getting':'boring'}}}}}}
    savejson('jfile', d, d=testingutils.repodir)
    nd = loadjson('jfile', d=testingutils.repodir)
    assert d == nd

# #  Test cartesian product  # #

def test_product_2_6():
    from itertools import product

    assert tuple(product([1,2,3,4,5,'O HI'],['a','b','c','d',42])) == (
        (1, 'a'), (1, 'b'), (1, 'c'), (1, 'd'), (1, 42), (2, 'a'), (2, 'b'), (2, 'c'), (2, 'd'), (2, 42),
        (3, 'a'), (3, 'b'), (3, 'c'), (3, 'd'), (3, 42), (4, 'a'), (4, 'b'), (4, 'c'), (4, 'd'), (4, 42),
        (5, 'a'), (5, 'b'), (5, 'c'), (5, 'd'), (5, 42), ('O HI', 'a'), ('O HI', 'b'), ('O HI', 'c'),
        ('O HI', 'd'), ('O HI', 42))


def test_product_2_5():
    from firelet.flutils import product

    assert tuple(product([1,2,3,4,5,'O HI'],['a','b','c','d',42])) == (
        (1, 'a'), (1, 'b'), (1, 'c'), (1, 'd'), (1, 42), (2, 'a'), (2, 'b'), (2, 'c'), (2, 'd'), (2, 42),
        (3, 'a'), (3, 'b'), (3, 'c'), (3, 'd'), (3, 42), (4, 'a'), (4, 'b'), (4, 'c'), (4, 'd'), (4, 42),
        (5, 'a'), (5, 'b'), (5, 'c'), (5, 'd'), (5, 42), ('O HI', 'a'), ('O HI', 'b'), ('O HI', 'c'),
        ('O HI', 'd'), ('O HI', 42))


# #  Bunch objects testing  # #

# Service bunch

def test_bunch_service1():
    d = dict(name='s1', protocol='NotAProtocol', ports='53')
    assert_raises(Exception, Bunch, d)

def test_bunch_service1b():
    d = dict(name='s1', protocol='TCP', ports='53')
    s = Bunch(**d)
    assert s.protocol == 'TCP'

def test_bunch_service2():
    d = dict(name='s1', protocol='TCP', ports='999999')
    assert_raises(Exception, Bunch, d)

def test_bunch_service3():
    d = dict(name='s1', protocol='TCP', ports='-1')
    assert_raises(Exception, Bunch, d)

def test_bunch_service4():
    d = dict(name='s1', protocol='TCP', ports='10:20:30')
    assert_raises(Exception, Bunch, d)

def test_bunch_service5():
    d = dict(name='s1', protocol='TCP', ports='30:20')
    assert_raises(Exception, Bunch, d)

def test_bunch_service6():
    d = dict(name='s1', protocol='TCP', ports='blah')
    assert_raises(Exception, Bunch, d)

def test_bunch_service7():
    d = dict(name='s1', protocol='TCP', ports='80')
    s = Bunch(**d)
    d = dict(name='s1', protocol='TCP', ports='blah')
    s.update(d) #FIXME: this should raise an Alert
    assert s.ports == 'blah'    # remove after the fix
#    assert_raises(Exception, s.update, d)

def test_bunch_service():
    s = Bunch(name='s1', protocol='UDP', ports='53')
    assert s.name == 's1', 'Incorrect bunch name'
    assert s.protocol == 'UDP', 'Incorrect bunch proto'
    assert s.ports == '53', 'Incorrect bunch ports'
    s.update({'name': 's2', 'protocol':'TCP', 'ports':'80'})
    assert s.name == 's2', 'Incorrect bunch name'
    assert s.protocol == 'TCP', 'Incorrect bunch proto'
    assert s.ports == '80', 'Incorrect bunch ports'
#    assert_raises(Alert, s.update,
#        {'name': 's2', 'protocol':'TCP', 'ports':'eighty    '}
#    )



# HostGroup Bunch

def test_bunch_hostgroup1():
    hg = HostGroup(['Servers'])
    assert hg.name == 'Servers'

def test_bunch_hostgroup2():
    hg = HostGroup(['Servers', 'a', 'b'])
    assert hg.childs == ['a', 'b']

#def test_bunch_hostgroup_flatten1():
#    hg = HostGroup(['Servers'])
#    dicts = [{}, {}, {}]
#    assert_raises(Exception, hg.flat, *dicts)

#def test_bunch_hostgroup_flatten2():
#    hg = HostGroup(['Servers'])
#    flat = hg.flat({'Servers':'a'}, {}, {})
#    print flat

#    dicts = [{}, {}, {}]
#    assert_raises(Exception, hg.flat, *dicts)


# Basic Bunch class

def test_bunch_repr():
    b = Bunch( c=42, a=3, b='44', _a=0)
    assert repr(b) == "{'a': 3, 'c': 42, 'b': '44', '_a': 0}", "Bunch repr is incorrect: %s" % repr(b)

def test_bunch_set_get():
    b = Bunch( c=42, a=3, b='44', _a=0)
    assert b.c == 42
    assert b['c'] == 42
    b.c = 17
    assert b.c == 17
    b['c'] = 18
    assert b.c == 18
    assert 'c' in b

def test_bunch_token():
    b = Bunch( c=42, a=3, b='44', _a=0)
    tok = b._token()
    b.validate_token(tok)
    assert_raises(Exception,  b.validate_token, '123456')

def test_bunch_update():
    b = Bunch( c=42, a=3, b='44', _a=0)
    d = dict(_a=1, a=2, b=3, c=4, extra=5)
    b.update(d)
    assert b.a == 2 and b.c == 4

def test_flag_true():
    for x in (1, True, '1', 'True', 'y', 'on' ):
        assert flag(x) == '1'

def test_flag_false():
    for x in (0, False, '0', 'False', 'n', 'off', ''):
        assert flag(x) == '0'

def test_flag_raise():
    for x in ('true', 'false'):
        assert_raises(Exception, flag, x)

# RSS generation
rss_msg = []
def setup_rss():
    from datetime import datetime
    global rss_msg
    rss_msg = [
        ['success', datetime(2011,01,01,10,10,10), 'Blah'],
        ['success', datetime(2011,01,01,10,10,20), 'Configuation saved: line'],
        ['success', datetime(2011,01,01,10,10,30), 'Configuration deployed.'],
    ]

@with_setup(setup_rss)
def test_get_rss_messages():
    d = get_rss_channels('messages', 'url', msg_list=rss_msg)
    assert 'items' in d
    items = d['items']
    assert len(items) == 3

@with_setup(setup_rss)
def test_get_rss_confsaves():
    d = get_rss_channels('confsaves', 'url', msg_list=rss_msg)
    assert 'items' in d
    items = d['items']
    assert len(items) == 1

@with_setup(setup_rss)
def test_get_rss_deployments():
    d = get_rss_channels('deployments', 'url', msg_list=rss_msg)
    assert 'items' in d
    items = d['items']
    assert len(items)
    assert 'title' in items[-1]
    assert 'Firelet success: Configuration deployed.' in items[-1]['title']

