from itertools import takewhile, chain, permutations
from weakref import ref
from operator import mul, attrgetter
from collections import Sequence, Iterable
from types import NoneType
from inspect import isgenerator
from werkzeug.utils import cached_property
from werkzeug.routing import BaseConverter

TOPBASE_OR_MIXIN_METHODS = ('get_names', 'get_child', 'get_cell', )
TOPBASE_OR_MIXIN_ATTRIBUTES = ( '__type__',
                                '__cell__',
                                '__converters__',
                                '__converter__')
class NOTEXIST(object):
    pass

NOTEXIST = NOTEXIST()

class class_property(property):
    def __get__(self, cls, owner):
        return self.fget.__get__(None, owner)()

def toseq(value, seqtype=tuple):
    if isinstance(value, basestring):
        return seqtype((value,))

    if isinstance(value, Sequence):
        return value

    if isinstance(value, Iterable) or isgenerator(value):
        return seqtype(value)

    else:
        return seqtype((value,))

class TreeViewBase(type):
    def mro(cls):
        try:
            return type.mro(cls)
        except TypeError:
            mro = [cls,]
            for base in cls.__bases__:
                if isinstance(base, type):
                    for superbase in type.mro(base):
                        if not superbase in mro:
                            mro.append(superbase)
                else:
                    mro.append(base)
            return mro

    def __new__(meta, name, bases, members):
        cls = type.__new__(meta, name, bases, members)
        if name == 'TreeView':
            return cls


        mixins = tuple(takewhile(lambda b:not issubclass(b, TreeView), bases))

        mixins_containing_topbase_methods = ((any((hasattr(mixin, meth)\
                                                    for mixin in mixins)), meth)\
                                              for meth\
                                              in TOPBASE_OR_MIXIN_METHODS if not meth in members)
        mixins_containing_topbase_attributes = ((any((hasattr(mixin, attr)\
                                                    for mixin in mixins)), attr)\
                                              for attr\
                                              in TOPBASE_OR_MIXIN_ATTRIBUTES if not attr in members)

        for in_mixin, name in mixins_containing_topbase_methods:
            if not in_mixin:
                method = getattr(TreeView, name)
                method = method.__get__(cls, cls)
                setattr(cls, name, method)

        mixins_containing_topbase_attributes = list(mixins_containing_topbase_attributes)
        for in_mixin, name in mixins_containing_topbase_attributes:
            if not in_mixin:
                setattr(cls, name, getattr(TreeView, name))

        if not hasattr(cls, '__type__'):
            raise AttributeError('TreeView subclasses must define `__type__` attribute')

        if cls.__type__ is object:
            raise TypeError('%s cannot be mapped because it is highest in types hierarchy'%object)

        return cls

class DummyMap(object):
    #flask Map compatible
    charset = 'utf-8'

class TreeView(object):
    __metaclass__ = TreeViewBase
    __type__ = NoneType
    __cell__ = {}
    __converters__ = {None:BaseConverter}
    __converter__ = None
    __sep__ = '/'
    __map__ = DummyMap()

    @property
    def __typename__(cls):
        if cls.__type__:
            return cls.__type__.__name__.lower()

    def get_names(self):
        if False:
            yield

    def get_child(self, name):
        pass

    def get_cell(self, item_name, cell_key):
        return None

    def __init__(self, obj, name, tree, parent=None):
        self.name = name
        self.obj = obj
        if isinstance(tree, ref):
            self._tree = tree
        else:
            self._tree = ref(tree)

        if isinstance(parent, ref):
            self._parent = parent

        elif parent is not None:
            self._parent = ref(parent)

        else:
            self._parent = parent

    @property
    def parent(self):
        if hasattr(self, '__the_one_and_only_parent_ref__'):
            return self.__the_one_and_only_parent_ref__
        if self._parent:
            return self._parent()

    @property
    def tree(self):
        if self._tree:
            return self._tree()

    @class_property
    @classmethod
    def viewmap(cls):
        if not '__viewmap__' in cls.__dict__: 
            setattr(cls, '__viewmap__', {})
            subclasses = cls.__subclasses__()
            if '__recursive__' in cls.__dict__:
                subclasses.append(cls)
            for sub in subclasses:
                if sub.__type__ is not NoneType:
                    cls.__viewmap__[sub.__type__] = sub
        return cls.__dict__['__viewmap__']

    def __iter__(self):
        if 'names' in self.__dict__:
            names = self.names
        else:
            names = self.get_names()

        for name in names:
            value = self.get_child(name)

            if value is self.obj:
                continue

            view = self.map(value,
                            name,
                            parent=self)
            if view is not NOTEXIST:
                yield (name, view)

    @cached_property
    def cell(self):
        return [self.get_cell(self.name, cell_key)\
                for cell_key\
                in self.tree.cell_keys]

    def map(self, obj, name, parent=None):
        tree = self.tree
        cls = type(self)
        objtype = type(obj)

        if tree.__view_classes__[cls] in tree.leaf_classes \
        and not getattr(tree.__view_classes__[cls], '__recursive__', False):
            return NOTEXIST


        if objtype in cls.viewmap:
            return cls.viewmap[objtype](obj, name, tree, parent=parent)

        if not tree.strict_types:
            subtyped = []
            for _type in cls.viewmap:
                if issubclass(objtype, _type):
                    subtyped.append(_type)
            if not len(subtyped):
                return NOTEXIST


            while len(subtyped) > 1:
                remove = []
                for t1, t2 in permutations(subtyped, 2):
                    if not issubclass(t1, t2):
                        remove.append(t1)

                for t in remove:
                    subtyped.remove(t)

            return cls.viewmap[subtyped[0]](obj, name, tree, parent=parent)

        return NOTEXIST

    def traverse(self, depth=0):
        for name, view in self:
            yield depth, name, view.cell, view
            for item in view.traverse(depth+1):
                yield item

    def pformat(self,                
                depth=0,
                indent='\t',
                format='%s%s: %s <%s>\n',
                indenter=mul):
        for depth, name, cell, view in self.traverse(depth):
            yield format%(indenter(indent, depth), view.path, cell, view)

    def pprint( self,
                depth=0,
                indent='\t',
                format='%s%s: %s <%s>\n',
                indenter=mul):
        for s in self.pformat(depth, indent, format):
            print s,

    @cached_property
    def names(self):
        return toseq(self.get_names())

    @cached_property
    def converter(self):
        return self.tree.converters[self.__converter__](self.__map__)

    def to_python(self, name):
        return self.converter.to_python(name)

    def to_path(self, name):
        return self.converter.to_url(name)

    @cached_property
    def path(self):
        if self.name is None:
            return ''
        else:
            ppath = self.parent.path
            sep = self.__sep__ if ppath else ''
            return ppath + sep + self.to_path(self.name)

    def get(self, path):
        '''
        returns (ViewItem, tail)
        if item does not exists - 
        return (NOTEXIST, 'non.existent.part')
        '''
        name, sep, tail = path.partition(self.__sep__)
        
        for viewname, view in self:
            if view.to_path(viewname) == name:
                view.__the_one_and_only_parent_ref__ = self
                if not tail:
                    return view, tail
                v, t = view.get(tail)

                if v is NOTEXIST and t is None:
                    return v, tail
                return v, t

        return NOTEXIST, None
    
    @cached_property
    def is_container(self):
        for i in self:
            return True
        return False


class ObjectViewMixin:
    def get_names(self):
        for k in self.obj.__dict__.iterkeys():
            if not k.startswith('__'):
                yield k

    def get_child(self, name, default=None):
        return getattr(self.obj, name)


class DictViewMixin:
    __type__ = dict
    def get_names(self):
        for k in self.obj.iterkeys():
            yield k

    def get_child(self, name, default=None):
        return self.obj[name]


class TreeRootView(TreeView):
    __type__ = dict
    def get_names(self):
        return self.obj.iterkeys()

    def get_child(self, name):
        return self.obj[name]


class Tree(object):
    def __init__(self,
                 objects,
                 root_view_classes,
                 root_class = TreeRootView,
                 leaf_classes=(),
                 converters={},
                 sep=None,
                 strict_types=False):

        self._root_view_classes = root_view_classes
        self._root_class = root_class
        self._leaf_classes = leaf_classes
        self.objects = objects
        self.__view_classes__ = {}
        self.sep = sep

        if converters:
            self.converters.update(converters)

        self.strict_types = strict_types

    def rebased_class(self, cls, to, _from=TreeView):
        bases = []
        for base in cls.__bases__:
            if base is _from:
                bases.append(to)
            elif issubclass(base, TreeView):
                bases.append(self.rebased_class(base, to, _from))
            else:
                bases.append(base)
        t = type(cls.__name__, tuple(bases), dict(cls.__dict__))
        self.__view_classes__[t] = base
        return t

    def rebase_subs(self, _from, _to):
        if _from in self.leaf_classes:
            return

        sub_types = []
        if getattr(_from, '__recursive__', False):
            sub_types.append(_from.__type__)

        for sub in _from.__subclasses__():
            if not sub.__type__ in sub_types:
                sub_types.append(sub.__type__)
            else:
                raise TypeError('Tree view hieararchy cant contain same types at one level')

            new_bases = tuple((b if not b is _from else _to for b in sub.__bases__))
            new_sub = type(sub.__name__, new_bases, dict(sub.__dict__))
            self.__view_classes__[new_sub] = sub
            self.rebase_subs(sub, new_sub)

    @cached_property
    def root_view_classes(self):
        sub_roots = []
        sub_types = []
        for sub_root in toseq(self._root_view_classes):
            sub_roots.append(self.rebased_class(sub_root, self.root_class))
            if sub_root.__type__ not in sub_types:
                sub_types.append(sub_root.__type__)
            else:
                raise TypeError('Tree view hieararchy cant contain same types at one level')
            self.rebase_subs(sub_root, sub_roots[-1])
        return sub_roots

    @cached_property
    def leaf_classes(self):
        return toseq(self._leaf_classes)

    @cached_property
    def root_class(self):
        '''
        creates new copy of root_class with copies of _root_view_classes based on root_class
        '''
        members = dict(self._root_class.__dict__)
        if self.sep:
            members['__sep__'] = self.sep
        root_view_class = type(self._root_class.__name__,
                         self._root_class.__bases__,
                         members
                         )
        self.__view_classes__[root_view_class] = self._root_class
        return root_view_class

    @cached_property
    def root(self):
        self.root_view_classes
        return self.root_class(self.objects, None, self)

    def __iter__(self):
        return iter(self.root)

    def _aggregate_typestree_attr(self, roots, attrname):
        #views here corresponds to dict.view***()
        getter = attrgetter(attrname)
        name, sep, tail = attrname.partition('.')
        for root in roots:
            if name in root.__dict__:
                try:
                    yield getter(root)
                except AttributeError:
                    pass

            subs = root.__subclasses__()
            for sub in subs:
                if sub in self.leaf_classes:
                    continue
                if attrname in sub.__dict__:
                    try:
                        yield getter(sub)
                    except AttributeError:
                        pass

            for item in self._aggregate_typestree_attr(subs, attrname):
                yield item

    def aggregate_typestree_dict(self, roots, attrname):
        #views here corresponds to dict.view***()
        for d in self._aggregate_typestree_attr(roots, attrname):
            yield d.viewitems()

    def aggregate_typestree_list(self, roots, attrname):
        items = []
        for item in chain(*self._aggregate_typestree_attr(roots, attrname)):
            if not item in items:
                yield item
                items.append(item)

    def _header_views(self):
        return self.aggregate_typestree_dict(self.root_view_classes, '__cell__')

    def _converters(self):
        return self.aggregate_typestree_dict(self.root_view_classes, '__converters__')

    @cached_property
    def converters(self):
        convs = {}
        for name, conv in chain(*self._converters()):
            if not name in convs:
                convs[name] = conv
        return convs

    @cached_property
    def _cell(self):
        keys = []
        labels = []
        for key, label in chain(*self._header_views()):
            if not label in labels:
                keys.append(key)
                labels.append(label)
        return keys, labels

    @cached_property
    def cell_labels(self):
        return self._cell[1]

    @cached_property
    def cell_keys(self):
        return self._cell[0]

    def traverse(self, **kwargs):
        return self.root.traverse(**kwargs)

    def pformat(self, **kwargs):
        return self.root.pformat(**kwargs)

    def pprint(self, **kwargs):
        self.root.pprint(**kwargs)

    def get(self, path):
        return self.root.get(path.strip(self.root.__sep__))
