from types import NoneType
from collections import OrderedDict as odict

import unittest
from pprint import pprint

from flask.ext.introspect import TreeView, Tree, DictViewMixin, ObjectViewMixin, NOTEXIST

class O(object):
    def __init__(self, **kwargs):
        self.__dict__.update(kwargs)

class O1(object):
    def __init__(self, **kwargs):
        self.__dict__.update(kwargs)

class SortedNamesObjectViewMixin(ObjectViewMixin):
    def get_names(self):
        names = (k for k in self.obj.__dict__.iterkeys() if not k.startswith('__'))
        return sorted(names)

class SortedNamesDictViewMixin(DictViewMixin):
    def get_names(self):
        return sorted(self.obj.iterkeys())

class TestIntegrational(unittest.TestCase):
    '''Use this case for learning'''

    def assertTree(self, tree, spec):
        '''
        Tree matching tester
        '''
        traverse = list(tree.traverse())
        self.assertTrue(len(traverse) == len(spec))
        for ((item_depth,
              item_name,
              item_headers,
              view_object),
             (req_depth,
              req_name,
              req_headers,
              req_view_class_name)) in zip(traverse, spec):
            self.assertTrue(item_depth==req_depth)
            self.assertTrue(item_name==req_name)
            self.assertTrue(item_headers==req_headers)
            self.assertTrue(type(view_object).__name__==req_view_class_name)

    def test_two_level_tree(self):
        '''
        Simple two level tree.
        Root objects are of type O and all attributes of O of type O1 are tree leafs
        '''
        class TopView(SortedNamesObjectViewMixin, TreeView):
            '''
              `__type__ defines what subitem type should be rendered with this view
            '''
            __type__ = O
            '''__cell__ is OrderedDict with key and header name
               Tree.cell_keys contains list of unique cell_keys 
               and Tree.cell_labels contains list of unique cell_labels in corresponding order
            '''
            __cell__ = odict((('a', 'The a'),))

            def get_cell(self, objname, key):
                '''Return cell value for key
                   if required we can test self.obj, current `objname` in root hierarchy `key` name
                   or self.parent (view)
                   '''
                return getattr(self.obj, key, None)

            def get_names(self):
                '''- return single value or list or yield
                   - this list is not required (but allowed) to make prior filtering
                     Note: type based filtering will occur later
                   Note: `ObjectViewMixin` defines its own (obj.__dict__.iterkeys()) `get_names`
                   method so as `DictViewMixin`. So this should be redefined only in special cases
                '''
                return (('sub', 'sub3'))

        class SubView(SortedNamesObjectViewMixin, TopView):
            '''
            Attention: `names`, `get_cell`, `get_children`, `__type__` and `__cell__`
            does not inherited from class hierarchy branch that leads to TreeView
            thats why that methods/attributes must be defined in mixin 
            or directly in TreeView subclass
            Note: Mixins SHOULD BE old-style classes for not to create MRO conflicts
            '''
            __type__ = O1
            __cell__ = odict((('b', 'The b'), ('a', 'The a'),))

        '''tree accepts dict of root objects and top hierarchy class (or classes list)
           type(s) of dict items must correlate with top classes __type__ attr
        '''
        tree = Tree({'root':O(a='root header value for `a` key',
                              sub=O1(),
                              sub2=O(subsub={'a':'b'}),
                              sub3=O1(subsub={'a':'b'}))},
                    TopView)

        self.assertTrue(tree.cell_keys == ['a', 'b'])
        self.assertTrue(tree.cell_labels == ['The a', 'The b'])

        
        self.assertTree(tree, [(0, 'root', ['root header value for `a` key', None], 'TopView'),
                               (1, 'sub', [None, None], 'SubView'),
                               (1, 'sub3', [None, None], 'SubView')])

    def test_two_level_plus_recursive_tree(self):
        '''
        Recursion tree with O or O1 objects as leafs
        or sub O1 dicts as leafs
        '''
        class OView(SortedNamesObjectViewMixin, TreeView):
            __type__ = O
            __cell__ = odict((('a', 'The a'),))
            __recursive__ = True

            def get_cell(self, objname, key):
                return getattr(self.obj, key, None)

        class SubView(SortedNamesObjectViewMixin, OView):
            __type__ = O1
            __cell__ = odict((('b', 'The b'), ('a', 'The a'),))

            def get_cell(self, objname, key):
                '''
                legacy mixins does not define `get_cell` since it is specific for each object and level
                so it is a good decision to define own get_cell for each level
                '''
                return getattr(self.obj, key, None)

            def get_child(self, name):
                '''
                just an example of get child
                '''
                return getattr(self.obj, name)
                # `names` should not return not existent names
                # or `get_child` should get rid of it

        class DictSubView(SortedNamesDictViewMixin, SubView):
            def get_cell(self, objname, key):
                return self.obj.get(key, None)

        tree = Tree({'root':O(a='root header value for `a` key',
                              sub=O1(),
                              sub2=O(subsub={'a':'Dict `a` header'}),
                              sub3=O1(subsub={'b':'Dict `b` header'}),
                              sub4=O(subsub=O1(b='sub4 header value for `b` key',
                                               a='sub4 header value for `a` key'
                                                )),

                    )},
                    OView)

        self.assertTree(tree, [(0, 'root', ['root header value for `a` key', None], 'OView'),
                               (1, 'sub', [None, None], 'SubView'),
                               (1, 'sub2', [None, None], 'OView'),
                               (1, 'sub3', [None, None], 'SubView'),
                               (2, 'subsub', [None, 'Dict `b` header'], 'DictSubView'),
                               (1, 'sub4', [None, None], 'OView'),
                               (2, 'subsub', ['sub4 header value for `a` key',
                                              'sub4 header value for `b` key'], 'SubView')]
                        )

    def test_limit_tree_bottom(self):
        class OView(SortedNamesObjectViewMixin, TreeView):
            __type__ = O
            __cell__ = odict((('a', 'The a'),))
            __recursive__ = True

            def get_cell(self, objname, key):
                return getattr(self.obj, key, None)

        class SubView(SortedNamesObjectViewMixin, OView):
            __type__ = O1
            __cell__ = odict((('b', 'The b'), ('a', 'The a'),))

            def get_cell(self, objname, key):
                return getattr(self.obj, key, None)



        class DictSubView(SortedNamesDictViewMixin, SubView):
            __cell__ = odict((('c', 'The c'),))
            def get_cell(self, objname, key):
                return self.obj.get(key, None)

        '''
        When tree should be generated partially
        `leaf_classes` kwarg allows to limit bottom levels
        and named class(es) become bottom views
        no headers or converters will be processed for views below `leaf_classes`
        and all objects below are NOTEXIST
        this allow to generate some parts of tree as one piece with `traverse()`
        and others prepared for xhr extending
        '''
        tree = Tree({'root':O(a='root header value for `a` key',
                              sub=O1(),
                              sub2=O(subsub={'a':'Dict `a` header'}),
                              sub3=O1(subsub={'b':'Dict `b` header'}),
                              sub4=O(subsub=O1(b='sub4 header value for `b` key',
                                               a='sub4 header value for `a` key'
                                                )),

                    )},
                    OView,
                    leaf_classes=SubView,
                    )
        self.assertTree(tree, [(0, 'root', ['root header value for `a` key', None], 'OView'),
                               (1, 'sub', [None, None], 'SubView'),
                               (1, 'sub2', [None, None], 'OView'),
                               (1, 'sub3', [None, None], 'SubView'),
                               (1, 'sub4', [None, None], 'OView'),
                               (2, 'subsub', ['sub4 header value for `a` key', 'sub4 header value for `b` key'], 'SubView')])

    def test_get_item_by_path(self):
        class OView(SortedNamesObjectViewMixin, TreeView):
            __type__ = O
            __cell__ = odict((('a', 'The a'),))
            __recursive__ = True

            def get_cell(self, objname, key):
                return getattr(self.obj, key, None)

        class SubView(SortedNamesObjectViewMixin, OView):
            __type__ = O1
            __cell__ = odict((('b', 'The b'), ('a', 'The a'),))

            def get_cell(self, objname, key):
                return getattr(self.obj, key, None)



        class DictSubView(SortedNamesDictViewMixin, SubView):
            __cell__ = odict((('c', 'The c'),))
            def get_cell(self, objname, key):
                return self.obj.get(key, None)
        
        obj = {'root':O(a='root header value for `a` key',
                              sub=O1(),
                              sub2=O(subsub={'a':'Dict `a` header'}),
                              sub3=O1(subsub={'b':'Dict `b` header'}),
                              sub4=O(subsub=O1(b='sub4 header value for `b` key',
                                               a='sub4 header value for `a` key'
                                                )),
                    )}
        tree = Tree(obj, OView, leaf_classes=SubView, sep='.')
        '''
        Separator `TreeView.__sep__` can be different for each level
        '''
        item, tail = tree.get('root.sub3.subsub')
        self.assertTrue(item is NOTEXIST)
        self.assertTrue(tail == 'subsub')

        item, tail = tree.get('root.sub3')
        self.assertTrue(item.cell == [None, None])
        
        tree = Tree(obj, OView, sep='.')
        item, tail = tree.get('root.sub3.subsub.sub')
        self.assertTrue(item is NOTEXIST)
        self.assertTrue(tail == 'sub')
        
        tree = Tree(obj, OView, sep='.')
        item, tail = tree.get('root.sub3.subsub')
        
        self.assertTrue(item.obj == {'b': 'Dict `b` header'})
        self.assertTrue(item.cell == [None, 'Dict `b` header', None])
        self.assertTrue(item.path == 'root.sub3.subsub')

    def test_converter(self):
        from werkzeug.routing import UnicodeConverter

        from base64 import (urlsafe_b64encode as encodestring,
                            urlsafe_b64decode as decodestring)

        class Base64Converter(UnicodeConverter):
            def to_python(self, value):
                tail = len(value) % 4 is 0 and 0 or 4 - len(value) % 4
                return decodestring(str(value) + tail*'=')

            def to_url(self, value):
                return encodestring(value).rstrip('=')            

        class OView(SortedNamesObjectViewMixin, TreeView):
            __type__ = O
            __cell__ = odict((('a', 'The a'),))
            __recursive__ = True

            def get_cell(self, objname, key):
                return getattr(self.obj, key, None)

        class SubView(SortedNamesObjectViewMixin, OView):
            __type__ = O1
            __cell__ = odict((('b', 'The b'), ('a', 'The a'),))
            '''
            path separator can be redefined for sublevels
            `sep` kwarg for `Tree.__init__` will not be respected in this case
            '''
            __sep__ = '.'


            def get_cell(self, objname, key):
                return getattr(self.obj, key, None)



        class DictSubView(SortedNamesDictViewMixin, SubView):
            __cell__ = odict((('c', 'The c'),))
            '''
            Converter defines name conversion rules for 
            `get` and `path` methods
            '''
            __converter__ = 'base64'

            '''Tree converters will be composed from TreeView subclasses (e.g. levels)
               `__converters__` attrs and `converters` kwarg for `Tree.__init__`
               So if hierarchy tree is complex and used in multiple tree generation processes
               it is better to define converters at level where it is used
               In simple cases its enough to set `converters` kwarg

            '''
            __converters__ = {'base64':Base64Converter}

            def get_cell(self, objname, key):
                return self.obj.get(key, None)
        
        obj = {'root':O(a='root header value for `a` key',
                              sub=O1(),
                              sub2=O(subsub={'a':'Dict `a` header'}),
                              sub3=O1(subsub={'b':'Dict `b` header'}),
                              sub4=O(subsub=O1(b='sub4 header value for `b` key',
                                               a='sub4 header value for `a` key'
                                                )),
                    )}
        tree = Tree(obj, OView, sep='.')

        item, tail = tree.get('root.sub3.c3Vic3Vi')
        self.assertTrue(item.obj == {'b': 'Dict `b` header'})

    def test_strict_types(self):
        class O2(O1):
            pass

        class OView(SortedNamesObjectViewMixin, TreeView):
            __type__ = O
            __cell__ = odict((('a', 'The a'),))
            __recursive__ = True

            def get_cell(self, objname, key):
                return getattr(self.obj, key, None)

        class SubView(SortedNamesObjectViewMixin, OView):
            __type__ = O1
            __cell__ = odict((('b', 'The b'), ('a', 'The a'),))

            def get_cell(self, objname, key):
                return getattr(self.obj, key, None)

        class DictSubView(SortedNamesDictViewMixin, SubView):
            __cell__ = odict((('c', 'The c'),))
            __recursive__ = True
            def get_cell(self, objname, key):
                return self.obj.get(key, None)
        
        obj = {'root':O(a='root header value for `a` key',
                              sub=O2(),
                              sub2=O1(subsub={'a':'Dict `a` header', 'd':{'e':'f'}}),
                              sub3=O1(subsub={'b':'Dict `b` header'}),
                              sub4=O(subsub=O1(b='sub4 header value for `b` key',
                                               a='sub4 header value for `a` key'
                                                )),
                    )}
        tree = Tree(obj,
                    OView,
                    sep='.',
                    )
        '''
        With `strict_types=False` (default) an object of `__type__` subtype treated as suitable child
        '''
        self.assertTree(tree, [(0, 'root', ['root header value for `a` key', None, None], 'OView'),
                               (1, 'sub', [None, None, None], 'SubView'),
                               (1, 'sub2', [None, None, None], 'SubView'),
                               (2, 'subsub', ['Dict `a` header', None, None], 'DictSubView'),
                               (3, 'd', [None, None, None], 'DictSubView'),
                               (1, 'sub3', [None, None, None], 'SubView'),
                               (2, 'subsub', [None, 'Dict `b` header', None], 'DictSubView'),
                               (1, 'sub4', [None, None, None], 'OView'),
                               (2, 'subsub', ['sub4 header value for `a` key', 'sub4 header value for `b` key', None], 'SubView')]
                    )
        tree = Tree(obj,
                    OView,
                    sep='.',
                    strict_types=True
                    )
        '''
        With `strict_types=True` only object with exact __type__ are suitable children
        '''
        self.assertTree(tree, [(0, 'root', ['root header value for `a` key', None, None], 'OView'),
                               (1, 'sub2', [None, None, None], 'SubView'),
                               (2, 'subsub', ['Dict `a` header', None, None], 'DictSubView'),
                               (3, 'd', [None, None, None], 'DictSubView'),
                               (1, 'sub3', [None, None, None], 'SubView'),
                               (2, 'subsub', [None, 'Dict `b` header', None], 'DictSubView'),
                               (1, 'sub4', [None, None, None], 'OView'),
                               (2, 'subsub', ['sub4 header value for `a` key', 'sub4 header value for `b` key', None], 'SubView')])

    def test_strict_types_tree_with_multiple_subtype_views(self):
        class O2(O1):
            pass

        class O3(O2):
            pass

        class O4(O3):
            pass

        class OView(SortedNamesObjectViewMixin, TreeView):
            __type__ = O
            __cell__ = odict((('a', 'The a'),))
            __recursive__ = True

            def get_cell(self, objname, key):
                return getattr(self.obj, key, None)

        class O2View(SortedNamesObjectViewMixin, OView):
            __type__ = O2
            __cell__ = odict((('b', 'The b'), ('a', 'The a'),))

            def get_cell(self, objname, key):
                return getattr(self.obj, key, None)

        class O3View(SortedNamesObjectViewMixin, OView):
            __type__ = O3
            __cell__ = odict((('b', 'The b'), ('a', 'The a'),))

            def get_cell(self, objname, key):
                return getattr(self.obj, key, None)

        class DictSubView(SortedNamesDictViewMixin, O3View):
            __cell__ = odict((('c', 'The c'),))
            __recursive__ = True
            def get_cell(self, objname, key):
                return self.obj.get(key, None)

        '''
        In case when two levels are supertypes for object - lowest type will be selected
        '''
        obj = {'root':O(a='root header value for `a` key',
                              sub=O2(),
                              sub2=O4(subsub={'a':'Dict `a` header', 'd':{'e':'f'}}),
                              sub3=O1(subsub={'b':'Dict `b` header'}),
                              sub4=O(subsub=O4(b='sub4 header value for `b` key',
                                               a='sub4 header value for `a` key'
                                                )),
                    )}
        tree = Tree(obj,
                    OView,
                    sep='.',
                    )


        self.assertTree(tree, [(0, 'root', ['root header value for `a` key', None, None], 'OView'),
                               (1, 'sub', [None, None, None], 'O2View'),
                               (1, 'sub2', [None, None, None], 'O3View'),
                               (2, 'subsub', ['Dict `a` header', None, None], 'DictSubView'),
                               (3, 'd', [None, None, None], 'DictSubView'),
                               (1, 'sub4', [None, None, None], 'OView'),
                               (2, 'subsub', ['sub4 header value for `a` key', 'sub4 header value for `b` key', None], 'O3View')])


        tree = Tree(obj,
                    OView,
                    sep='.',
                    strict_types=True
                    )
        self.assertTree(tree, [(0, 'root', ['root header value for `a` key', None, None], 'OView'),
                               (1, 'sub', [None, None, None], 'O2View'),
                               (1, 'sub4', [None, None, None], 'OView')])


if __name__ == '__main__':
  from unittest import main
  main()