# coding: utf-8
"""
 :copyright: (c) 2011 Philipp Benjamin Köppchen
 :license: GPLv3, see LICENSE for more details.
"""
from __future__ import with_statement, absolute_import

import os.path
from glob import glob

import flask
from flask import render_template, redirect, request, url_for


class Blueprint(flask.Blueprint):
    """ The Blueprint to register with your application
    """

    def __init__(self):
        flask.Blueprint.__init__(self, 'locust', __name__,
                                 template_folder='templates')
        self.route('/', methods=["GET", "POST"])(self.handle_migrations)

    def get_migrations_path(self):
        """ Override, to return a path where the migrations can be found """
        raise NotImplementedError("must be overriden")

    def execute_sql(self, sql, *args):
        """ Override, to execute sql.

        Must return a list rows (tuples)
        """
        raise NotImplementedError("must be overriden")

    def commit(self):
        """ Override, to commit the changes """
        pass

    def handle_migrations(self):
        """ Displays all outstanding migrations, and when POSTed to, executes
            them.
        """
        if request.method == 'POST':
            self.execute_all()
            return redirect(url_for('.handle_migrations'))

        migrations = [name for name, _ in self.get_necessary_migrations()]
        return render_template('locust/migrations.html.jinja',
                                                         migrations=migrations)

    def get_executed_migrations(self):
        """ Returns a list of names of all executed migrations.
        """
        self.ensure_table()
        rows = self.execute_sql(
                             'select `name` from `migrations` order by `name`')
        return [name for name, in rows]

    def get_avialable_migrations(self):
        """ Returns a list of (name, statments) tuples for every migration that
        is aviable (as file in migrations_dir)
        """
        filenames = glob(os.path.join(self.get_migrations_path(), '*.sql'))
        migrations = []

        for filename in sorted(filenames):
            with open(filename) as fp:
                migration_name = unicode(os.path.basename(filename))
                statements = fp.read().split('\n--\n')
                migrations.append((migration_name, statements))

        return migrations

    def get_necessary_migrations(self):
        """ Returns a list of (name, statements) tuples, like
        `get_avialable_migrations`, but returns only migrations that have not
        been executed.
        """
        executed = set(self.get_executed_migrations())
        avialable = self.get_avialable_migrations()
        return [(name, statements) for name, statements in avialable
                                                      if name not in executed]

    def execute_all(self):
        """ Executes all neccessary migrations. """
        self.ensure_table()
        for name, statements in self.get_necessary_migrations():
            for statement in statements:
                self.execute_sql(statement)
            self.execute_sql("insert into `migrations` (`name`) values (%s)",
                                                                          name)
        self.commit()

    def ensure_table(self):
        """ creates the migrations table, if neccessary. """
        self.execute_sql('''create table if not exists `migrations` (
                         `name` varchar(50) not null, primary key (`name`))''')
