#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
"""
    This extension is mainly for using Mongokit in a flask project more
    naturally.
    To use this extension, define all your document classes inherited from
    flask_supmongokit.BaseDocument instead of the mongokit.Document
    
    Why use this extension instead of raw mongokit?

    We don't want to always use connection.User.find_one()
    if we can just type User.find_one() as we know that User is registered with a
    certain database and collection.
    More over, we don't want to call the connection.register method every time we define a new document
    SupMongokit did that for you behind the scene.

    Actually, we can have the host name, port name, database name provided in the configuration object
    Use  `MONGODB_HOST`,`MONGODB_PORT`,`MONGODB_DATABASE` in the config object .
    they all ship with a default value if not configured, host is 'localhost', port is 27017 by default and
    database name is 'default'.

    Last but not least, we'd like to give each document a default collection name based on the document name,
    conform to a certain rule ,
    In SupMongokit, the rule is simple , underscore and pluralize the document's class name.
    And if you insist to give the database and collection name manually, you can do that in the way
    you do in Mongokit, define __collection__ and __database__ attribute in your document class

    Supmongokit supports multi connections to different mongod instances
    if you need set up another connection , call the `add_connection` method
    with connectionname, host , port and an optional default dbname parameter.
    By default, Models will be registered to the default connection configured in your
    flask config object, to register some model to a certain connection you added,
    declare the `__connection__` class field in your model with is the same as the
    parameter you pass in to the `add_connection` method so that Supmongokit will
    recognize it.


    SupMongokit is perfectly compatible with Mongokit,
    the SupMongokit instance is just a wrapper around the Mongokit connection instance,
    you can use it as the raw Mongokit instance directly.
     So if you need more features not found in SupMongokit,
    feel happy to refer to MongoKit, even the native PyMongo driver for more advance usage.
"""
from uuid import uuid4

from bson import ObjectId
from mongokit import Connection, Document
from mongokit.document import DocumentProperties

from .helper import pluralize, underscore
from .error import NotFound, ConnectionNameConflict


__version__ = '0.1.0'
__author__ = 'Vita John (hwwangwang@gmail.com)'


class SupMongokit(object):
    def __init__(self, app=None):
        """
        `connection` attribute is the default connection
        `connection_store` attribute is a dict object stored all connection instances
        `dbname_dict` stores default database name for each connection
        """
        if app is not None:
            self.init_app(app)
        else:
            self.connection = None
            self._app = app
        self.connection_store = {}
        self.dbname_dict = {}

    def init_app(self, app):
        app.config.setdefault('MONGODB_HOST', '127.0.0.1')
        app.config.setdefault('MONGODB_PORT', 27017)
        app.config.setdefault('MONGODB_DATABASE', 'default')
        self.connection = Connection(app.config.get('MONGODB_HOST'),
                                     app.config.get('MONGODB_PORT'))
        self._app = app
        self.connection_store.update({'default': self.connection})
        for model in _tobe_registered:
            if model is not BaseDocument:
                if hasattr(model, '__connection__'):
                    self.connection_store[model.__connection__].register(model)
                else:
                    self.connection.register(model)
                    model.__connection__ = 'default'
                if not hasattr(model, '__database__'):
                    model.__database__ = self.dbname_dict.get(model.__connection__,
                                                              app.config.get('MONGODB_DATABASE'))
        BaseDocument.sup_mongokit = self
        app.extensions = getattr(app, 'extensions', {})
        app.extensions['supmongokit'] = self

    def __getattr__(self, item):
        return getattr(self.connection, item)

    def __getitem__(self, conn):
        return self.connection_store.get(conn)

    def add_connection(self, name, host, port, default_dbname=None):
        if name in self.connection_store:
            raise ConnectionNameConflict('A connection with name {} already eixsts'.format(name))
        self.connection_store[name] = Connection(host, port)
        if default_dbname:
            self.dbname_dict[name] = default_dbname


_tobe_registered = []


class MetaDoc(DocumentProperties):
    """The MetaDoc metaclass is used to give each Document a default __collection__ name
    based on the Document's classname,
    The rule is simple ,firstly underscore and then pluralize
    e.g: User -> users GaExtension -> ga_extensions
    """

    def __init__(cls, name, bases, ns):
        super(MetaDoc, cls).__init__(name, bases, ns)
        if not name.startswith('Callable'):
            if '__collection__' not in cls.__dict__:
                cls.__collection__ = pluralize(underscore(cls.__name__))

            # #Incase that you import the model class before we initialize supmongokit,
            # # We add this cls to the _tobe_registerd list, so that when we
            # #iniitalize the app, it can be registerd then.
            _tobe_registered.append(cls)
            # if you firstly import the model class after we initialize supmongokit,
            # Then the BaseDocument already has the conn attribute,
            # So we also register this class into the connection. pretty nifty!
            # # by default, the models is registered to conneciton set in config.py
            # # but if we'd like to register to a diffrenet connection by name,
            # # just specify the __connection__ attriubte in model

            if hasattr(cls, 'sup_mongokit'):
                # print cls.connection_store
                if hasattr(cls, '__connection__'):
                    cls.conn = cls.sup_mongokit.connection_store[cls.__connection__]
                    cls.conn.register(cls)
                else:
                    cls.__connection__ = 'default'
                    cls.sup_mongokit.connection_store[cls.__connection__].register(cls)
                if not hasattr(cls, '__database__'):
                    cls.__database__ = cls.sup_mongokit.dbname_dict.get(
                        cls.__connection__, cls.sup_mongokit._app.config.get('MONGODB_DATABASE'))


class BaseDocument(Document):
    __metaclass__ = MetaDoc
    # make the use_dot_notation support True by default
    use_dot_notation = True

    def __init__(self, *args, **kwargs):
        super(BaseDocument, self).__init__(*args, **kwargs)
        self.__dict__['connection'] = self.sup_mongokit.connection_store[self.__connection__]

    @classmethod
    def get_collection(cls):
        return getattr(getattr(cls.sup_mongokit.connection_store[cls.__connection__], cls.__database__),
                       cls.__collection__)

    @classmethod
    def create(cls, obj=None, **kargs):
        """Create a new concrete document based on a given dict-like object or ky-value pairs

        Returns
            the newly added document
        """
        instance = cls()
        instance.update(obj, **kargs)
        return instance

    def update(self, obj=None, **kwargs):
        """Update the model based on a given dict-like object, or key-value pairs

        Example::

            user = User.find_one()
            user.update(username='Vita', email='Vita@gmail.com')
            or just
            user.update({'uername': 'Vita', email: 'Vita@gmail.com'})


        Parameters
            obj
                a dict-like object stored the key-value pairs to replace current counterparts
            kwargs
                named arguments, key-value pairs

        """
        if obj is not None:
            for k, v in obj.iteritems():
                self[k] = v
        for k, v in kwargs.iteritems():
            self[k] = v
        self.save()

    def save(self, uuid=False, validate=None, safe=True, **kwargs):
        """
        save the document into the db.

        if uuid is True, a uuid4 will be automatically generated
        else, the bson.ObjectId will be used.

        If validate is True, the `validate` method will be called before
        saving. Not that the `validate` method will be called *before* the
        uuid is generated.

        `save()` follow the pymongo.collection.save arguments
        """
        if validate is True or (validate is None and self.skip_validation is False):
            self.validate(auto_migrate=False)
        else:
            if self.use_autorefs:
                self._make_reference(self, self.structure)
        if '_id' not in self:
            if uuid:
                self['_id'] = unicode("%s-%s" % (self.__class__.__name__, uuid4()))
        self._process_custom_type('bson', self, self.structure)
        self.get_collection().save(self, safe=safe, **kwargs)
        self._process_custom_type('python', self, self.structure)

    @classmethod
    def find_one(cls, *args, **kwargs):
        return cls.get_collection().find_one(wrap=cls, *args, **kwargs)

    @classmethod
    def find_by_id(cls, _id):
        return cls.find_one({'_id': _id}) or cls.find_one({'_id': ObjectId(_id)})

    @classmethod
    def find_by_id_or_404(cls, _id):
        ret = cls.find_by_id(_id)
        if ret is None:
            raise NotFound("resource not found")
        return ret

    @classmethod
    def find_one_or_404(cls, *args, **kwargs):
        ret = cls.find_one(*args, **kwargs)
        if ret is None:
            raise NotFound("{} instance not found".format(cls.__name__))
        return ret

    @classmethod
    def find(cls, *args, **kwargs):
        return cls.get_collection().find(wrap=cls, *args, **kwargs)

    def remove(self):
        self.get_collection().remove({'_id': self['_id']})
