import mimetools
import os
import time
import urllib
import urllib2
from xml.dom import minidom

import utils

class BackendError(Exception):
    """This exception class represents any kind of error raised by the
    Flickr API."""
    pass
    
class BaseBackend(object):
    """The abstract class representing the machinery of the request/response
    formats.
    
    Any implementor *must* override:
        * response_format
        * _get()
        * _get_backend_params()
        * _handle_error()
        * _parse_response()
        
        * auth_checkToken()
        * auth_getToken()
        * auth_getFrob()
        
    For the authentication step also "authenticate_step_one()" can be overridden.
    
    This backend uses REST as a Request format."""
    def __init__(self, api_key, secret_key):
        self.api_key = api_key
        self.secret_key = secret_key
        self.token = None
        self.frob = None
        # This is the string representing the HTTP response format
        # It can be "json", "rest", "soap", "xmlrpc", "php_serial"
        # See the Flickr API documentation for details: <http://www.flickr.com/services/api/>
        self.response_format = None # override this in the backends
        self.cache = utils.TokenCache(self.api_key)
        
    def _get(self, method, auth, **params):
        """Retrieves the data from the Flickr server.
        
        * method: the name of the remote method to invoke.
        * auth: True if the call has to be authenticated. False otherwise.
        * params: any argument for the call."""
        raise NotImplementedError
        
    def _get_backend_params(self):
        """Used to setup specific backend parameters for the HTTP request.
        
        To be overridden.
        
        Must return a dict() of params."""
        raise NotImplementedError
        
    def _handle_error(self, response):
        """Here goes all the error handling for the response.
        
        To be overridden.
        
        It should raise BackendError extracting the proper message data
        from the response.
        The response format is already converted by _parse_response."""
        raise NotImplementedError
        
    def _parse_response(self, response):
        """Should convert the response into valid Python data types.
        
        To be overridden."""
        raise NotImplementedError
        
    def call_api_method(self, api_method_name, auth, **params):
        """Calls the remote API method.
        
        * api_method_name is the Flickr name (eg. flickr.photos.getInfo)
        * auth is a boolean indicating whether or not the call requires authentication
        * params contains any additional parameter"""
        response = self._get(api_method_name, auth, **params)
        response = self._parse_response(response)
        self._handle_error(response)
        return response
        
    def authenticate(self, requested_perms='write'):
        """All the authentication machinery.
        
        * requested_perms can be 'read', 'write', 'delete'
        
        See: <http://www.flickr.com/services/api/auth.spec.html> 
        
        Takes also care of storing the token on the file system."""
        self.token = self.token or self.cache.get()

        if self.token:
            self.token, perms = self.auth_checkToken()
            
            # if permissions don't match we need a new token
            if perms == 'read' and requested_perms in ('write', 'delete'):
                self.token = None
            if perms == 'write' and requested_perms == 'delete':
                self.token = None 
        
        if not self.token:
            self.authenticate_step_one(requested_perms)
            self.token = self.auth_getToken()
            self.cache.put(self.token)
            
        
    def authenticate_step_one(self, perms='write', sleep_time=3):
        """ This implement the first step of authentication machinery.
        
        Note: override this to do your first step. By default uses the system
        browser to authenticate and desktop authentication."""
        
        frob = self.auth_getFrob()
        auth_url = self._get_authentication_url(perms, frob)
        utils.open_url_in_browser(auth_url)
        time.sleep(sleep_time) # needed to authenticate
        
    def _get_authentication_url(self, perms, frob):
        d = dict(api_key=self.api_key, frob=frob, perms=perms)
        d['api_sig'] = utils.get_api_signature(self.secret_key, **d)
        auth_url = "%s?%s" % (utils.AUTH_API_URL, urllib.urlencode(d))
        return auth_url
        
    def auth_checkToken(self):
        """flickr.auth.checkToken:
            Returns the credentials attached to an authentication token.
            This method call must be signed.
        
        To be overridden.
        
        See: <http://www.flickr.com/services/api/flickr.auth.checkToken.html>"""
        raise NotImplementedError
        
    def auth_getFrob(self):
        """flickr.auth.getFrob:
            Returns a frob to be used during authentication.
            This method call must be signed.
        
        To be overridden.
        
        See: <http://www.flickr.com/services/api/flickr.auth.getFrob.html>"""
        raise NotImplementedError
        
    def auth_getToken(self):
        """flickr.auth.getToken:
            Returns the auth token for the given frob, if one has been attached.
            This method call must be signed.
        
        To be overridden.
        
        See: <http://www.flickr.com/services/api/flickr.auth.getToken.html>"""
        raise NotImplementedError

    def photo_upload(self, filename=None, photo_data=None,
        data_content_type='image/jpeg', **params):
        """Uploading photos:
            It works outside the normal Flickr API framework because it
            involves sending binary files over the wire.
            
        You can specify either the filename or pass the binary image data.
        The image content type defaults to JPEG.
        
        See: <http://www.flickr.com/services/api/upload.api.html>
        """
        params['api_key'] = self.api_key
        params['auth_token'] = self.token
        params['api_sig'] = utils.get_api_signature(self.secret_key, **params)
        
        # build data to POST
        # see: http://www.flickr.com/services/api/upload.example.html
        boundary = mimetools.choose_boundary()
        request_body = []
        
        # required params
        for field in ('api_key', 'auth_token', 'api_sig'):
            request_body.append("--%s\r\n" % boundary)
            request_body.append(
                'Content-Disposition: form-data; name="%s"\r\n\r\n' % field)
            request_body.append("%s\r\n" % params[field])
        
        # optional params
        for field in ('title', 'description', 'tags', 'is_public', 'is_friend',
            'is_family', 'safety_level', 'content_type', 'hidden'):
            if field in params:
                request_body.append("--%s\r\n" % boundary)
                request_body.append(
                    'Content-Disposition: form-data; name="%s"\r\n\r\n' % field)
                request_body.append("%s\r\n" % params[field])
        
        # photo field
        request_body.append("--%s\r\n" % boundary)
        request_body.append(
            'Content-Disposition: form-data;'
            'name="photo"; filename="%s"\r\n' % filename)
        request_body.append(
            'Content-Type: %s\r\n\r\n' % data_content_type or 'image/jpeg')
        
        if filename and photo_data is None:
            path = os.path.abspath(os.path.normpath(filename))
            f = open(path, "rb")
            photo_data = f.read()
            f.close()
		
        request_data = []
        request_data.append("".join(request_body).encode('utf-8'))
        request_data.append(photo_data)
        request_data.append(('\r\n--%s--' % boundary).encode('utf-8'))
        request_data = "".join(request_data)
        
        request = urllib2.Request(utils.UPLOAD_URL)
        request.add_data(request_data)
        request.add_header(
            "Content-Type", "multipart/form-data; boundary=%s" % boundary)
        
        response = urllib2.urlopen(request)
        photo_data = response.read()
        
        def _handle_error(photo_data):
            dom = minidom.parseString(photo_data)
            rsp = dom.getElementsByTagName('rsp')[0]
            if rsp.getAttribute('stat') == 'fail':
                err = rsp.getElementsByTagName('err')[0]
                code = err.getAttribute('code')
                message = err.getAttribute('msg')
                value = "%s: %s" % (code, message)
                raise BackendError(value)
        
        def _get_photo_id(photo_data):
            dom = minidom.parseString(photo_data)
            photo_id_tag = dom.getElementsByTagName('photoid')[0]
            return photo_id_tag.firstChild.nodeValue
        
        _handle_error(photo_data)
        photo_id = _get_photo_id(photo_data)
        return photo_id
