import urllib
import urllib2

import simplejson as json

import base
import utils


class JSONBackend(base.BaseBackend):
    """This backend uses JSON as format for HTTP Response
    and REST for the Request.
    
    See <http://www.flickr.com/services/api/request.rest.html>
    and <http://www.flickr.com/services/api/response.json.html>."""
    def __init__(self, api_key, secret_key):
        super(JSONBackend, self).__init__(api_key, secret_key)
        self.response_format = 'json'

    def _get(self, method, auth, **params):
        params.update(self._get_backend_params())
        if 'format' not in params:
            params['format'] = self.response_format
        
        params['api_key'] = self.api_key
        
        for key, value in params.iteritems():
            if isinstance(value, list):
                params[key] = ','.join([item for item in value])

        if method:
            params['method'] = method

        if auth and self.token:
            params['auth_token'] = self.token

        params['api_sig'] = utils.get_api_signature(self.secret_key, **params)
        
        return urllib2.urlopen(utils.REST_API_URL,
            urllib.urlencode(params)).read()

    def _get_backend_params(self):
        return dict(format=self.response_format, nojsoncallback=1)
        
    def _parse_response(self, response):
        return json.loads(response)

    def _handle_error(self, response):
        if response['stat'] == 'fail':
            msg = "%s: %s" % (response['code'], response['message'])
            raise base.BackendError(msg)

    def auth_checkToken(self):
        try:
            response = self.call_api_method('flickr.auth.checkToken', False,
                auth_token=self.token)
        except base.BackendError:
            return (None, None)
        
        token = response['auth']['token']['_content']
        perms = response['auth']['perms']['_content']
        
        return token, perms
            
    def auth_getFrob(self):
        response = self.call_api_method('flickr.auth.getFrob', False, **{})
        self.frob = response['frob']['_content']
        return self.frob
    
    def auth_getToken(self):
        if not self.frob:
            raise base.BackendError("You must call auth_getFrob() first.")
        
        response = self.call_api_method('flickr.auth.getToken', False,
            frob=self.frob)
        self.token = response['auth']['token']['_content']
        return self.token
        
    def get_buddy_icon_url(self, user_id=None):
        """
        get_buddy_icon_url:
            Returns the URL of the given user's buddy icon.
            This works outside the Flickr API documentation.
            
        See: <http://flickr.com/services/api/misc.buddyicons.html>"""
        base_url = (
            "http://farm%(icon_farm)s.static.flickr.com/"
            "%(icon_server)s/buddyicons/%(nsid)s.jpg")
            
        user_info = self.call_api_method('flickr.people.getInfo', True,
            user_id=user_id)['person']
        
        d = dict(icon_farm=user_info['iconfarm'],
            icon_server=user_info['iconserver'], nsid=user_id)
        return base_url % d
