import tempfile
import unittest

from flickyou import backend, FlickrClient

import fixture
import utils

class JSONBackendTestCase(unittest.TestCase):
    def setUp(self):
        self.json_backend = backend.JSONBackend('barfoo', 'foobar')
        self.json_backend.cache = backend.utils.TokenCache('barfoo',
            tempfile.mkdtemp())
        self.client = FlickrClient('barfoo', 'foobar')
        self.response_buffer = []
        
    def test_creation(self):
        self.assertEquals('barfoo', self.json_backend.api_key)
        self.assertEquals('foobar', self.json_backend.secret_key)
        self.assert_(self.json_backend.token is None)
        self.assert_(self.json_backend.frob is None)
        self.assertEquals('json', self.json_backend.response_format)
    
    def test_wrong_format(self):
        response = """
        <?xml version="1.0" encoding="utf-8" ?>
        <rsp stat="fail">
            <err code="111" msg="Format &quot;foobar&quot; not found" />
        </rsp>
        """
        self.assertRaises(ValueError, self.json_backend._parse_response,
            response)
        
    def test_any_json_error_from_flickr(self):
        err_response = {u'code': 112,
            u'message': u'Method "flickr.foo.getFrob" not found',
            u'stat': u'fail'}
            
        self.assertRaises(backend.BackendError,
            self.json_backend._handle_error, err_response)
            
    def test_get_backend_params(self):
        self.assert_(isinstance(self.json_backend._get_backend_params(),
            dict))
            
    def test_getFrob(self):
        utils.TESTCASE = "getFrob"
        self.assertEquals(fixture.expectedFrob, self.client.auth_getFrob())
        self.assertEquals(fixture.expectedFrob, self.client.backend.frob)
        
    def test_getToken(self):
        utils.TESTCASE = "getToken"
        self.assertRaises(backend.BackendError, self.client.auth_getToken)
        
        self.client.backend.frob = fixture.expectedFrob
        self.assertEquals(fixture.expectedGetToken,
            self.client.auth_getToken())
        self.assertEquals(fixture.expectedGetToken, self.client.backend.token)
            
    def test_checkToken_fails(self):
        utils.TESTCASE = "checkTokenFailure"
        self.assertEquals((None, None), self.client.auth_checkToken())
        
    def test_checkToken(self):
        utils.TESTCASE = "checkToken"
        self.assertEquals(fixture.expectedCheckToken,
            self.client.auth_checkToken())

    def test_get_buddy_icon_url(self):
        utils.TESTCASE = 'buddy_icon'
        self.assertEquals(fixture.expected_buddy_icon,
            self.client.get_buddy_icon_url(user_id=1234))
            
    def test_authenticate(self):
        self.test_getToken()
        
        old_token = self.client.backend.token
        
        utils.TESTCASE = 'perms_read'
        self.client.authenticate(requested_perms='read')
        self.assertEqual(old_token, self.client.backend.token)

        # XXX: need mocks?!
        # self.client.authenticate(requested_perms='write')
        # self.assertEqual(old_token, self.client.backend.token)
        

if __name__ == '__main__':
    unittest.main()
