"""
mfwel module.  Contains the ModflowWel class. Note that the user can access
the ModflowWel class as `flopy.modflow.ModflowWel`.

Additional information for this MODFLOW package can be found at the `Online
MODFLOW Guide
<http://water.usgs.gov/ogw/modflow/MODFLOW-2005-Guide/index.html?wel.htm>`_.

"""

import numpy as np
from flopy.mbase import Package
from flopy.utils import util_2d,util_3d

class ModflowWel(Package):
    """
    MODFLOW Well Package Class.

    Parameters
    ----------
    model : model object
        The model object (of type :class:`flopy.modflow.mf.Modflow`) to which
        this package will be added.
    iwelcb : int
        is a flag and a unit number. (the default is 0).
    layer_row_column_Q : list of records
        In its most general form, this is a triple list of well records  The
        innermost list is the layer, row, column, and flux rate for a single
        well.  Then for a stress period, there can be a list of wells.  Then
        for a simulation, there can be a separate list for each stress period.
        This gives the form of
            lrcq = [
                     [  #stress period 1
                       [l1, r1, c1, q1],
                       [l2, r2, c2, q2],
                       [l3, r3, c3, q3],
                       ],
                     [  #stress period 2
                       [l1, r1, c1, q1],
                       [l2, r2, c2, q2],
                       [l3, r3, c3, q3],
                       ], ...
                     [  #stress period kper
                       [l1, r1, c1, q1],
                       [l2, r2, c2, q2],
                       [l3, r3, c3, q3],
                       ],
                    ]
        Note that if there are not records in layer_row_column_Q, then the last
        group of wells will apply until the end of the simulation.
    options : list of strings
        Package options. (default is None).
    naux : int
        number of auxiliary variables
    extension : string
        Filename extension (default is 'wel')
    unitnumber : int
        File unit number (default is 11).


    Attributes
    ----------
    mxactw : int
        Maximum number of wells for a stress period.  This is calculated
        automatically by FloPy based on the information in layer_row_column_Q.

    Methods
    -------

    See Also
    --------

    Notes
    -----
    Parameters are not supported in FloPy.

    Examples
    --------

    >>> import flopy
    >>> m = flopy.modflow.Modflow()
    >>> lrcq = [[[2, 3, 4, -100.]]]  #this well will be applied to all stress
    >>>                              #periods
    >>> wel = flopy.modflow.ModflowWel(m, layer_row_column_Q=lrcq)

    """
    def __init__(self, model, iwelcb=0, layer_row_column_Q=None,
                 extension ='wel', unitnumber=20, options=None, naux=0):
        """
        Package constructor.

        """
        # Call parent init to set self.parent, extension, name and unit number
        Package.__init__(self, model, extension, 'WEL', unitnumber)
        self.heading = '# Well file for MODFLOW, generated by Flopy.'
        self.url = 'wel.htm'
        self.iwelcb = iwelcb # no cell by cell terms are written
        self.mxactw = 0
        self.mxactw, self.layer_row_column_Q = \
            self.assign_layer_row_column_data(layer_row_column_Q, 4+naux)
        self.np = 0
        if options is None:
            options = []
        self.options = options
        self.parent.add_package(self)

    def __repr__( self ):
        return 'Well package class'

    def ncells( self):
        # Returns the  maximum number of cells that have a well
        # (developed for MT3DMS SSM package)
        return self.mxactw

    def write_file(self):
        """
        Write the file.

        """
        f_wel = open(self.fn_path, 'w')
        f_wel.write('%s\n' % self.heading)
        line = ('%10i%10i' % (self.mxactw, self.iwelcb))
        for opt in self.options:
            line += ' ' + str(opt)
        line += '\n'
        f_wel.write(line)
        self.write_layer_row_column_data(f_wel, self.layer_row_column_Q)
        f_wel.close()

    @staticmethod
    def load(f, model, nper=None, ext_unit_dict=None):
        """
        Load an existing package.

        Parameters
        ----------
        f : filename or file handle
            File to load.
        model : model object
            The model object (of type :class:`flopy.modflow.mf.Modflow`) to
            which this package will be added.
        nper : int
            The number of stress periods.  If nper is None, then nper will be
            obtained from the model object. (default is None).
        ext_unit_dict : dictionary, optional
            If the arrays in the file are specified using EXTERNAL,
            or older style array control records, then `f` should be a file
            handle.  In this case ext_unit_dict is required, which can be
            constructed using the function
            :class:`flopy.utils.mfreadnam.parsenamefile`.

        Returns
        -------
        wel : ModflowWel object
            ModflowWel object.

        Examples
        --------

        >>> import flopy
        >>> m = flopy.modflow.Modflow()
        >>> wel = flopy.modflow.mfwel.load('test.wel', m)

        """
        dtype = np.dtype([("layer", np.int), ("row", np.int),
                          ("column", np.int), ("flux", np.float32)])
        if type(f) is not file:
            filename = f
            f = open(filename, 'r')
        #dataset 0 -- header
        while True:
            line = f.readline()
            if line[0] != '#':
                break
        #--check for parameters
        if "parameter" in line.lower():
            raw = line.strip().split()
            assert int(raw[1]) == 0,"Parameters are not supported"
            line = f.readline()
        #dataset 2a
        t = line.strip().split()
        #mxactw, iwelcb = t[0:3]
        mxactw = int(t[0])
        iwelcb = 0
        try:
            if int(t[1]) != 0:
                iwelcb = 53
        except:
            pass

        options = []
        naux = 0
        if len(t) > 2:
            for toption in t[3:-1]:
                if toption.lower() is 'noprint':
                    options.append(toption)
                elif 'aux' in toption.lower():
                    naux += 1
                    options.append(toption)
        if nper is None:
            nrow, ncol, nlay, nper = model.get_nrow_ncol_nlay_nper()
        #read data for every stress period
        layer_row_column_Q = []
        current = []
        nitems = 4 + naux
        for iper in xrange(nper):
            print "   loading wells for kper {0:5d}".format(iper+1)
            line = f.readline()
            t = line.strip().split()
            itmp = int(t[0])
            if itmp == 0:
                current = []
            elif itmp > 0:
                for ibnd in xrange(itmp):
                    line = f.readline()
                    t = line.strip().split()
                    bnd = []
                    for jdx in xrange(nitems):
                        if jdx < 3:
                            bnd.append(int(t[jdx]))
                        else:
                            bnd.append(float(t[jdx]))
                    current.append(bnd)
                #current = np.fromfile(f,dtype=dtype,count=itmp,sep=" ")
            layer_row_column_Q.append(current)
        wel = ModflowWel(model, iwelcb, layer_row_column_Q, options=options,
                         naux=naux)
        return wel

