"""
mfdrn module.  Contains the ModflowDrn class. Note that the user can access
the ModflowDrn class as `flopy.modflow.ModflowDrn`.

Additional information for this MODFLOW package can be found at the `Online
MODFLOW Guide
<http://water.usgs.gov/ogw/modflow/MODFLOW-2005-Guide/index.html?drn.htm>`_.

"""
from flopy.mbase import Package


class ModflowDrn(Package):
    """
    MODFLOW Drain Package Class.

    Parameters
    ----------
    model : model object
        The model object (of type :class:`flopy.modflow.mf.Modflow`) to which
        this package will be added.
    idrncb : int
        is a flag and a unit number. (the default is 0).
    layer_row_column_data : list of records
        In its most general form, this is a triple list of drain records  The
        innermost list is the layer, row, column, elevation and conductance for
        a single drain.  Then for a stress period, there can be a list of
        drains.  Then for a simulation, there can be a separate list for each
        stress period. This gives the form of
            lrcd = [
                     [  #stress period 1
                       [l1, r1, c1, elev1, cond1],
                       [l2, r2, c2, elev2, cond2],
                       [l3, r3, c3, elev3, cond3],
                       ],
                     [  #stress period 2
                       [l1, r1, c1, elev1, cond1],
                       [l2, r2, c2, elev2, cond2],
                       [l3, r3, c3, elev3, cond3],
                       ], ...
                     [  #stress period kper
                       [l1, r1, c1, elev1, cond1],
                       [l2, r2, c2, elev2, cond2],
                       [l3, r3, c3, elev3, cond3],
                       ],
                    ]
        Note that if there are not records in layer_row_column_data, then the
        last group of drains will apply until the end of the simulation.
    layer_row_column_elevation_cond : list of records
        Deprecated - use layer_row_column_data instead.
    options : list of strings
        Package options. (default is None).
    naux : int
        number of auxiliary variables
    extension : string
        Filename extension (default is 'drn')
    unitnumber : int
        File unit number (default is 21).


    Attributes
    ----------
    mxactd : int
        Maximum number of drains for a stress period.  This is calculated
        automatically by FloPy based on the information in
        layer_row_column_data.

    Methods
    -------

    See Also
    --------

    Notes
    -----
    Parameters are not supported in FloPy.

    Examples
    --------

    >>> import flopy
    >>> m = flopy.modflow.Modflow()
    >>> lrcd = [[[2, 3, 4, 10., 100.]]]  #this drain will be applied to all
    >>>                                  #stress periods
    >>> drn = flopy.modflow.ModflowDrn(m, layer_row_column_data=lrcd)

    """
    def __init__(self, model, idrncb = 0, layer_row_column_data=None,
                 layer_row_column_elevation_cond=None, 
                 extension ='drn', unitnumber=21, options=None, naux=0):
        Package.__init__(self, model, extension, 'DRN', unitnumber) # Call ancestor's init to set self.parent, extension, name and unit number
        self.heading = '# DRN for MODFLOW, generated by Flopy.'
        self.url = 'drn.htm'
        self.idrncb = idrncb  # 0: no cell by cell terms are written
        if layer_row_column_data is None:
          if layer_row_column_elevation_cond is not None:
            print '\nWARNING: ModflowDrn - Do not use layer_row_column_elevation_cond!\n'+\
                  '                      Use layer_row_column_data instead.'
            layer_row_column_data = layer_row_column_elevation_cond
          else:
            raise Exception('Failed to specify layer_row_column_elevation_cond or layer_row_column_data.')

        self.mxactd, self.layer_row_column_data = self.assign_layer_row_column_data(layer_row_column_data, 5+naux)
        self.np = 0
        if options is None:
            options = []
        self.options = options
        self.parent.add_package(self)
    def ncells( self):
        # Returns the  maximum number of cells that have drains (developped for MT3DMS SSM package)
        #print 'Function must be implemented properly for drn package'
        
        return self.mxactd
    def write_file(self):
        f_drn = open(self.fn_path, 'w')
        f_drn.write('%s\n' % self.heading)
        #f_drn.write('%10i%10i\n' % (self.mxactd, self.idrncb))
        line = '{0:10d}{1:10d}'.format(self.mxactd, self.idrncb)
        for opt in self.options:
            line += ' ' + str(opt)
        line += '\n'
        f_drn.write(line)
        self.write_layer_row_column_data(f_drn, self.layer_row_column_data)
        f_drn.close()

    @staticmethod
    def load(f, model, nper=None, ext_unit_dict=None):
        '''
        f is either a filename or a file handle.  if the arrays in the file
        are specified using interal, external, or older style array control
        records, then f should be a file handle, and the ext_unit_dict
        dictionary of unitnumber:open(filename, 'r') must be included.
        '''
        if type(f) is not file:
            filename = f
            f = open(filename, 'r')
        #dataset 0 -- header
        while True:
            line = f.readline()
            if line[0] != '#':
                break
        #--check for parameters
        if "parameter" in line.lower():
            raw = line.strip().split()
            assert int(raw[1]) == 0,"Parameters are not supported"
            line = f.readline()
        #dataset 2a
        t = line.strip().split()
        mxactd = int(t[0])
        idrncb = 0
        try:
            if int(t[1]) != 0:
                idrncb = 53
        except:
            pass

        options = []
        naux = 0
        if len(t) > 2:
            for toption in t[3:-1]:
                if toption.lower() is 'noprint':
                    options.append(toption)
                elif 'aux' in toption.lower():
                    naux += 1
                    options.append(toption)
        if nper is None:
            nrow, ncol, nlay, nper = model.get_nrow_ncol_nlay_nper()
        #read data for every stress period
        layer_row_column_data = []
        current = []
        nitems = 5 + naux
        for iper in xrange(nper):
            line = f.readline()
            t = line.strip().split()
            itmp = int(t[0])
            if itmp == 0:
                current = []
            elif itmp > 0:
                for ibnd in xrange(itmp):
                    line = f.readline()
                    t = line.strip().split()
                    current.append(t[0:nitems])
            layer_row_column_data.append(current)
        drn = ModflowDrn(model,idrncb=idrncb,layer_row_column_data=layer_row_column_data,options=options,naux=naux)
        return drn
