"""Information on a solver (:mod:`fluiddyn.simul.base.info_solver_params`)
==========================================================================

.. currentmodule:: fluiddyn.simul.base.info_solver_params

Provides:

.. autoclass:: InfoSolverBase
   :members:
   :private-members:

.. autoclass:: Parameters
   :members:
   :private-members:


"""

from __future__ import division, print_function

import os
from copy import deepcopy

from fluiddyn.util.containerxml import ContainerXML
from fluiddyn.util.util import import_class


def _merged_element(el1, el2):
    result = deepcopy(el1)
    result.extend(deepcopy(el2))
    return result


class InfoSolverBase(ContainerXML):
    """Contain the information on a solver."""
    def __init__(self, **kargs):

        if 'tag' not in kargs:
            kargs['tag'] = 'solver'

        super(InfoSolverBase, self).__init__(**kargs)

        if kargs['tag'] == 'solver' and 'path_file' not in kargs:
            self._init_root()


    def _init_root(self):

        self.set_attribs({'module_name': 'fluiddyn.simul.base.solver',
                          'class_name': 'SimulBase',
                          'short_name': 'Base'})

        self.set_child('classes')
        self.classes.set_child(
            'State',
            attribs={'module_name': 'fluiddyn.simul.base.state',
                     'class_name': 'StatePseudoSpectral'})

        self.classes.set_child(
            'InitFields',
            attribs={'module_name': 'fluiddyn.simul.base.init_fields',
                     'class_name': 'InitFieldsBase'})

        self.classes.set_child(
            'TimeStepping',
            attribs={'module_name': 'fluiddyn.simul.base.time_stepping',
                     'class_name': 'TimeSteppingPseudoSpectral'})

        self.classes.set_child(
            'Operators',
            attribs={'module_name': 'fluiddyn.simul.operators.operators',
                     'class_name': 'OperatorsPseudoSpectral2D'})

        self.classes.set_child(
            'Forcing',
            attribs={'module_name': 'fluiddyn.simul.base.forcing',
                     'class_name': 'ForcingBase'})

        self.classes.set_child(
            'Output',
            attribs={'module_name': 'fluiddyn.simul.base.output.base_output',
                     'class_name': 'OutputBase'})


    def import_classes(self):
        """Import the classes and return a dictionary."""
        classes = self._elemxml.findall('classes')
        classes = reduce(_merged_element, classes)
        dict_classes = {}
        for c in classes.getchildren():
            try:
                module_name = c.attrib['module_name']
                class_name = c.attrib['class_name']
            except KeyError:
                pass
            else:
                Class = import_class(module_name, class_name)
                dict_classes[c.tag] = Class

        return dict_classes

    def complete_with_classes(self):
        dict_classes = self.import_classes()
        for Class in dict_classes.values():
            if hasattr(Class, '_complete_info_solver'):
                Class._complete_info_solver(self)



class Parameters(ContainerXML):
    """Contain the parameters."""
    def check_and_modify(self):
        pass



def create_params(input_info_solver):
    """Create a Parameters instance from an InfoSolverBase instance."""
    if isinstance(input_info_solver, InfoSolverBase):
        info_solver = input_info_solver
    elif hasattr(input_info_solver, 'info_solver'):
        info_solver = input_info_solver.info_solver
    else:
        raise ValueError('input_info_solver is not related '
                         'to a InfoSolver instance.')

    params = Parameters(tag='params')
    dict_classes = info_solver.import_classes()

    dict_classes['Solver'] = import_class(
        info_solver.module_name, info_solver.class_name)

    for Class in dict_classes.values():
        if hasattr(Class, '_complete_params_with_default'):
            Class._complete_params_with_default(params)
    return params




def create_info_simul(info_solver, params):
    """Create a ContainerXML instance gathering info_solver and params."""
    info = ContainerXML(tag='info_simul')
    info.set_as_child(info_solver)
    info.set_as_child(params)
    return info


def load_params_simul(path_dir=None):
    """Load the parameters and return a Parameters instance."""
    if path_dir is None:
        path_dir = os.getcwd()
    return Parameters(
        path_file=os.path.join(path_dir, 'params_simul.xml'))


def load_info_solver(path_dir=None):
    """Load the solver information, return an InfoSolverBase instance.

    """
    if path_dir is None:
        path_dir = os.getcwd()
    return InfoSolverBase(
        path_file=os.path.join(path_dir, 'info_solver.xml'))


def load_info_simul(path_dir=None):
    """Load the data and gather them in a ContainerXML instance."""

    if path_dir is None:
        path_dir = os.getcwd()
    info_solver = load_info_solver(path_dir=path_dir)
    params = load_params_simul(path_dir=path_dir)
    info = ContainerXML(tag='info_simul')
    info.set_as_child(info_solver)
    info.set_as_child(params)
    return info






if __name__ == '__main__':
    info_solver = InfoSolverBase(tag='solver')

    info_solver.complete_with_classes()

    params = create_params(info_solver)

    info = create_info_simul(info_solver, params)
