import os

from django.db import models
from django.contrib.auth.models import User
from django.conf import settings

from hadrian.utils.slugs import unique_slugify
from hadrian.contrib.locations.models import *

from sorl.thumbnail import get_thumbnail

from PIL import Image
from PIL.ExifTags import TAGS

from fotochest.photo_manager.managers import PhotoManager
from fotochest.administrator.tasks import thumbnail_cleanup_task, thumbnail_task

class Album(models.Model):
    title = models.CharField(max_length=250)
    slug = models.SlugField(editable=False, blank=True)
    description = models.TextField(null=True, blank=True)
    date_created = models.DateTimeField(auto_now=True, auto_now_add=True)
    parent_album = models.ForeignKey('self', blank=True, null=True)
    album_cover = models.ImageField(upload_to="cover_art/", max_length=400, blank=True, null=True)
    user = models.ForeignKey(User)
    
    def __unicode__(self):
        return self.title

    def save(self, *args, **kwargs):
        unique_slugify(self, self.title)
        super(Album, self).save(*args, **kwargs)
        
    def get_preview_photos(self):
        # Test Func
        photos = Photo.objects.filter(album=self)[:5]
        return photos
        
    def get_album_cover(self):
        this_photo = ""
        try:
            photos = Photo.objects.filter(album=self)[:1]
            return photos[0]
        except:
            # Try for first Child.
            try:
                albums = Album.objects.filter(parent_album=self)[:1]
                album = albums[0]
                photos = Photo.objects.filter(album=album)[:1]
                return photos[0]
                    
            except:
                # one final layer down
                try:
                    albums = Album.objects.filter(parent_album=self)[:1]
                    album = albums[0]
                    new_album = Album.objects.filter(parent_album=album)
                    use_album = new_album[0]
                    photos = Photo.objects.filter(album=use_album)[:1]
                    return photos[0]
                except:
                    
                    pass
                    this_photo = ""
        return this_photo
    
    @property
    def has_child_albums(self):
        album_count = Album.objects.filter(parent_album=self).count()
        if album_count == 0:
            return False
        else:
            return True

    @property
    def count(self):
        return Photo.objects.filter(album=self).count()

    @models.permalink
    def get_absolute_url(self):
        return ('album_detail', (), {'album_id': self.id, 'album_slug': self.slug})
        

    @models.permalink
    def get_admin_url(self):
        return ('administrator.views.album_detail', (), {'album_id':self.id})

class Photo(models.Model):
    title = models.CharField(max_length=250)
    slug = models.SlugField(editable=False, blank=True)
    file_name = models.CharField(max_length=400, editable=False)
    image = models.ImageField(upload_to="images/", max_length=400)
    description = models.TextField(null=True, blank=True)
    date_uploaded = models.DateTimeField(auto_now=False, auto_now_add=True)
    album = models.ForeignKey(Album)
    user = models.ForeignKey(User, blank=True, null=True)
    location = models.ForeignKey(Location, blank=True, null=True)
    thumbs_created = models.BooleanField(default=False, editable=False)
    deleted = models.BooleanField(default=False, editable=False)
    
    objects = PhotoManager()
    
    def __unicode__(self):
        return self.title
    
    def save(self, *args, **kwargs):
        unique_slugify(self, self.title)
        super(Photo, self).save()
    
    @property    
    def filename(self):
        return os.path.basename(self.image.name)
    
    @models.permalink
    def get_next(self):
        try:
            next_photo = Photo.objects.filter(id__lt=self.id, user=self.user)[:1]
            photo = next_photo[0]
        except:
            return None
        return ('photo_manager.views.photo', (), {'photo_id': photo.id, 'photo_slug': photo.slug, 'album_slug': photo.album.slug})
    
    @models.permalink
    def get_previous(self):
        try:
            prev_photo = Photo.objects.filter(id__gt=self.id, user=self.user).order_by('id')[:1]
            photo = prev_photo[0]
        except:
            return None
        return ('photo_manager.views.photo', (), {'photo_id': photo.id, 'photo_slug': photo.slug, 'album_slug': photo.album.slug})
        
    def image_preview(self):
        im = get_thumbnail(self.image, "150x150")
        return '<img src="%s" width="150"/>'  % im.url
    image_preview.allow_tags = True
    
    
    def make_thumbnails(self):
        # Current Thumb list
        # 240x165 (streams)
        # 75x75 for map (Other location photos)
        # 1024x768 for photo.html
        
        get_thumbnail(self.image, '75x75', crop="center", quality=50)
        get_thumbnail(self.image, '1024x650', quality=100)
        get_thumbnail(self.image, '240x165')
        get_thumbnail(self.image, '240x161', crop="center", quality=75)
        get_thumbnail(self.image, '300x220')
        get_thumbnail(self.image, '300x300')

    def rotate(self, right=True):
        path = "%s/%s" % (settings.MEDIA_ROOT, self.image)
        im = Image.open(path)
        if right:
            im.rotate(270).save(path)
        else:
            im.rotate(90).save(path)
        thumbnail_cleanup_task.delay(self)
        thumbnail_task.delay(self)
        
    def get_exif_data(self):
        exif_data = {}
        i = Image.open(self.image)
        info = i._getexif()
        for tag, value in info.items():
            decoded = TAGS.get(tag, tag)
            exif_data[decoded] = value
        return exif_data

    #@todo - point to new class ivew.
    @models.permalink
    def get_absolute_url(self):
        return ('regular_photo_url', (), {'photo_id': self.id, 'photo_slug': self.slug, 'album_slug': self.album.slug})
    
    @models.permalink
    def get_fullscreen(self):
        # update with enable multi user
        return ('photo_fullscreen', (), {'photo_id': self.id, 'photo_slug': self.slug, 'album_slug': self.album.slug})

    class Meta:
        ordering = ['-id']

def photos_by_location(location):
    return Photo.objects.filter(deleted=False, location=location).count()
         