from __future__ import absolute_import

import psycopg2
import psycopg2.extensions
from .impl import _filter_row, FDBSQLResultContext
from .api import NESTED_CURSOR
from . import compat
import datetime

__all__ = ['connect', 'NESTED_CURSOR']

def connect(*arg, **kw):
    """Establish a connection to a database.

    This function is a proxy into psycopg2's :func:`psycopg2.connect`
    function, e.g.::

      from foundationdb_sql.psycopg2 import connect
      connection = connect(host="localhost", port=15432)

    :func:`.connect` calls directly into psycopg2's connection
    function at :func:`psycopg2.connect`, establishing only
    a single additional parameter as the ``connection_factory``.

    The return value is an instance of :class:`.Connection`.

    .. seealso::

        :func:`psycopg2.connect` - full description of arguments.

    """
    kw['connection_factory'] = Connection
    return psycopg2.connect(*arg, **kw)

class _adapt_dict(dict):
    def __missing__(self, typ_):
        for super_ in typ_.__mro__:
            if super_ in self:
                adapter = self[typ_] = self[super_]
                break
        else:
            adapter = self[typ_] = None
        return adapter

class PG2RootCursor(psycopg2.extensions.cursor):
    """A :class:`psycopg2.extensions.cursor` extension class.

    Provides the root behavior for switching between
    "nested" output format and regular.


    .. seealso::

        :class:`.PG2NestedCursor`

        :class:`.PG2PlainCursor`

        :class:`.NestedCursor`

    """
    def _super(self):
        return super(PG2RootCursor, self)

    def _check_nesting(self):
        if self.connection._nested is not self.nested:
            self._super().execute("set OutputFormat='%s'" %
                        ('json_with_meta_data' if self.nested else 'table')
                    )
            self.connection._nested = self.nested


    def _adapt(self, parameters):
        """unfortunately, psycopg2 gives us no high performing method
        of adapting types on a per-connection basis.  So we have to scan
        through parameters looking for types that need correction.

        """
        if parameters is None:
            return None
        elif hasattr(parameters, "keys"):
            return dict(
                (k, adapter(v) if adapter is not None else v)

                for k, v, adapter in [
                    (k, v, bind_adapters[type(v)]) for k, v in parameters.items()
                ]
            )
        else:
            return [
                adapter(v) if adapter is not None else v

                for v, adapter in [
                    (v, bind_adapters[type(v)]) for v in parameters
                ]
            ]

    def execute(self, operation, parameters=None):
        """Execute a statement as per the DBAPI specifciation.

        This method ultimately calls into the :meth:`psycopg2:cursor.execute`
        method; see that method for invocation details.

        """
        return super(PG2RootCursor, self).execute(operation,
            self._adapt(parameters))

    def executemany(self, operation, seq_of_parameters=None):
        """Execute a statement as "executemany" as per the DBAPI specification.

        This method ultimately calls into the :meth:`psycopg2:cursor.executemany`
        method; see that method for invocation details.

        """

        return super(PG2RootCursor, self).executemany(operation,
                    [self._adapt(v) for v in seq_of_parameters or ()])


class PG2NestedCursor(PG2RootCursor):
    """:class:`.PG2RootCursor` subclass which provides for nested result-set
    fetching.

    The :class:`.PG2NestedCursor` makes use of the JSON-decoding services
    of :class:`.FDBSQLResultContext` in order to produce result sets,
    and assumes the connection has been
    placed into FoundationDB's ``json_with_meta_data`` output format.

    .. seealso::

        :meth:`.Connection.cursor` - factory for :class:`.PG2NestedCursor`

    """
    nested = True

    def execute(self, *arg, **kw):
        """Execute a statement as per the DBAPI specifciation.

        This method ultimately calls into the :meth:`psycopg2:cursor.execute`
        method; see that method for invocation details.

        """

        self._check_nesting()
        ret = super(PG2NestedCursor, self).execute(*arg, **kw)
        self._setup_description()
        return ret

    def executemany(self, *arg, **kw):
        """Execute a statement as "executemany" as per the DBAPI specification.

        This method ultimately calls into the :meth:`psycopg2:cursor.executemany`
        method; see that method for invocation details.

        """

        self._check_nesting()
        ret = super(PG2NestedCursor, self).executemany(*arg, **kw)
        self._setup_description()
        return ret

    def _setup_description(self):
        if self._super().description:
            self._fdbsql_ctx = PG2ResultContext(
                                self, self._super().fetchone()
                            )
        else:
            self._fdbsql_ctx = None

    def fetchone(self):
        """Fetch a single row as per the DBAPI specification."""

        return _filter_row(
                    self._super().fetchone(),
                    self._fdbsql_ctx
                )

    def fetchall(self):
        """Fetch all rows as per the DBAPI specification."""

        return [
            _filter_row(row, self._fdbsql_ctx)
            for row in self._super().fetchall()
        ]

    def fetchmany(self, size=None):
        """Fetch a batch of rows as per the DBAPI specification."""

        return [
            _filter_row(row, self._fdbsql_ctx)
            for row in self._super().fetchmany(size)
        ]

    @property
    def fdbsql_description(self):
        """Return the FDB-SQL specific cursor.description which includes
        additional metadata about nested result sets.

        .. seealso::

            :attr:`.PG2NestedCursor.description` - DBAPI-compliant cursor
            description.

            :ref:`nested_result_sets` - usage example.

        """
        if self._fdbsql_ctx:
            return self._fdbsql_ctx.fdbsql_description
        else:
            return None

    @property
    def description(self):
        """Return the description of this cursor as per the DBAPI specifciation.

        .. seealso::

            :attr:`.PG2NestedCursor.fdbsql_description` - cursor description with
            additional data specific to nested result sets.

            :attr:`.NestedCursor.description` - corresponding description on
            a nested result set.

            :ref:`nested_result_sets` - usage example.

        """
        # TODO: I'm going on a "convenient" behavior here,
        # that the ".description" attribute on psycopg2.cursor
        # acts like a method that we override below.
        # Would need to confirm that the Python
        # C API and/or psycopg2 supports this pattern.
        if self._fdbsql_ctx:
            return self._fdbsql_ctx.description
        else:
            return None


class PG2PlainCursor(PG2RootCursor):
    """:class:`.PG2RootCursor` subclass which provides for traditional result-set
    fetching.

    The :class:`.PG2PlainCursor` assumes the connection has been
    placed into FoundationDB's ``table`` output format.

    .. seealso::

        :meth:`.Connection.cursor` - factory for :class:`.PG2PlainCursor`

    """

    nested = False

    def execute(self, *arg, **kw):
        self._check_nesting()
        return super(PG2PlainCursor, self).execute(*arg, **kw)

    def executemany(self, *arg, **kw):
        self._check_nesting()
        return super(PG2PlainCursor, self).executemany(*arg, **kw)



_psycopg2_adapter_cache = {
}

_none_type = type(None)

class PG2ResultContext(FDBSQLResultContext):
    """Implement the :class:`.FDBSQLResultContext` interface specific to the
    psycopg2 DBAPI.


    """
    def gen_description(self, fields):
        return [
            (rec['name'], rec['type_oid'],
                    None, None, None, None, None)
            for rec in fields
        ]

    @property
    def description(self):
        return self.gen_description(self.fields)

    @property
    def fdbsql_description(self):
        return self.gen_fdbsql_description(self.fields)

    def operational_error(self, message):
        raise psycopg2.OperationalError(message)

    def gen_fdbsql_description(self, fields):
        return [
            (rec['name'], rec['type_oid'],
                    None, None, None, None, None,
                    self.gen_fdbsql_description(rec['foundationdb_sql.fields'])
                    if 'foundationdb_sql.fields' in rec else None
            )
            for rec in fields
        ]

    def typecast(self, value, oid):
        try:
            # return a cached "adpater" for this oid.
            # for a particular oid that's been seen before,
            # this is the only codepath.
            adapter = _psycopg2_adapter_cache[oid]
        except KeyError:
            # no "adapter".  figure it out.   we don't want to be
            # calling isinstance() on every row so we cache whether or
            # not psycopg2 returns this particular oid as a string
            # or not, assuming it will be consistent per oid.
            if isinstance(value, (compat.string_types, )):
                adapter = _psycopg2_adapter_cache[oid] = \
                        psycopg2.extensions.string_types[oid]
            elif value is None:
                # can't make a determination here as to whether
                # we need an adapter or not
                return value
            else:
                adapter = _psycopg2_adapter_cache[oid] = None

            # hardwire STRING types to not use adapters, since we're
            # getting string data back already.  FoundationDB seems to be
            # sending fully unicode data back even without using psycopg2
            # unicode extensions.
            if adapter is not None and adapter == psycopg2.STRING:
                _psycopg2_adapter_cache[oid] = adapter = None

        if adapter:
            # TODO: do we send the oid or the adapter
            # as the argument here?
            return adapter(value, adapter)
        else:
            return value


class Connection(psycopg2.extensions.connection):
    """An implementation of :class:`psycopg2.extensions.connection`
    which adds support for nested cursors, as well as FoundationDB-specific
    SQL and type support.

    .. seealso::

        :func:`.connect` - factory for :class:`.Connection`.

        :class:`psycopg2:connection` - psycopg2's connection object implementing
        the majority of functionality.

    """
    def __init__(self, dsn, async=0):
        super(Connection, self).__init__(dsn, async=async)
        self._nested = False

    def _super_cursor(self, *arg, **kw):
        return super(Connection, self).cursor(*arg, **kw)

    def cursor(self, nested=False):
        """Create a new cursor.

        :param nested: if ``True``, return a :class:`.PG2NestedCursor`.  This
         cursor, when used, will first ensure that the FDB ``OutputFormat``
         is set to ``json_with_meta_data``.  If ``False``, will return a
         :class:`.PG2PlainCursor`, which ensures that ``OutputFormat`` is
         set to ``table``.

        """
        if nested:
            cursor = self._super_cursor(cursor_factory=PG2NestedCursor)
        else:
            cursor = self._super_cursor(cursor_factory=PG2PlainCursor)
        return cursor

bind_adapters = _adapt_dict()

def datetime_to_sql(dt):
    return dt.isoformat()

def date_to_sql(dt):
    return dt.isoformat()

def time_to_sql(dt):
    return dt.strftime("%H:%M:%S")

bind_adapters[datetime.datetime] = datetime_to_sql
bind_adapters[datetime.date] = date_to_sql
bind_adapters[datetime.time] = time_to_sql
