# coding: utf-8

import unittest
try:
    from nose import SkipTest
except ImportError:
    from _pytest.runner import Skipped as SkipTest

from . import fixtures, fails, assert_raises_message
import foundationdb_sql
import datetime

try:
    import psycopg2
    import psycopg2.extensions
except ImportError:
    raise SkipTest("psycopg2 is not installed")

class Psycopg2Test(object):
    @classmethod
    def setup_class(cls):
        from foundationdb_sql.psycopg2 import connect

        cls.connection = connect(port=15432, host="localhost")

    @classmethod
    def teardown_class(cls):
        cls.connection.close()

    def tearDown(self):
        self.connection.rollback()

class NestedResultTest(Psycopg2Test, unittest.TestCase):

    @classmethod
    def setup_class(cls):
        super(NestedResultTest, cls).setup_class()
        fixtures._table_fixture(cls.connection)
        fixtures._data_fixture(cls.connection)

    @classmethod
    def teardown_class(cls):
        fixtures._drop_tables(cls.connection)
        super(NestedResultTest, cls).teardown_class()


    def test_transactions_yes_or_no(self):
        cursor = self.connection.cursor()
        cursor.execute("insert into customers (customer_id, name) VALUES (12, 'xyz')")
        cursor.close()
        self.connection.rollback()
        cursor = self.connection.cursor()
        cursor.execute("select * from customers where customer_id=12")
        assert not cursor.fetchone()
        cursor.close()

    def test_basic_description_nested(self):
        cursor = self.connection.cursor(nested=True)
        cursor.execute(
                "select customer_id, name from customers where customer_id=5")
        self.assertEquals(
            cursor.description,
            [
                ('customer_id', psycopg2.extensions.INTEGER, None, None,
                        None, None, None),
                ('name', psycopg2.STRING, None, None, None, None, None)
            ]
        )

    def test_basic_description_plain(self):
        cursor = self.connection.cursor(nested=False)
        cursor.execute(
                "select customer_id, name from customers where customer_id=5")
        self.assertEquals(
            [desc[0:2] for desc in cursor.description],
            [
                ('customer_id', psycopg2.extensions.INTEGER),
                ('name', psycopg2.STRING)
            ]
        )

    def test_typecodes_hashable(self):
        cursor = self.connection.cursor()
        cursor.execute(
                "select customer_id, name from customers where customer_id=5")
        self.assertEquals(
            set([cursor.description[1][1]]),
            set([1043])
        )

    def test_nested_description_toplevel(self):
        cursor = self.connection.cursor(nested=True)
        cursor.execute(
                "select customer_id, name, "
                "(select order_id from orders where orders.customer_id=5) "
                "as orders from customers where customer_id=5")
        self.assertEquals(
            cursor.description,
            [
                ('customer_id', psycopg2.extensions.INTEGER, None, None,
                        None, None, None),
                ('name', psycopg2.STRING, None, None, None, None, None),
                ('orders', foundationdb_sql.NESTED_CURSOR, None, None, None, None, None),
            ]
        )

    def _test_nested_fetch(self, getrows, padding=[]):
        cursor = self.connection.cursor(nested=True)
        cursor.execute(
                "select "
                "(select order_id, order_info, order_date from orders "
                    "where customer_id=customers.customer_id) from customers "
                " where customer_id in (5, 6)"
                "order by customer_id")
        row = cursor.fetchone()
        nested_cursor = row[0]

        self.assertEquals(getrows(nested_cursor),
         [
            (113, 'some order info', datetime.datetime(2012, 9, 5, 17, 24, 12)),
            (114, 'some order info', datetime.datetime(2012, 9, 5, 17, 24, 12)),
            (115, 'some order info', datetime.datetime(2012, 9, 5, 17, 24, 12)),
        ] + padding
        )
        row = cursor.fetchone()
        nested_cursor = row[0]

        self.assertEquals(getrows(nested_cursor),
         [
            (116, 'some order info',
                    datetime.datetime(2012, 9, 5, 17, 24, 12)),
            (117, 'some order info',
                    datetime.datetime(2012, 9, 5, 17, 24, 12)),
            (118, 'some order info',
                    datetime.datetime(2012, 9, 5, 17, 24, 12)),
          ] + padding
        )

    def test_fetchone_nested(self):
        self._test_nested_fetch(
                lambda cursor: [cursor.fetchone()
                                for i in range(5)], [None, None])

    def test_fetchmany_nested(self):
        self._test_nested_fetch(lambda cursor: cursor.fetchmany(5))

    def test_fetchall_nested(self):
        self._test_nested_fetch(lambda cursor: cursor.fetchall())

    def test_none_cursor_nested(self):
        cursor = self.connection.cursor(nested=True)
        cursor.execute("insert into items VALUES (%s, %s, %s, %s)",
            (1012, 101, 9.99, 1)
            )
        self.assertEquals(cursor.description, None)
        self.assertEquals(cursor.fdbsql_description, None)

    def test_none_cursor_plain(self):
        cursor = self.connection.cursor(nested=False)
        cursor.execute("insert into items VALUES (%s, %s, %s, %s)",
            (1012, 101, 9.99, 1)
            )
        self.assertEquals(cursor.description, None)

    def test_actual_scalar_select_not_nested(self):
        cursor = self.connection.cursor(nested=False)
        cursor.execute(
                    "select customer_id, "
                    "(select name from customers where customer_id=1) as c2 "
                    "from customers where customer_id in (3, 7)")
        self.assertEquals(
            [c[0:2] for c in cursor.description],
            [
                (u'customer_id',
                    psycopg2.extensions.INTEGER),
                (u'c2',
                    psycopg2.STRING),
            ]
        )

    def test_fdbsql_nested_description(self):
        cursor = self.connection.cursor(nested=True)
        cursor.execute(
                "select customer_id, "
                "(select order_id, (select item_id from items where "
                    "order_id=orders.order_id) as i1 from orders "
                    "where customer_id=customers.customer_id) as o1 "
                "from customers "
                "where customer_id in (1, 2, 3)"
                "order by customer_id")
        self.assertEquals(
            cursor.fdbsql_description,
            [
                (u'customer_id',
                    psycopg2.extensions.INTEGER,
                    None, None, None, None, None, None),
                (u'o1',
                    foundationdb_sql.NESTED_CURSOR,
                    None, None, None, None, None,
                    [
                        (u'order_id', psycopg2.extensions.INTEGER,
                        None, None, None, None, None, None),
                        (u'i1', foundationdb_sql.NESTED_CURSOR,
                            None, None, None, None, None,
                            [(u'item_id', psycopg2.extensions.INTEGER,
                                None, None, None, None, None, None)
                            ]
                        )
                    ]
                )
            ]
        )
    def test_multiple_nest(self):
        cursor = self.connection.cursor(nested=True)
        cursor.execute(
                "select customer_id, "
                "(select order_id, (select item_id from items where "
                    "order_id=orders.order_id) from orders "
                    "where customer_id=customers.customer_id) from customers "
                " where customer_id in (1, 2, 3)"
                "order by customer_id")

        def expand_row(row):
            return [
                expand_rows(col) for col in row
            ]
        def expand_rows(result):
            if hasattr(result, 'fetchall'):
                return [expand_row(elem) for elem in result.fetchall()]
            else:
                return result
        expanded = expand_rows(cursor)
        self.assertEquals(expanded,
            [
                [1,
                    [
                        [101, [[1001], [1002]]],
                        [102, [[1003]]],
                        [103, [[1004]]]
                    ]
                ],
                [2,
                    [
                        [104, [[1005]]],
                        [105, [[1006]]],
                        [106, [[1007]]]
                    ]
                ],
                [3,
                    [
                        [107, [[1008], [1009]]],
                        [108, [[1010]]],
                        [109, [[1011]]]
                    ]
                ]
            ]
        )

    def test_dupe_names_nested(self):
        cursor = self.connection.cursor(nested=True)

        assert_raises_message(
            psycopg2.OperationalError,
            "Duplicate name in JSON-formatted result set: x",
            cursor.execute, "select 1 as x, 2 as x"
        )

    def test_dupe_names_plain(self):
        cursor = self.connection.cursor(nested=False)

        cursor.execute("select 1 as x, 2 as x")
        self.assertEquals(cursor.fetchone(), (1, 2))

    def test_typecast_date_outer(self):
        cursor = self.connection.cursor(nested=True)
        cursor.execute(
                "select birthdate from customers where "
                "customer_id=5")
        self.assertEquals(cursor.fetchone()[0], datetime.date(1982, 7, 16))

    def test_typecast_date_inner(self):
        cursor = self.connection.cursor(nested=True)
        cursor.execute(
                'select (select order_date from orders where '
                    'customer_id=5) as orders from customers')
        self.assertEquals(cursor.fetchone()[0].fetchone()[0],
                    datetime.datetime(2012, 9, 5, 17, 24, 12))

    def test_typecast_boolean_outer(self):
        cursor = self.connection.cursor(nested=True)
        cursor.execute(
                "select some_bool from customers where "
                "customer_id=3")
        self.assertEquals(cursor.fetchone()[0], True)

        cursor.execute(
                "select some_bool from customers where "
                "customer_id=5")
        self.assertEquals(cursor.fetchone()[0], False)

    def test_typecast_unicode(self):
        cursor = self.connection.cursor(nested=True)

        # wow
        cursor.execute(u"select 'drôle m’a réveillé'")
        self.assertEquals(cursor.fetchone()[0], u'drôle m’a réveillé')

    def test_fetchone_plain(self):
        cursor = self.connection.cursor(nested=False)
        cursor.execute(
                "select customer_id, name from customers where "
                "customer_id in (5, 6, 7) order by customer_id")
        self.assertEquals([
                cursor.fetchone() for i in range(5)
            ],
            [(5, 'Peter Beaman'), (6, u'Thomas Jones-Low'), (7, u'Mike McMahon'),
                None, None]
        )

    def test_fetchall_plain(self):
        cursor = self.connection.cursor(nested=False)
        cursor.execute(
                "select customer_id, name from customers where "
                "customer_id in (5, 6, 7) order by customer_id")
        self.assertEquals(cursor.fetchall(),
            [(5, 'Peter Beaman'), (6, u'Thomas Jones-Low'), (7, u'Mike McMahon')]
        )

    def test_fetchmany_plain(self):
        cursor = self.connection.cursor(nested=False)
        cursor.execute(
                "select customer_id, name from customers where "
                "customer_id in (5, 6, 7) order by customer_id")
        self.assertEquals(cursor.fetchmany(2),
            [(5, 'Peter Beaman'), (6, u'Thomas Jones-Low')]
        )
        self.assertEquals(cursor.fetchmany(2),
            [(7, u'Mike McMahon')]
            )

        self.assertEquals(cursor.fetchmany(2), [])

    def test_non_nested_disables_json(self):
        cursor = self.connection.cursor(nested=False)
        self.assertRaises(
            psycopg2.ProgrammingError,
            cursor.execute,
            "select customer_id, (select order_id from orders) from customers"
        )

    def test_multiple_cursor_nested_non_nested(self):
        nested = self.connection.cursor(nested=True)
        non_nested = self.connection.cursor(nested=False)

        non_nested.execute("select 1")
        self.assertEquals(non_nested.fetchone(), (1, ))

        nested.execute("select 1")
        self.assertEquals(nested.fetchone(), (1, ))

        non_nested.execute("select 1")
        self.assertEquals(non_nested.fetchone(), (1, ))

    def test_non_nested_returns_rows(self):
        cursor = self.connection.cursor(nested=False)
        cursor.execute("select customer_id, name from customers "
                            "where customer_id in (5, 6, 7)")
        self.assertEquals(
            cursor.fetchall(),
            [(5, 'Peter Beaman'), (6, 'Thomas Jones-Low'), (7, 'Mike McMahon')]
        )

class DatatypeTest(Psycopg2Test, unittest.TestCase):
    def _roundtrip(self, datatype, value):
        connection = self.connection
        cursor = connection.cursor(nested=True)

        cursor.execute("""DROP TABLE IF EXISTS typed_values""")

        cursor.execute("""
        CREATE TABLE IF NOT EXISTS typed_values
          (
             id   INT NOT NULL PRIMARY KEY,
             data %s
          )
        """ % datatype)
        cursor.close()
        connection.commit()

        cursor = connection.cursor()
        cursor.execute(
                "INSERT INTO typed_values (id, data) VALUES (%(id)s, %(data)s)",
                {"id": 1, "data": value}
        )
        cursor.execute("SELECT data FROM typed_values WHERE id=1")
        self.assertEquals(cursor.fetchone()[0], value)

        cursor.executemany(
                "INSERT INTO typed_values (id, data) VALUES (%(id)s, %(data)s)",
                [
                    {"id": id_, "data": value} for id_ in range(2, 4)
                ]
        )
        cursor.execute("SELECT data FROM typed_values WHERE id IN (2, 3)")
        for row in cursor:
            self.assertEquals(cursor.fetchone()[0], value)


    def test_adapter_none(self):
        from foundationdb_sql import psycopg2
        psycopg2._psycopg2_adapter_cache.clear()

        connection = self.connection
        cursor = connection.cursor(nested=True)
        cursor.execute("SELECT CAST(NULL AS date)")
        row = cursor.fetchone()
        self.assertEquals(row[0], None)

        cursor.execute("SELECT CAST('2013-10-5' AS date)")
        row = cursor.fetchone()
        self.assertEquals(row[0], datetime.date(2013, 10, 5))

    def test_adapter_none_two(self):
        from foundationdb_sql import psycopg2
        psycopg2._psycopg2_adapter_cache.clear()

        connection = self.connection
        cursor = connection.cursor(nested=True)
        cursor.execute("SELECT CAST(NULL AS bigint)")
        row = cursor.fetchone()
        self.assertEquals(row[0], None)

        cursor.execute("SELECT CAST(5 AS bigint)")
        row = cursor.fetchone()
        self.assertEquals(row[0], 5)

    def test_date(self):
        self._roundtrip("DATETIME", datetime.datetime(2013, 5, 18))

    def test_date_historic(self):
        self._roundtrip("DATETIME", datetime.datetime(1850, 5, 18))

    def test_time(self):
        self._roundtrip("TIME", datetime.time(15, 31, 5))

    def test_datetime(self):
        self._roundtrip("DATETIME", datetime.datetime(2013, 5, 18, 15, 31, 5))

    def test_datetime_historic(self):
        self._roundtrip("DATETIME", datetime.datetime(1850, 5, 18, 15, 31, 5))
