import collections

class NestedCursorType(object):
    """Placeholder for the 'nested cursor type'.

    This object defines a "virtual" OID value of 5001.

    """
    def __eq__(self, other):
        return other == 5001 or isinstance(other, NestedCursorType)

    def __hash__(self):
        return hash(5001)

    def __repr__(self):
        return "<foundationdb_sql.api.NESTED_CURSOR>"

NESTED_CURSOR = NestedCursorType()
"""DBAPI type indicator for a data member that is of type "nested cursor".

When using the psycopg2 implementation, :data:`.NESTED_CURSOR` is present
within the :attr:`.PG2NestedCursor.description` and
:attr:`.PG2NestedCursor.fdbsql_description` for those columns that indicate
a nested cursor type.

.. seealso::

    :ref:`nested_result_sets` - usage example.

"""


class NestedCursor(object):
    """Represent a cursor that's specific to a nested result set.

    The :class:`.NestedCursor` is always linked to a parent result set.
    Using the psycopg2 implementation, this parent cursor is always either
    another :class:`.NestedCursor`, or if at the root level is an
    instance of :class:`.PG2NestedCursor`.

    :class:`.NestedCursor` follows the same general convention as that
    of the DBAPI cursor, including that it has its own description and
    fetch methods.

    .. seealso::

        :ref:`nested_result_sets` - usage example.

    """
    def __init__(self, ctx, arraysize, fields, description_factory):
        self.ctx = ctx
        self._fields = fields
        self._description_factory = description_factory
        self._rows = collections.deque()
        self.arraysize = arraysize

    @property
    def description(self):
        """Return the description of this cursor as per the DBAPI specifciation.

        .. seealso::

            :attr:`.PG2NestedCursor.description` - corresponding description on
            the outermost result set.

            :attr:`.PG2NestedCursor.fdbsql_description` - cursor description
            on the outermost result set with additional data specific to nested
            result sets.

            :ref:`nested_result_sets`

        """

        return self._description_factory(self._fields)

    def fetchone(self):
        """Fetch a single row as per the DBAPI specification."""
        if self._rows:
            return self._rows.popleft()
        else:
            return None

    def fetchall(self):
        """Fetch all rows as per the DBAPI specification."""
        r = list(self._rows)
        self._rows.clear()
        return r

    def fetchmany(self, size=None):
        """Fetch a batch of rows as per the DBAPI specification."""
        if size is None:
            size = self.arraysize
        l = list(self._rows)
        r, self._rows = l[0:size], collections.deque(l[size:])
        return r

    def close(self):
        """close this :class:`.NestedCursor`.

        As a :class:`.NestedCursor` is only a wrapper on top of a single
        JSON-formatted result set, the :meth:`.NestedCursor.close` method
        does nothing.  The :meth:`psycopg2:cursor.close` method established
        on the root :class:`.PG2NestedCursor` will close the actual underlying
        cursor.

        """
