"""Routines used to decode result sets from FDB's ``json_with_meta_data``
output format into nested DBAPI-compliant result cursors.

"""
import json
from .api import NestedCursor

json_decoder = json.JSONDecoder()

def _json_decode(col):
    if isinstance(col, dict) or isinstance(col, list):
        return col                      # Already JSON decoded by driver.
    else:
        return json_decoder.decode(col)

_NESTED_OID = 5001

class FDBSQLResultContext(object):
    """Represent a result set from the FDB sql layer.

    The :class:`.FDBSQLResultContext` class is a mediating class between
    a traditional DBAPI cursor and that of the JSON-formatted result set
    returned by the FDB SQL layer when the ``json_with_meta_data`` output
    format is used.

    .. seealso::

        :class:`.PG2ResultContext` - the psycopg2 implementation of
        this class.

    """
    def gen_description(self, fields):  # pragma: no cover
        raise NotImplementedError()

    def typecast(self, value, oid):  # pragma: no cover
        raise NotImplementedError()

    def operational_error(self, message):
        raise ValueError(message)

    def __init__(self, cursor, firstrow):
        self.cursor = cursor
        self.fields = _fields_from_row(firstrow, self)

    @property
    def arraysize(self):
        return self.cursor.arraysize

def _fields_from_row(row, ctx):
    document = _json_decode(row[0])
    return _format_fields(document, ctx)

def _filter_row(row, ctx):
    if row is None:
        return None
    document = _json_decode(row[0])
    return _format_row(document, ctx.fields, ctx)

def _create_rowset(document, fields, ctx):
    return [
        _format_row(row, fields, ctx)
        for row in document
    ]

def _format_row(document, fields, ctx):
    row = []
    for field in fields:
        if field['type_oid'] == _NESTED_OID:
            value = NestedCursor(
                        ctx,
                        ctx.arraysize,
                        field['foundationdb_sql.fields'],
                        ctx.gen_description
            )
            value._rows.extend(
                _create_rowset(
                    document[field['name']],
                    field['foundationdb_sql.fields'],
                    ctx
                )
            )

        else:
            value = ctx.typecast(
                            document[field['name']],
                            field['type_oid']
                        )
        row.append(value)
    return tuple(row)

def _format_fields(document, ctx):
    ret = []
    dupes = set()
    for attrnum, rec in enumerate(document):
        name = rec['name']
        if name in dupes:
            ctx.operational_error("Duplicate name in "
                    "JSON-formatted result set: %s" % name)
        dupes.add(name)
        newrec = {
            'table_oid': None,
            'name': name,
            'column_attrnum': attrnum,
            'format': None,
            'type_modifier': -1,
            'type_size': -1
        }
        if 'columns' in rec:
            newrec['type_oid'] = _NESTED_OID
            newrec['foundationdb_sql.fields'] = _format_fields(rec['columns'], ctx)
        else:
            newrec['type_oid'] = rec['oid']
        ret.append(newrec)
    return ret
