# FoundationDB Python API
# Copyright (c) 2012 FoundationDB, LLC

# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:

# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.

"""Documentation for this API can be found at
https://foundationdb.com/documentation/api-python.html"""

def open(*args, **kwargs):
    raise RuntimeError('You must call api_version(100) before using any fdb methods')

init = open

def api_version(ver):
    if '_version' in globals():
        if globals()['_version'] != ver:
            raise RuntimeError('FDB API already loaded at version %d' % _version)
        return

    if ver < 13:
        raise RuntimeError('FDB API versions before 13 are not supported')

    if ver > 100:
        raise RuntimeError('Latest known FDB API version is 100')

    import fdb.impl

    err = fdb.impl._capi.fdb_select_api_version_impl(ver, 100)
    if err != 0:
        raise RuntimeError('FoundationDB API error')

    list = (
        'FDBError',
        'Future',
        'Cluster',
        'Database',
        'Transaction',
        'KeyValue',
        'KeySelector',
        'init',
        'create_cluster',
        'open',
        'transactional',
        'options',
        'StreamingMode',
        )

    for sym in list:
        globals()[sym] = getattr(fdb.impl, sym)

    if ver > 22:
        import fdb.locality
        
    if ver == 13:
        globals()["open"] = getattr(fdb.impl, "open_v13")
        globals()["init"] = getattr(fdb.impl, "init_v13")

        # Future.get got renamed to Future.wait in v14 to make room for
        # Database.get, we have to undo that here
        for name in dir(fdb.impl):
            o = getattr(fdb.impl, name)
            try:
                if issubclass(o, fdb.impl.Future):
                    if hasattr(o, "wait"):
                        o.get = o.wait
            except TypeError: pass

        # FDBRange used to be called FDBRangeIter and was an iterator,
        # but it's now a container. In v13 we have to make it act like
        # an iterator again.
        def next(self):
            if not hasattr(self, "__iterating"):
                self.__iterating = iter(self)
            return next(self.__iterating)
        setattr(fdb.impl.FDBRange, "next", next)

    globals()['_version'] = ver
