# FoundationDB Python API
# Copyright (c) 2012 FoundationDB, LLC

# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:

# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.

"""Documentation for this API can be found at
https://foundationdb.com/documentation/api-python.html"""

from fdb import impl as _impl

def _get_boundary_keys(db_or_tr, begin, end):
    if isinstance(db_or_tr, _impl.Transaction):
        tr = db_or_tr.db.create_transaction()
                # This does not guarantee transactionality because of the exception handling below,
                # but it does hide latency for the new transaction's start
        tr.set_read_version(db_or_tr.get_read_version().wait())
    else:
        tr = db_or_tr.create_transaction()
    first_time = True
    while begin < end:
        try:
            lastbegin = begin
            tr.options.set_access_system_keys()
            kvs = tr.snapshot.get_range(b'\xff'+b'/keyServers/'+begin, b'\xff'+b'/keyServers/'+end)
            if first_time:
                first_time = False
                yield None # trick to get the above get_range to be asynchronously dispatched before get_boundary_keys() returns.
            for kv in kvs:
                yield kv.key[13:]
                begin = kv.key[13:]+b'\x00'
            begin = end
        except _impl.FDBError as e:
            if e.code == 1007 and begin != lastbegin: # if we get a past_version and *something* has happened, then we are no longer transactional
                tr = tr.db.create_transaction()
            else:
                tr.on_error(e).wait()

def get_boundary_keys(db_or_tr, begin, end):
    begin = _impl.keyToBytes(begin)
    end = _impl.keyToBytes(end)

    gen = _get_boundary_keys(db_or_tr, begin, end)
    try:
        next(gen)
    except StopIteration: # if _get_boundary_keys() never yields a value, e.g. begin > end
        return (x for x in list())
    return gen


@_impl.transactional
def get_addresses_for_key(tr, key):
    keyBytes = _impl.keyToBytes(key)
    return _impl.FutureStringArray(tr.capi.fdb_transaction_get_addresses_for_key(tr.tpointer, keyBytes, len(keyBytes)))
