# FoundationDB Python API
# Copyright (c) 2012 FoundationDB, LLC

# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:

# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.

import fdb.tuple

class Subspace (object):

    def __init__(self, prefixTuple=tuple(), rawPrefix=b''):
        self.rawPrefix = rawPrefix + fdb.tuple.pack(prefixTuple)

    def __repr__(self):
        return 'Subspace(rawPrefix=' + repr(self.rawPrefix) + ')'

    def __getitem__(self, name):
        return Subspace((name,), self.rawPrefix)

    def key(self):
        return self.rawPrefix

    def pack(self, t=tuple()):
        return self.rawPrefix + fdb.tuple.pack(t)

    def unpack(self, key):
        if not self.contains(key):
            raise ValueError('Cannot unpack key that is not in subspace.')

        return fdb.tuple.unpack(key[len(self.rawPrefix):])

    def range(self, t=tuple()):
        p = fdb.tuple.range(t)
        return slice(self.rawPrefix + p.start, self.rawPrefix + p.stop)

    def contains(self, key):
        return key.startswith(self.rawPrefix)

    def as_foundationdb_key(self):
        return self.rawPrefix

    def subspace(self, tuple):
        return Subspace(tuple, self.rawPrefix)
