from request import Request
from response import Response
from _routes import routes
from dotdict import DotDict
from errors import HTTPError, Error404, Error500
import traceback
from jinja2 import Environment, ChoiceLoader, PackageLoader, FileSystemLoader
import os
import sys
import types
from threading import current_thread, RLock
from pkg_resources import iter_entry_points

# For jinja2 toolset
from toolset import toolset

# Import StaticDispatcher to retrieve static files
from staticdispatcher import StaticDispatcher

# Import global config
from _config import config

# Import logger
from _logger import logger

from util import truncate, Singleton

import driverdatabase
import contextlib


class App(Singleton):
  '''
  The backbone behind Frame's dispatching. :class:`App` is a WSGI compliant application
  that handles all of the request/response sent to and generated by the application.
  Currently, it is also in charge of dispatching URI's to controllers/actions, but that
  may change in the future.
  
  There shouldn't be a lot of reason to interact directly with objects of this class,
  as most configuration options can be (and should be) set via :obj:`frame._config.config`.
  However, if you need to make some modifications to the Jinja2 environment or other
  hacking, this is the place to do it.
  '''
  
  def __init__(self, debug=True):
    '''
    Initialize the application with some sane defaults.
    '''
    
    self.static_map = StaticDispatcher(self)
    self.path = os.path.dirname(os.path.abspath(sys.argv[0]))
    self.routes = routes
    self.debug = debug

    # Jinja2 environment placeholder
    self.environment = None
    
    # Store data for each thread indiviually
    self.thread_data = {}

    self.toolset = toolset
    self.toolset.app = self
    
    self.drivers = self._setup_driver_database()
      
    # Setup pre and post processor lists
    self.preprocessors = []
    self.postprocessors = []
    
    self.config = config

    # A global lock for the application
    self.lock = RLock()
    
    # A variable to store whether or not _prep_start has been run
    self._prepped = False

    # Load config entry points
    self.load_config()
    
  def load_drivers(self):
    for entry_point in iter_entry_points('frame.drivers'):
      logger.log_info("Loading '%s' driver..." % entry_point.name)
      register_driver = entry_point.load()
      register_driver(self.drivers)

  def load_config(self):
    for entry_point in iter_entry_points('frame.config'):
      register_config = entry_point.load()
      register_config(config)

  def _setup_driver_database(self):
    drivers = driverdatabase.DriverDatabase(self)
    
    # Add postprocessor interface
    drivers.add_interface(
      'postprocessor',
      driverdatabase.PostprocessorInterface)
      
    # Add preprocessor interface
    drivers.add_interface(
      'preprocessor',
      driverdatabase.PreprocessorInterface)
    
    # Add dispatcher interface
    drivers.add_interface(
      'dispatcher',
      driverdatabase.DispatcherInterface,
      config=config.application)
      
    # Add hooks interface
    drivers.add_interface(
      'hook',
      driverdatabase.DriverInterface,
      config=config.hooks)

    # Add init_hooks interface
    drivers.add_interface(
      'init_hook',
      driverdatabase.DriverInterface,
      config=config.init_hooks)
    
    return drivers
    
  def _setup_thread(self):
    thread = current_thread()
    if thread not in self.thread_data:
      self.thread_data[thread] = {}
    
  @property
  def request(self):
    return self.thread_data[current_thread()]['request']
    
  @request.setter
  def request(self, value):
    self._setup_thread()
    self.thread_data[current_thread()]['request'] = value
    
  @request.deleter
  def request(self):
    del(self.thread_data[current_thread()]['request'])
    
  @property
  def response(self):
    return self.thread_data[current_thread()]['response']
    
  @response.setter
  def response(self, value):
    self._setup_thread()
    self.thread_data[current_thread()]['response'] = value
    
  @response.deleter
  def response(self):
    del(self.thread_data[current_thread()]['response'])
    
  @property
  def current_controller(self):
    '''
    Return the active :mod:`frame.controller.Controller`.
    '''
    return self.routes.current_controller

  def _dispatch(self, environ):
    '''
    Dispatches the HTTP request. Fetches the appropriate controller and action (if any)
    and instantiates :mod:`frame.request.Request`, :mod:`frame.response.Response`, and
    :mod:`frame.session.Session`. If any error is received while loading the controller
    action, an :exc:`frame.errors.Error500` exception is thrown and rendered neatly
    to the screen. If debugging is enabled, a traceback is sent as an HTTP response.
    
    :param environ: WSGI environment
    :return: A tuple of data to pass on to the WSGI server. The tuple consists of
      ``(status_line, headers, response_body)``
    '''
    
    self.request = Request(environ)
    
    if config.application.strip_trailing_slash and environ['PATH_INFO'] != '/':
      environ['PATH_INFO'] = environ.get('PATH_INFO', '').rstrip('/')

    try:
      match, params = self.dispatcher.handle(environ=environ)

    # If TypeError or AttributeError occurs then no match was found; we should check
    # if the URI matches any static mappings, which will return a 404 response if
    # no match is found.
    except (TypeError, AttributeError):
      return self.static_map.match(environ)
      
    # Otherwise, we should be good to handle the request
    else:
      for key, value in params.items():
        if key in ('controller', 'action', 'method'):
          del(params[key])
          
      hooks = map(
        lambda x: self.drivers.hook.load_driver(x, self, match.im_self),
        config.hooks)

      response = Response(self, match, params)
      
      with contextlib.nested(*hooks):
        for i in self.preprocessors:
          i(self.request, response)
        response.render()
      
      return response

  def __call__(self, environ, start_response):
    '''
    A callable to allow WSGI compliance.
    
    :param environ: WSGI environment
    :param start_response: The start_response function passed by the WSGI server
    :yield: Response body
    '''
    
    if not self._prepped:
      self.lock.acquire()
      self._prep_start()
      self.lock.release()
    
    try:
      response = self._dispatch(environ)
    except Exception, e:
      if hasattr(e, 'response'):
        response = e.response
      else:
        response = Error500().response

    # Need to do something more elegant to handle generators/chunked encoding...
    # Also need to come up with a better way to log chunked encodings
    if type(response.body) is types.GeneratorType:
      start_response(response.status, response.headers.items())
      response_length = 0
      for i in response.body:
        yield str(i)
        response_length += len(i)
      logger.log_request(self.request, response, response_length)

    else:
      # Apply post processors
      temp_data = {'response': response}

      def apply_postprocessors():
        for i in self.postprocessors:
          new_response = i(self.request, temp_data['response'])
          if new_response and isinstance(new_response, Response):
            temp_data['response'] = new_response
            apply_postprocessors()

      apply_postprocessors()

      response = temp_data['response']
              
      # Deliver the goods
      start_response(response.status, response.headers.items())
      yield str(response.body)
      try:
        logger.log_request(self.request, response, len(response.body) if response.body else 0)
      except Exception, e:
        pass
      
    self._remove_thread_data()
    
  def _remove_thread_data(self):
    thread = current_thread()
    if thread in self.thread_data:
      del(self.thread_data[thread])
    if thread in self.routes.thread_data:
      del(self.routes.thread_data[thread])
      
  def _prep_start(self):
    '''
    Populate data gathered from global config.
    '''
    
    logger.log_info("Preparing WSGI Application...")
    
    # Load modules
    self.load_drivers()
    
    for i in config.preprocessors:
      self.preprocessors.append(self.drivers.preprocessor[i])
    
    for i in config.postprocessors:
      self.postprocessors.append(self.drivers.postprocessor[i])

    for mapping, path in config.static_map.items():
      logger.log_info("Mapping static directory: '%s' => '%s'" % (
        mapping, truncate(path, 40)))
      self.static_map[mapping] = path
      
    # Initialize dispatcher
    self.dispatcher = self.drivers.dispatcher.current(self)

    for i, init_hook in self.drivers.init_hook.iteritems():
      init_hook(self)
    
    # Signal that the application has been prepped
    self._prepped = True
      
  def start_fcgi(self, *args, **kwargs):
    '''
    Start the Flup FastCGI/WSGIServer interface. Any options passed to this method are
    automatically passed to the Flup WSGIServer.
    '''
    
    from flup.server.fcgi import WSGIServer
    
    self.server_type = 'fcgi'
    self._prep_start()
    logger.log_info("Starting FLUP WSGI Server...")
    WSGIServer(self, *args, **kwargs).run()

  def start_http(self, host='127.0.0.1', port=8080, *args, **kwargs):
    '''
    Start the Frame Development HTTPServer. Defaults to listen on localhost (port 8080).
    Note: Does not work on Windows unless ``auto_reload=False`` is passed, as currently
    the :mod:`frame.server.modulemonitor.ModuleMonitor` only works on *nix systems.
    
    Like :meth:`start_fcgi`, options are all passed to the HTTP Server. For additional
    parameters, please reference :mod:`frame.server.http.HTTPServer`.
    
    :param host: Listen host/address
    :param port: Listen port
    '''
    
    from frame.server.http import HTTPServer
    
    self.server_type = 'http'
    self._prep_start()
    logger.log_info("Starting Frame HTTP Server on %s:%s..." % (host, port))
    HTTPServer(self, host=host, port=port, *args, **kwargs).run()


app = App()
