import os
from os.path import expanduser, realpath

standard_file_types = {
    'python' : ('.py',),
    'html'   : ('.html', '.htm'),
    'css'    : ('.css',),
    'js'     : ('.js',),
    'rest'   : ('.rst', '.rest'),
    'text'   : ('.txt',)
    }


# -----------------------------------------------------------------------------

class ConfigurationRule(object):
    
    def __call__(self, argv, config):
        raise NotImplementedError()

# -----------------------------------------------------------------------------

class AcceptFiles(ConfigurationRule):
    '''Validate and combine the commandline arguments with the 
    user provile options and return a unified option object.
    '''
    def __call__(self, argv, config):
        # specific file type operations override the .furrc
        type_list = set() 
        for key in standard_file_types:
            if hasattr(argv, key) and getattr(argv, key) == True:
                type_list.update(standard_file_types[key])                    

        if len(type_list) == 0:
            for sft in standard_file_types.values():
                type_list.update(sft)

        argv.accept_file_types = list(type_list)


# -----------------------------------------------------------------------------

class ExcludeDirectories(ConfigurationRule):
    
    def __call__(self, argv, config):
        dir_list = set()

        if 'exclude_dirs' in config:
            dir_list.update(config['exclude_dirs']) 

        argv.exclude_dirs = list(dir_list)


# -----------------------------------------------------------------------------

class SetRoot(ConfigurationRule):
    
    def __call__(self, argv, config):
        root = '.'
        root_options = {}
        if 'root' in config:
            root_options = config['root']
        if 'default' in root_options:
            root = root_options['default']
        if argv.root in root_options:
            root = root_options[argv.root]
        root = expanduser(realpath(root))
        if not os.path.exists(root):
            raise ValueError('Invalid directory: {0}'.format(root))
        argv.root = root

# -----------------------------------------------------------------------------


def validate(argv, config):
    
    rules = [ AcceptFiles(),
              ExcludeDirectories(),
              SetRoot(),
            ]
    
    if argv.noenv:
        config={}

    for rule in rules:
        rule(argv, config)

