import os
import argparse
from configobj import ConfigObj
from validate import validate, standard_file_types

__version__ = '0.1a3'

descrip  = '''fur {0}
-----------------

Find and replace.

Find matches of regular EXPRESSION in each source file in the tree from 
ROOT on down.

If [FILES] is specified, then only those files/directories are checked.

Default switches may be specified in an .furrc file. If you want no 
dependency on the environment, turn it off with --noenv.

Examples: 
---------
    Basic search only returns matches with no substitution options:
    fur "Find this"

    Basic substitution allows to replace matched values:
    fur -s "substitute this" "find this"

    Set root directory to 'python' and only search on 'python' files:
    fur -r python --python -s ".prop = \1" "\.set_prop\((value)\)"


'''.format(__version__)

args = argparse.ArgumentParser(
        prog='fur',
        usage='\nfur [OPTIONS]... [--sub] EXPRESSION',
        formatter_class=argparse.RawDescriptionHelpFormatter,
        description=descrip, 
        version='fur {0}'.format(__version__),
        epilog='That all she wrote folks.')

regex_args = args.add_argument_group('Regular Expression Arguments')

regex_args .add_argument(
    '--root',
    '-r',
    type=str,
    default='.',
    help='root search directory')

regex_args.add_argument(
    '--sub',
    '-s',
    type=str,
    default=None,
    help='substitution value')

regex_args.add_argument(
    '--ignore-case',
    '-i',
    action='store_true',
    help='ignore case distinctions in EXPRESSION')

regex_args.add_argument('expression',
    help='the regular expression for the search.')

file_type_args = args.add_argument_group('File Type Selections')
for file_type in standard_file_types.keys():
    file_type_args.add_argument(
        '--{0}'.format(file_type),
        action='store_true',
        help='flag {0} files for inclusion in search.'.format(file_type))

misc_args = args.add_argument_group('Miscellaneous Arguments')

misc_args.add_argument(
    '--noenv',
    action='store_true',
    help='use to ignore defaults set by .furrc file')

# -----------------------------------------------------------------------------

def get_config_dir():
    cfg_pth = os.path.join('~','.fur')
    cfg_pth = os.path.expanduser(cfg_pth)
    if not os.path.exists(cfg_pth):
        os.mkdir(cfg_pth)
    if not os.path.isdir(cfg_pth):
        raise ValueError('{0} is not a directory.'.format(cfg_pth))
    return cfg_pth


# -----------------------------------------------------------------------------

def get_config_file(cfg_dir):
    cfg_file = os.path.join(cfg_dir, 'fur.cfg')
    config = ConfigObj(cfg_file, list_values=True)

    if not os.path.exists(cfg_file):
        # create default config file options
        config['accept_file_types'] = ['.py','.txt','.html']
        config['exclude_dirs'] = ['.git','build','dist','samples']
        config['root'] = {} 
        config['root']['default'] = '.'
        config.write()

    return config


# -----------------------------------------------------------------------------

def main():
    # commandline arguments
    argv = args.parse_args()

    # user profile options from ~/.fur/config.cfg
    cfg_dir = get_config_dir()
    config = get_config_file(cfg_dir)

    validate(argv, config)

    import fur
    fur.find_und_replace(argv)


# -----------------------------------------------------------------------------

if __name__ == '__main__':
    
    main()
