import os
from os.path import join, splitext
import re


# -----------------------------------------------------------------------------

class FileInfo(object):
    '''Class that wraps standard file/path information and functions into 
    one handy object. An instance of FileInfo represents one and only one
    file.

    The FileInfo object is also responsible for collecting and reporting on
    any regular expression matches that are found in its contents.
    '''
    
    def __init__(self, dir_path, file_name):
        self.dir_path = dir_path
        self.file_name = file_name
        self.matches = {}

    @property
    def path(self):
        return join(self.dir_path, self.file_name)
    
    @property
    def ext(self):
        return splitext(self.file_name)[1]

    def add_match(self, regex_match):
        self.matches[regex_match.line_no] = regex_match

    def has_match(self):
        return len(self.matches)

    def count(self):
        return self.has_match()

    def replacements(self):
        return sum([ 1 for p in self.matches.values() if p.do_replace ])    

    def __str__(self):
        return self.path
        

# -----------------------------------------------------------------------------

class RegexMatchInfo(object):
    '''Data structure that represents a line of text from a file that has
    successfully matched the regular expression.

    :line_no prop: the line number in the file.
    :line_str prop: the actual line of text.
    :matchinfo prop: the regular expression match object.
    :do_replace prop: bool value used to indicate whether a substitution 
    should be applied to any matches in this line. 
    '''

    def __init__(self, line_no, line_str, match_info):
        self.line_no = line_no
        self.line_str = line_str
        self.match_info = match_info
        self.do_replace = False


# -----------------------------------------------------------------------------

class FileList(list):
    '''A list of all files where one or more
    lines of text have matched the regular expression.

    :root prop: the root directory where we start the directory walk

    :harvest method: populates the list, applying rules for inclusion/
    exclusion of files.

    :add_dir_rule method: add a rule to the list. A directory rule is a callable
    that takes a path string and determines if it is to be excluded.

    :add_file_rule method: add a rule to the list. A file rule is a callable
    that takes a FileInfo object and determines whether it is of an accepable type.

    '''
    
    def __init__(self, root_path):
        list.__init__(self)
        self.root_path = root_path 
        self.dir_rules = []
        self.file_rules = []

    def add_dir_rule(self, rule):
        self.dir_rules.append(rule)
    
    def add_file_rule(self, rule):
        self.file_rules.append(rule)

    def harvest(self):
        tree = os.walk(self.root_path)
        for current, dirs, files in tree:
            self._harvest_dir(current, files)
       
    def _harvest_dir(self, current, files):

        for rule in self.dir_rules:
            skip_dir = rule(current)

        if skip_dir:
            return
        else:
            self._harvest_files(current, files)       

    def _generate_file_info(self, current, files):
        for f in files:
            yield FileInfo(current, f)

    def _harvest_files(self, current, files):
        for fi in self._generate_file_info(current, files):
            for rule in self.file_rules:
                match_found = rule(fi)  
                if not match_found:
                    break
            if match_found:
                self.append(fi)

       
# -----------------------------------------------------------------------------

class MatchList(list):
    '''A list of all FileInfo objects that contain regular expression matches
    to one or more of their lines.
    
    :regex prop: Regex object.

    :harvest method: Takes a FileList object and adds those FileInfo objects that
    have lines that match the regex.
    '''

    def __init__(self, regex):
        list.__init__(self)
        self._regex = regex 

    def harvest(self, matching_files):
        for fi in matching_files:
            self._regex.inspect(fi)    
            if fi.has_match():
                self.append(fi)


# -----------------------------------------------------------------------------

class Regex(object):
    '''Wraps standard regular expression functionality.
    '''

    def __init__(self, expression, options):
        self._exp = expression
        re_opts = 0
        if options.ignore_case:
           re_opts |= re.IGNORECASE

        self._re = re.compile(expression, re_opts)
        self._sub = options.sub
   
    def inspect(self, file_info):
        lines = open(file_info.path, 'r').readlines()
        for line_no in range(len(lines)):
            line_str = lines[line_no]
            match_info = self._re.findall(line_str)
            if match_info:
                regex_match = RegexMatchInfo(line_no, line_str, match_info) 
                file_info.add_match(regex_match)
 
    def replace(self, line):
        return self._re.sub(self._sub, line)

    def __str__(self):
        return self._exp

