from threading import Timer
from time import time
from types import StringType
from os import path, devnull
from glob import glob
from subprocess import Popen, PIPE
from sys import stdin

class Importer(object):

    def __init__(self, gauged, namespace_reload=None, debug=None, logger=None):
        self.gauged = gauged
        self.namespace_reload = namespace_reload
        self.debug = debug
        self.logger = logger
        self.load_namespaces_timer()
        self.realtime_check = False
        self.realtime_from = long(time() * 1000)
        self.realtime_handlers = None
        self.namespaces = self.load_namespaces()

    def load_namespaces(self):
        return None

    def load_namespaces_timer(self):
        self.namespaces = self.load_namespaces()
        if self.namespace_reload:
            timer = Timer(self.namespace_reload, self.load_namespaces_timer)
            timer.setDaemon(True)
            timer.start()

    def parse_line(self, line):
        return line.split(' ', 4)

    def add_realtime_handler(self, handler):
        if self.realtime_handlers is None:
            self.realtime_handlers = []
        self.realtime_handlers.append(handler)

    def queue(self, writer, line):
        timestamp, ip, uuid, namespace_key, data = self.parse_line(line)
        timestamp = long(timestamp)
        if self.namespaces is None:
            namespace = None
        else:
            namespace = self.namespaces.get(namespace_key, None)
            if namespace is None:
                return
        writer.add(data, timestamp=timestamp,
            namespace=namespace, debug=self.debug)
        if self.realtime_handlers is not None:
            if timestamp > self.realtime_from:
                for handler in self.realtime_handlers:
                    handler(timestamp, ip, uuid, namespace_key, data)

    def start(self, stream=None, resume_from=None):
        with self.gauged.writer as writer:
            if resume_from is None:
                resume_from = writer.resume_from()
            if self.logger is not None:
                self.logger.info('Resuming from %s', resume_from)
            if stream is None:
                stream = iter(stdin.readline, '')
            elif type(stream) == StringType:
                stream = LogStream(stream, resume_from, self.logger)
            queue = self.queue
            if resume_from:
                starts_before_resume_point = False
                for line in stream:
                    timestamp, _ = line.split(' ', 1)
                    timestamp = long(timestamp)
                    if timestamp < resume_from:
                        starts_before_resume_point = True
                    elif timestamp > resume_from and not starts_before_resume_point:
                        raise RuntimeError('Log file starts after resume point')
                    else:
                        queue(writer, line)
                        break
            for line in stream:
                queue(writer, line)

class LogStream(object):

    def __init__(self, log, resume_point=0, logger=None):
        self.log = log
        self.resume_point = resume_point
        self.logger = logger

    def cat(self, log):
        ext = path.splitext(log)[1]
        if ext == '.xz':
            command = 'xzcat'
        elif ext == '.bz2':
            command = 'bzcat'
        elif ext == '.gz':
            command = 'gzip -dc'
        else:
            command = 'cat'
        return '%s "%s"' % (command, log)

    def __iter__(self):
        lines = Popen(self.command, shell=True, stdout=PIPE, stderr=PIPE)
        try:
            while True:
                yield lines.stdout.readline()
        except StopIteration:
            pass
        lines.terminate()
        lines.stdout.close()
        lines.stderr.close()

    def start(self, log):
        command = self.cat(log) + ' | head -1'
        with open(devnull, 'w') as null:
            head = Popen(command, shell=True, stdout=PIPE, stderr=null)
            line = head.stdout.readline()
            head.terminate()
            head.stdout.close()
        timestamp = line.split(' ', 1)[0]
        if timestamp:
            timestamp = long(timestamp.replace('.', ''))
        else:
            timestamp = None
        if self.logger is not None:
            self.logger.debug('Log file %s has start point %s', log, timestamp)
        return timestamp

    @property
    def command(self):
        command = ['tail -F -c +0 "%s"' % self.log]
        start_point = self.start(self.log)
        archived = sorted(glob(self.log + '-*'), reverse=True)
        if start_point is None:
            if not len(archived):
                return command[0]
        elif start_point <= self.resume_point:
            return command[0]
        for log in archived:
            command.append(self.cat(log))
            start_point = self.start(log)
            if start_point is not None and start_point <= self.resume_point:
                break
        if self.resume_point and start_point > self.resume_point:
            msg = 'Trying to resume from %s but earliest log ' \
                'starts from %s' % (self.resume_point, start_point)
            raise RuntimeError(msg)
        return ' && '.join(reversed(command))
