import sys

import requests

__API_ENDPOINT__ = 'http://api.genderize.io'
__FEMALE__ = 'female'
__MALE__ = 'male'

def _collect_payload_single(name, country_id, language_id):
	"""Return the query params from the given arguments

		Keyword arguments:

		- `name` -- The name to lookup
		- `country_id` -- Restrict name lookup for a specific country
		- `language_id` -- Restrict name lookup for a specific language
	"""
	params = {'name': name}

	if (country_id != None):
		params['country_id'] = country_id

	if (language_id != None):
		params['language_id'] = language_id

	return params

def _collect_payload_multi(names, country_id, language_id):
	"""Return the query params from the given arguments

		Keyword arguments:

		- `names` -- The list of names to lookup
		- `country_id` -- Restrict name lookup for a specific country
		- `language_id` -- Restrict name lookup for a specific language
	"""
	params = {}

	for i in range(len(names)):
		key = 'name[{}]'.format(i)
		params[key] = names[i]

	if (country_id != None):
		params['country_id'] = country_id

	if (language_id != None):
		params['language_id'] = language_id

	return params

def genderize(name, country_id=None, language_id=None):
	"""Query the gender of the given name with the given options

		Keyword arguments:

		- `name` -- The name to lookup (**mandatory**, *no default*)
		- `country_id` -- Restrict name lookup for a specific country (*default:* `None`)
		- `language_id` -- Restrict name lookup for a specific language (*default:* `None`)
	"""
	r = requests.get(__API_ENDPOINT__, params=_collect_payload_single(name=name, country_id=country_id, language_id=language_id))

	# Raise if the request failed
	r.raise_for_status()

	return r.json()

def genderize_all(names, country_id=None, language_id=None):
	"""Query the gender of the given names with the given options

		Keyword arguments:

		- `names` -- The list of names to lookup (**mandatory**, *no default*)
		- `country_id` -- Restrict name lookup for a specific country (*default:* `None`)
		- `language_id` -- Restrict name lookup for a specific language (*default:* `None`)
	"""
	r = requests.get(__API_ENDPOINT__, params=_collect_payload_multi(names=names, country_id=country_id, language_id=language_id))

	# Raise if the request failed
	r.raise_for_status()

	return r.json()

def is_female(name, country_id=None, language_id=None):
	"""Check if the given name is female

		Keyword arguments:

		- `name` -- The name to lookup (**mandatory**, *no default*)
		- `country_id` -- Restrict name lookup for a specific country (*default:* `None`)
		- `language_id` -- Restrict name lookup for a specific language (*default:* `None`)
	"""
	r = requests.get(__API_ENDPOINT__, params=_collect_payload_single(name=name, country_id=country_id, language_id=language_id))

	# Raise if the request failed
	r.raise_for_status()

	return r.json().get('gender', 'white rabbit') == __FEMALE__

def is_male(name, country_id=None, language_id=None):
	"""Check if the given name is male

		Keyword arguments:

		- `name` -- The name to lookup (**mandatory**, *no default*)
		- `country_id` -- Restrict name lookup for a specific country (*default:* `None`)
		- `language_id` -- Restrict name lookup for a specific language (*default:* `None`)
	"""
	r = requests.get(__API_ENDPOINT__, params=_collect_payload_single(name=name, country_id=country_id, language_id=language_id))

	# Raise if the request failed
	r.raise_for_status()

	return r.json().get('gender', 'go ask alice') == __MALE__

def possessive_determiner(name, country_id=None, language_id=None):
	"""Return the possessive determiner ('his' or 'her'; see http://en.wikipedia.org/wiki/Possessive_determiner for more) for the given name

		Keyword arguments:

		- `name` -- The name to lookup (**mandatory**, *no default*)
		- `country_id` -- Restrict name lookup for a specific country (*default:* `None`)
		- `language_id` -- Restrict name lookup for a specific language (*default:* `None`)
	"""
	return 'his' if is_male(name=name, country_id=country_id, languate_id=language_id) else 'her'

def female_probability(name, country_id=None, language_id=None, return_sample_size=False):
	"""Return the probability of the given name being female

		Keyword arguments:

		- `name` -- The name to lookup (**mandatory**, *no default*)
		- `country_id` -- Restrict name lookup for a specific country (*default:* `None`)
		- `language_id` -- Restrict name lookup for a specific language (*default:* `None`)
		- `return_sample_size` -- If true, the return value is a tuple consisting of `(female_probability, sample_size)` (*default:* `False`)
	"""
	r = requests.get(__API_ENDPOINT__, params=_collect_payload_single(name=name, country_id=country_id, language_id=language_id))

	# Raise if the request failed
	r.raise_for_status()

	p = float(r.json().get('probability', -1.0))
	p = p if (r.json().get('gender') == __FEMALE__) else 1 - p

	return (p, r.json().get('count', -1)) if return_sample_size else p

def male_probability(name, country_id=None, language_id=None, return_sample_size=False):
	"""Return the probability of the given name being male

		Keyword arguments:

		- `name` -- The name to lookup (**mandatory**, *no default*)
		- `country_id` -- Restrict name lookup for a specific country (*default:* `None`)
		- `language_id` -- Restrict name lookup for a specific language (*default:* `None`)
		- `return_sample_size` -- If true, the return value is a tuple consisting of `(male_probability, sample_size)` (*default:* `False`)
	"""
	r = requests.get(__API_ENDPOINT__, params=_collect_payload_single(name=name, country_id=country_id, language_id=language_id))

	# Raise if the request failed
	r.raise_for_status()

	p = float(r.json().get('probability', -1.0))
	p =  p if (r.json().get('gender') == __MALE__) else 1 - p

	return (p, r.json().get('count', -1)) if return_sample_size else p

if (__name__ == '__main__'):

	if (len(sys.argv) > 2):
		names = sys.argv[1:]
		print genderize_all(names=names)
	else:
		n = sys.argv[1] if len(sys.argv) > 1 else 'julia'
		print genderize(name=n)
